package org.wildfly.swarm.config;

import org.wildfly.swarm.config.runtime.Address;
import java.util.HashMap;
import org.wildfly.swarm.config.runtime.ResourceType;
import org.wildfly.swarm.config.runtime.Implicit;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
/**
 * The configuration of the weld subsystem.
 */
@Address("/subsystem=weld")
@ResourceType("subsystem")
@Implicit
public class Weld<T extends Weld<T>> extends HashMap {

	private String key;
	private PropertyChangeSupport pcs;
	private Boolean developmentMode;
	private Boolean nonPortableMode;
	private Boolean requireBeanDescriptor;

	public Weld() {
		super();
		this.key = "weld";
		this.pcs = new PropertyChangeSupport(this);
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	/**
	 * Weld comes with a special mode for application development. When the
	 * development mode is enabled, certain built-in tools, which facilitate the
	 * development of CDI applications, are available. Setting this attribute to
	 * true activates the development mode.
	 */
	@ModelNodeBinding(detypedName = "development-mode")
	public Boolean developmentMode() {
		return this.developmentMode;
	}

	/**
	 * Weld comes with a special mode for application development. When the
	 * development mode is enabled, certain built-in tools, which facilitate the
	 * development of CDI applications, are available. Setting this attribute to
	 * true activates the development mode.
	 */
	@SuppressWarnings("unchecked")
	public T developmentMode(Boolean value) {
		Object oldValue = this.developmentMode;
		this.developmentMode = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("developmentMode", oldValue, value);
		return (T) this;
	}

	/**
	 * If true then the non-portable mode is enabled. The non-portable mode is
	 * suggested by the specification to overcome problems with legacy
	 * applications that do not use CDI SPI properly and may be rejected by more
	 * strict validation in CDI 1.1.
	 */
	@ModelNodeBinding(detypedName = "non-portable-mode")
	public Boolean nonPortableMode() {
		return this.nonPortableMode;
	}

	/**
	 * If true then the non-portable mode is enabled. The non-portable mode is
	 * suggested by the specification to overcome problems with legacy
	 * applications that do not use CDI SPI properly and may be rejected by more
	 * strict validation in CDI 1.1.
	 */
	@SuppressWarnings("unchecked")
	public T nonPortableMode(Boolean value) {
		Object oldValue = this.nonPortableMode;
		this.nonPortableMode = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("nonPortableMode", oldValue, value);
		return (T) this;
	}

	/**
	 * If true then implicit bean archives without bean descriptor file
	 * (beans.xml) are ignored by Weld
	 */
	@ModelNodeBinding(detypedName = "require-bean-descriptor")
	public Boolean requireBeanDescriptor() {
		return this.requireBeanDescriptor;
	}

	/**
	 * If true then implicit bean archives without bean descriptor file
	 * (beans.xml) are ignored by Weld
	 */
	@SuppressWarnings("unchecked")
	public T requireBeanDescriptor(Boolean value) {
		Object oldValue = this.requireBeanDescriptor;
		this.requireBeanDescriptor = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("requireBeanDescriptor", oldValue,
					value);
		return (T) this;
	}
}