package org.wildfly.swarm.config;

import org.wildfly.swarm.config.runtime.Address;
import java.util.HashMap;
import org.wildfly.swarm.config.runtime.ResourceType;
import org.wildfly.swarm.config.runtime.Implicit;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;

/**
 * The request controller subsystem. Used for request limiting and graceful
 * shutdown
 */
@Address("/subsystem=request-controller")
@ResourceType("subsystem")
@Implicit
public class RequestController<T extends RequestController<T>> extends HashMap {

	private String key;
	private PropertyChangeSupport pcs;
	private Integer activeRequests;
	private Integer maxRequests;
	private Boolean trackIndividualEndpoints;

	public RequestController() {
		super();
		this.key = "request-controller";
		this.pcs = new PropertyChangeSupport(this);
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(
			java.beans.PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	/**
	 * The number of requests that are currently running in the server
	 */
	@ModelNodeBinding(detypedName = "active-requests")
	public Integer activeRequests() {
		return this.activeRequests;
	}

	/**
	 * The number of requests that are currently running in the server
	 */
	@SuppressWarnings("unchecked")
	public T activeRequests(java.lang.Integer value) {
		Object oldValue = this.activeRequests;
		this.activeRequests = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("activeRequests", oldValue, value);
		return (T) this;
	}

	/**
	 * The maximum number of all types of requests that can be running in a
	 * server at a time
	 */
	@ModelNodeBinding(detypedName = "max-requests")
	public Integer maxRequests() {
		return this.maxRequests;
	}

	/**
	 * The maximum number of all types of requests that can be running in a
	 * server at a time
	 */
	@SuppressWarnings("unchecked")
	public T maxRequests(java.lang.Integer value) {
		Object oldValue = this.maxRequests;
		this.maxRequests = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("maxRequests", oldValue, value);
		return (T) this;
	}

	/**
	 * If this is true requests are tracked at an endpoint level, which will
	 * allow individual deployments to be suspended
	 */
	@ModelNodeBinding(detypedName = "track-individual-endpoints")
	public Boolean trackIndividualEndpoints() {
		return this.trackIndividualEndpoints;
	}

	/**
	 * If this is true requests are tracked at an endpoint level, which will
	 * allow individual deployments to be suspended
	 */
	@SuppressWarnings("unchecked")
	public T trackIndividualEndpoints(java.lang.Boolean value) {
		Object oldValue = this.trackIndividualEndpoints;
		this.trackIndividualEndpoints = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("trackIndividualEndpoints", oldValue,
					value);
		return (T) this;
	}
}