package org.wildfly.swarm.config.management.security_realm;

import org.wildfly.swarm.config.runtime.Addresses;
import java.util.HashMap;
import org.wildfly.swarm.config.runtime.ResourceType;
import org.wildfly.swarm.config.runtime.Implicit;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;

/**
 * A cache to hold the results of previous LDAP interactions.
 */
@Addresses({
		"/core-service=management/security-realm=*/authorization=ldap/username-to-dn=username-filter/cache=by-access-time",
		"/core-service=management/security-realm=*/authorization=ldap/group-search=group-to-principal/cache=by-access-time",
		"/core-service=management/security-realm=*/authorization=ldap/username-to-dn=advanced-filter/cache=by-access-time",
		"/core-service=management/security-realm=*/authorization=ldap/username-to-dn=username-is-dn/cache=by-access-time",
		"/core-service=management/security-realm=*/authorization=ldap/group-search=principal-to-group/cache=by-access-time",
		"/core-service=management/security-realm=*/authentication=ldap/cache=by-access-time"})
@ResourceType("cache")
@Implicit
public class ByAccessTimeCache<T extends ByAccessTimeCache<T>> extends HashMap {

	private String key;
	private PropertyChangeSupport pcs;
	private Boolean cacheFailures;
	private Integer cacheSize;
	private Integer evictionTime;
	private Integer maxCacheSize;

	public ByAccessTimeCache() {
		super();
		this.key = "by-access-time";
		this.pcs = new PropertyChangeSupport(this);
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(
			java.beans.PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	/**
	 * Should failures be cached?
	 */
	@ModelNodeBinding(detypedName = "cache-failures")
	public Boolean cacheFailures() {
		return this.cacheFailures;
	}

	/**
	 * Should failures be cached?
	 */
	@SuppressWarnings("unchecked")
	public T cacheFailures(java.lang.Boolean value) {
		Object oldValue = this.cacheFailures;
		this.cacheFailures = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("cacheFailures", oldValue, value);
		return (T) this;
	}

	/**
	 * The current size of the cache.
	 */
	@ModelNodeBinding(detypedName = "cache-size")
	public Integer cacheSize() {
		return this.cacheSize;
	}

	/**
	 * The current size of the cache.
	 */
	@SuppressWarnings("unchecked")
	public T cacheSize(java.lang.Integer value) {
		Object oldValue = this.cacheSize;
		this.cacheSize = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("cacheSize", oldValue, value);
		return (T) this;
	}

	/**
	 * The time in seconds until an entry should be evicted from the cache.
	 */
	@ModelNodeBinding(detypedName = "eviction-time")
	public Integer evictionTime() {
		return this.evictionTime;
	}

	/**
	 * The time in seconds until an entry should be evicted from the cache.
	 */
	@SuppressWarnings("unchecked")
	public T evictionTime(java.lang.Integer value) {
		Object oldValue = this.evictionTime;
		this.evictionTime = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("evictionTime", oldValue, value);
		return (T) this;
	}

	/**
	 * The maximum size of the cache before the oldest items are removed to make
	 * room for new entries.
	 */
	@ModelNodeBinding(detypedName = "max-cache-size")
	public Integer maxCacheSize() {
		return this.maxCacheSize;
	}

	/**
	 * The maximum size of the cache before the oldest items are removed to make
	 * room for new entries.
	 */
	@SuppressWarnings("unchecked")
	public T maxCacheSize(java.lang.Integer value) {
		Object oldValue = this.maxCacheSize;
		this.maxCacheSize = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("maxCacheSize", oldValue, value);
		return (T) this;
	}
}