package org.wildfly.swarm.config;

import org.wildfly.swarm.config.runtime.Address;
import java.util.HashMap;
import org.wildfly.swarm.config.runtime.ResourceType;
import org.wildfly.swarm.config.runtime.Implicit;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import java.util.List;
import org.wildfly.swarm.config.runtime.Subresource;
import org.wildfly.swarm.config.security.SecurityDomainConsumer;
import org.wildfly.swarm.config.security.SecurityDomainSupplier;
import org.wildfly.swarm.config.security.SecurityDomain;
import org.wildfly.swarm.config.security.ClassicVault;
import org.wildfly.swarm.config.security.ClassicVaultConsumer;
import org.wildfly.swarm.config.security.ClassicVaultSupplier;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;

/**
 * The configuration of the security subsystem.
 */
@Address("/subsystem=security")
@ResourceType("subsystem")
@Implicit
public class Security<T extends Security<T>> extends HashMap {

	private String key;
	private PropertyChangeSupport pcs;
	private SecurityResources subresources = new SecurityResources();
	private Boolean deepCopySubjectMode;

	public Security() {
		super();
		this.key = "security";
		this.pcs = new PropertyChangeSupport(this);
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(
			java.beans.PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	public SecurityResources subresources() {
		return this.subresources;
	}

	/**
	 * Add all SecurityDomain objects to this subresource
	 * 
	 * @return this
	 * @param value
	 *            List of SecurityDomain objects.
	 */
	@SuppressWarnings("unchecked")
	public T securityDomains(java.util.List<SecurityDomain> value) {
		this.subresources.securityDomains = value;
		return (T) this;
	}

	/**
	 * Add the SecurityDomain object to the list of subresources
	 * 
	 * @param value
	 *            The SecurityDomain to add
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T securityDomain(SecurityDomain value) {
		this.subresources.securityDomains.add(value);
		return (T) this;
	}

	/**
	 * Create and configure a SecurityDomain object to the list of subresources
	 * 
	 * @param key
	 *            The key for the SecurityDomain resource
	 * @param config
	 *            The SecurityDomainConsumer to use
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T securityDomain(java.lang.String childKey,
			SecurityDomainConsumer consumer) {
		SecurityDomain<? extends SecurityDomain> child = new SecurityDomain<>(
				childKey);
		if (consumer != null) {
			consumer.accept(child);
		}
		securityDomain(child);
		return (T) this;
	}

	/**
	 * Create and configure a SecurityDomain object to the list of subresources
	 * 
	 * @param key
	 *            The key for the SecurityDomain resource
	 * @return this
	 */
	@SuppressWarnings("unchecked")
	public T securityDomain(java.lang.String childKey) {
		securityDomain(childKey, null);
		return (T) this;
	}

	/**
	 * Install a supplied SecurityDomain object to the list of subresources
	 */
	@SuppressWarnings("unchecked")
	public T securityDomain(SecurityDomainSupplier supplier) {
		securityDomain(supplier.get());
		return (T) this;
	}

	/**
	 * Security Vault for attributes.
	 */
	@SuppressWarnings("unchecked")
	public T classicVault(ClassicVault value) {
		this.subresources.classicVault = value;
		return (T) this;
	}

	/**
	 * Security Vault for attributes.
	 */
	@SuppressWarnings("unchecked")
	public T classicVault(ClassicVaultConsumer consumer) {
		ClassicVault<? extends ClassicVault> child = new ClassicVault<>();
		if (consumer != null) {
			consumer.accept(child);
		}
		this.subresources.classicVault = child;
		return (T) this;
	}

	/**
	 * Security Vault for attributes.
	 */
	@SuppressWarnings("unchecked")
	public T classicVault() {
		ClassicVault<? extends ClassicVault> child = new ClassicVault<>();
		this.subresources.classicVault = child;
		return (T) this;
	}

	/**
	 * Security Vault for attributes.
	 */
	@SuppressWarnings("unchecked")
	public T classicVault(ClassicVaultSupplier supplier) {
		this.subresources.classicVault = supplier.get();
		return (T) this;
	}

	/**
	 * Child mutators for Security
	 */
	public static class SecurityResources {
		/**
		 * Configures a security domain. Authentication, authorization, ACL,
		 * mapping, auditing and identity trust are configured here.
		 */
		private List<SecurityDomain> securityDomains = new java.util.ArrayList<>();
		private ClassicVault classicVault;

		/**
		 * Get the list of SecurityDomain resources
		 * 
		 * @return the list of resources
		 */
		@Subresource
		public List<SecurityDomain> securityDomains() {
			return this.securityDomains;
		}

		public SecurityDomain securityDomain(java.lang.String key) {
			return this.securityDomains.stream()
					.filter(e -> e.getKey().equals(key)).findFirst()
					.orElse(null);
		}
		/**
		 * Security Vault for attributes.
		 */
		@Subresource
		public ClassicVault classicVault() {
			return this.classicVault;
		}
	}

	/**
	 * Sets the copy mode of subjects done by the security managers to be deep
	 * copies that makes copies of the subject principals and credentials if
	 * they are cloneable. It should be set to true if subject include mutable
	 * content that can be corrupted when multiple threads have the same
	 * identity and cache flushes/logout clearing the subject in one thread
	 * results in subject references affecting other threads.
	 */
	@ModelNodeBinding(detypedName = "deep-copy-subject-mode")
	public Boolean deepCopySubjectMode() {
		return this.deepCopySubjectMode;
	}

	/**
	 * Sets the copy mode of subjects done by the security managers to be deep
	 * copies that makes copies of the subject principals and credentials if
	 * they are cloneable. It should be set to true if subject include mutable
	 * content that can be corrupted when multiple threads have the same
	 * identity and cache flushes/logout clearing the subject in one thread
	 * results in subject references affecting other threads.
	 */
	@SuppressWarnings("unchecked")
	public T deepCopySubjectMode(java.lang.Boolean value) {
		Object oldValue = this.deepCopySubjectMode;
		this.deepCopySubjectMode = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("deepCopySubjectMode", oldValue, value);
		return (T) this;
	}
}