package org.wildfly.swarm.config.infinispan.cache_container;

import org.wildfly.swarm.config.runtime.Addresses;
import java.util.HashMap;
import org.wildfly.swarm.config.runtime.ResourceType;
import org.wildfly.swarm.config.runtime.Implicit;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
import java.util.Arrays;

/**
 * The cache eviction configuration.
 */
@Addresses({
		"/subsystem=infinispan/cache-container=*/invalidation-cache=*/component=eviction",
		"/subsystem=infinispan/cache-container=*/local-cache=*/component=eviction",
		"/subsystem=infinispan/cache-container=*/replicated-cache=*/component=eviction",
		"/subsystem=infinispan/cache-container=*/distributed-cache=*/component=eviction"})
@ResourceType("component")
@Implicit
public class EvictionComponent<T extends EvictionComponent<T>> extends HashMap {

	private String key;
	private PropertyChangeSupport pcs;
	private Long evictions;
	private Long maxEntries;
	private Strategy strategy;

	public EvictionComponent() {
		super();
		this.key = "eviction";
		this.pcs = new PropertyChangeSupport(this);
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(
			java.beans.PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	public static enum Strategy {
		NONE("NONE"), UNORDERED("UNORDERED"), FIFO("FIFO"), LRU("LRU"), LIRS(
				"LIRS");
		private final String allowedValue;

		/**
		 * Returns the allowed value for the management model.
		 * 
		 * @return the allowed model value
		 */
		public String getAllowedValue() {
			return allowedValue;
		}

		Strategy(java.lang.String allowedValue) {
			this.allowedValue = allowedValue;
		}

		@Override
		public String toString() {
			return allowedValue;
		}
	}

	/**
	 * The number of cache eviction operations. May return null if the cache is
	 * not started.
	 */
	@ModelNodeBinding(detypedName = "evictions")
	public Long evictions() {
		return this.evictions;
	}

	/**
	 * The number of cache eviction operations. May return null if the cache is
	 * not started.
	 */
	@SuppressWarnings("unchecked")
	public T evictions(java.lang.Long value) {
		Object oldValue = this.evictions;
		this.evictions = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("evictions", oldValue, value);
		return (T) this;
	}

	/**
	 * Maximum number of entries in a cache instance. If selected value is not a
	 * power of two the actual value will default to the least power of two
	 * larger than selected value. -1 means no limit.
	 */
	@ModelNodeBinding(detypedName = "max-entries")
	public Long maxEntries() {
		return this.maxEntries;
	}

	/**
	 * Maximum number of entries in a cache instance. If selected value is not a
	 * power of two the actual value will default to the least power of two
	 * larger than selected value. -1 means no limit.
	 */
	@SuppressWarnings("unchecked")
	public T maxEntries(java.lang.Long value) {
		Object oldValue = this.maxEntries;
		this.maxEntries = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("maxEntries", oldValue, value);
		return (T) this;
	}

	/**
	 * Sets the cache eviction strategy. Available options are 'UNORDERED',
	 * 'FIFO', 'LRU', 'LIRS' and 'NONE' (to disable eviction).
	 */
	@ModelNodeBinding(detypedName = "strategy")
	public Strategy strategy() {
		return this.strategy;
	}

	/**
	 * Sets the cache eviction strategy. Available options are 'UNORDERED',
	 * 'FIFO', 'LRU', 'LIRS' and 'NONE' (to disable eviction).
	 */
	@SuppressWarnings("unchecked")
	public T strategy(Strategy value) {
		Object oldValue = this.strategy;
		this.strategy = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("strategy", oldValue, value);
		return (T) this;
	}
}