package org.wildfly.swarm.config.messaging.activemq.server;

import org.wildfly.swarm.config.runtime.Address;
import java.util.HashMap;
import org.wildfly.swarm.config.runtime.ResourceType;
import org.wildfly.swarm.config.runtime.Implicit;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeListener;
import org.wildfly.swarm.config.runtime.ModelNodeBinding;
import java.util.List;
import java.util.Arrays;
import java.util.stream.Collectors;

/**
 * A messaging resource that allows you to configure High Availability for the
 * ActiveMQ server (the value of ha-policy can be live-only, replication-master,
 * replication-slave, or replication-colocated).
 */
@Address("/subsystem=messaging-activemq/server=*/ha-policy=live-only")
@ResourceType("ha-policy")
@Implicit
public class LiveOnlyHAPolicy<T extends LiveOnlyHAPolicy<T>> extends HashMap
		implements
			org.wildfly.swarm.config.runtime.Keyed {

	private String key;
	private PropertyChangeSupport pcs;
	private Boolean scaleDown;
	private String scaleDownClusterName;
	private List<String> scaleDownConnectors;
	private String scaleDownDiscoveryGroup;
	private String scaleDownGroupName;

	public LiveOnlyHAPolicy() {
		super();
		this.key = "live-only";
		this.pcs = new PropertyChangeSupport(this);
	}

	public String getKey() {
		return this.key;
	}

	/**
	 * Adds a property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener listener) {
		if (null == this.pcs)
			this.pcs = new PropertyChangeSupport(this);
		this.pcs.addPropertyChangeListener(listener);
	}

	/**
	 * Removes a property change listener
	 */
	public void removePropertyChangeListener(
			java.beans.PropertyChangeListener listener) {
		if (this.pcs != null)
			this.pcs.removePropertyChangeListener(listener);
	}

	/**
	 * Configure whether this server send its messages to another live server in
	 * the scale-down cluster when it is shutdown cleanly.
	 */
	@ModelNodeBinding(detypedName = "scale-down")
	public Boolean scaleDown() {
		return this.scaleDown;
	}

	/**
	 * Configure whether this server send its messages to another live server in
	 * the scale-down cluster when it is shutdown cleanly.
	 */
	@SuppressWarnings("unchecked")
	public T scaleDown(java.lang.Boolean value) {
		Object oldValue = this.scaleDown;
		this.scaleDown = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("scaleDown", oldValue, value);
		return (T) this;
	}

	/**
	 * Name of the cluster used to scale down.
	 */
	@ModelNodeBinding(detypedName = "scale-down-cluster-name")
	public String scaleDownClusterName() {
		return this.scaleDownClusterName;
	}

	/**
	 * Name of the cluster used to scale down.
	 */
	@SuppressWarnings("unchecked")
	public T scaleDownClusterName(java.lang.String value) {
		Object oldValue = this.scaleDownClusterName;
		this.scaleDownClusterName = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("scaleDownClusterName", oldValue, value);
		return (T) this;
	}

	/**
	 * List of connectors used to form the scale-down cluster.
	 */
	@ModelNodeBinding(detypedName = "scale-down-connectors")
	public List<String> scaleDownConnectors() {
		return this.scaleDownConnectors;
	}

	/**
	 * List of connectors used to form the scale-down cluster.
	 */
	@SuppressWarnings("unchecked")
	public T scaleDownConnectors(java.util.List<String> value) {
		Object oldValue = this.scaleDownConnectors;
		this.scaleDownConnectors = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("scaleDownConnectors", oldValue, value);
		return (T) this;
	}

	/**
	 * List of connectors used to form the scale-down cluster.
	 */
	@SuppressWarnings("unchecked")
	public T scaleDownConnector(String value) {
		if (this.scaleDownConnectors == null) {
			this.scaleDownConnectors = new java.util.ArrayList<>();
		}
		this.scaleDownConnectors.add(value);
		return (T) this;
	}

	/**
	 * List of connectors used to form the scale-down cluster.
	 */
	@SuppressWarnings("unchecked")
	public T scaleDownConnectors(String... args) {
		scaleDownConnectors(Arrays.stream(args).collect(Collectors.toList()));
		return (T) this;
	}

	/**
	 * Name of the discovery group used to build the scale-down cluster
	 */
	@ModelNodeBinding(detypedName = "scale-down-discovery-group")
	public String scaleDownDiscoveryGroup() {
		return this.scaleDownDiscoveryGroup;
	}

	/**
	 * Name of the discovery group used to build the scale-down cluster
	 */
	@SuppressWarnings("unchecked")
	public T scaleDownDiscoveryGroup(java.lang.String value) {
		Object oldValue = this.scaleDownDiscoveryGroup;
		this.scaleDownDiscoveryGroup = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("scaleDownDiscoveryGroup", oldValue,
					value);
		return (T) this;
	}

	/**
	 * Name of the group used to scale down.
	 */
	@ModelNodeBinding(detypedName = "scale-down-group-name")
	public String scaleDownGroupName() {
		return this.scaleDownGroupName;
	}

	/**
	 * Name of the group used to scale down.
	 */
	@SuppressWarnings("unchecked")
	public T scaleDownGroupName(java.lang.String value) {
		Object oldValue = this.scaleDownGroupName;
		this.scaleDownGroupName = value;
		if (this.pcs != null)
			this.pcs.firePropertyChange("scaleDownGroupName", oldValue, value);
		return (T) this;
	}
}