/**
 * Copyright 2015-2016 Red Hat, Inc, and individual contributors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.wildfly.swarm.container.internal;

import org.jboss.shrinkwrap.api.Archive;
import org.jboss.shrinkwrap.api.ArchivePath;
import org.jboss.shrinkwrap.api.Filter;
import org.jboss.shrinkwrap.impl.base.container.ContainerBase;
import org.jboss.shrinkwrap.impl.base.path.BasicPath;
import org.wildfly.swarm.spi.api.JARArchive;

/**
 * @author Bob McWhirter
 */
public class JARArchiveImpl extends ContainerBase<JARArchive> implements JARArchive {

    // -------------------------------------------------------------------------------------||
    // Class Members ----------------------------------------------------------------------||
    // -------------------------------------------------------------------------------------||

    /**
     * Create a new JAXRS Archive with any type storage engine as backing.
     *
     * @param delegate The storage backing.
     */
    public JARArchiveImpl(Archive<?> delegate) {
        super(JARArchive.class, delegate);
    }

    /**
     * Libraries are not supported by JavaArchive.
     *
     * @throws UnsupportedOperationException Libraries are not supported by JavaArchive
     */
    @Override
    public ArchivePath getLibraryPath() {
        throw new UnsupportedOperationException("JavaArchive spec does not support Libraries");
    }

    /**
     * {@inheritDoc}
     *
     * @see ContainerBase#getManifestPath()
     */
    @Override
    protected ArchivePath getManifestPath() {
        return PATH_MANIFEST;
    }

    // -------------------------------------------------------------------------------------||
    // Instance Members -------------------------------------------------------------------||
    // -------------------------------------------------------------------------------------||

    // -------------------------------------------------------------------------------------||
    // Constructor ------------------------------------------------------------------------||
    // -------------------------------------------------------------------------------------||

    /**
     * {@inheritDoc}
     *
     * @see ContainerBase#getClassesPath()
     */
    @Override
    protected ArchivePath getClassesPath() {
        return PATH_CLASSES;
    }

    @Override
    public JARArchive addPackage(Package pack) throws IllegalArgumentException {
        AddPackageWarning.addPackage();
        return super.addPackage(pack);
    }

    @Override
    public JARArchive addPackage(String pack) throws IllegalArgumentException {
        AddPackageWarning.addPackage();
        return super.addPackage(pack);
    }

    @Override
    public JARArchive addPackages(boolean recursive, Filter<ArchivePath> filter, String... packageNames) throws IllegalArgumentException {
        AddPackageWarning.addPackage();
        return super.addPackages(recursive, filter, packageNames);
    }

    @Override
    public JARArchive addPackages(boolean recursive, Filter<ArchivePath> filter, Package... packages) throws IllegalArgumentException {
        AddPackageWarning.addPackage();
        return super.addPackages(recursive, filter, packages);
    }

    @Override
    public JARArchive addPackages(boolean recursive, Package... packages) throws IllegalArgumentException {
        AddPackageWarning.addPackage();
        return super.addPackages(recursive, packages);
    }

    @Override
    public JARArchive addPackages(boolean recursive, String... packages) throws IllegalArgumentException {
        AddPackageWarning.addPackage();
        return super.addPackages(recursive, packages);
    }

    @Override
    public JARArchive addDefaultPackage() {
        AddPackageWarning.addPackage();
        return super.addDefaultPackage();
    }

    // -------------------------------------------------------------------------------------||
    // Required Implementations -----------------------------------------------------------||
    // -------------------------------------------------------------------------------------||

    /**
     * {@inheritDoc}
     *
     * @see ContainerBase#getResourcePath()
     */
    @Override
    protected ArchivePath getResourcePath() {
        return PATH_RESOURCE;
    }

    /**
     * Path to the manifests inside of the Archive.
     */
    private static final ArchivePath PATH_MANIFEST = new BasicPath("META-INF");

    /**
     * Path to the resources inside of the Archive.
     */
    private static final ArchivePath PATH_RESOURCE = new BasicPath("/");

    /**
     * Path to the classes inside of the Archive.
     */
    private static final ArchivePath PATH_CLASSES = new BasicPath("/");

}
