/*
 * Decompiled with CFR 0.152.
 */
package org.wildfly.security.auth.realm.token.validator;

import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.PublicKey;
import java.security.Signature;
import java.security.SignatureException;
import java.util.Arrays;
import java.util.Base64;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.Set;
import javax.json.Json;
import javax.json.JsonArray;
import javax.json.JsonObject;
import javax.json.JsonString;
import javax.json.JsonValue;
import org.wildfly.common.Assert;
import org.wildfly.security._private.ElytronMessages;
import org.wildfly.security.auth.realm.token.TokenValidator;
import org.wildfly.security.auth.server.RealmUnavailableException;
import org.wildfly.security.authz.Attributes;
import org.wildfly.security.evidence.BearerTokenEvidence;
import org.wildfly.security.pem.Pem;
import org.wildfly.security.pem.PemEntry;
import org.wildfly.security.util.ByteIterator;
import org.wildfly.security.util.CodePointIterator;
import org.wildfly.security.util.JsonUtil;

public class JwtValidator
implements TokenValidator {
    private final Set<String> issuers;
    private final Set<String> audiences;
    private final PublicKey publicKey;

    public static Builder builder() {
        return new Builder();
    }

    JwtValidator(Builder configuration) {
        this.issuers = Assert.checkNotNullParam("issuers", configuration.issuers);
        this.audiences = Assert.checkNotNullParam("audience", configuration.audience);
        this.publicKey = configuration.publicKey;
        if (this.issuers.isEmpty()) {
            ElytronMessages.log.tokenRealmJwtWarnNoIssuerIgnoringIssuerCheck();
        }
        if (this.audiences.isEmpty()) {
            ElytronMessages.log.tokenRealmJwtWarnNoAudienceIgnoringAudienceCheck();
        }
        if (this.publicKey == null) {
            ElytronMessages.log.tokenRealmJwtWarnNoPublicKeyIgnoringSignatureCheck();
        }
    }

    @Override
    public Attributes validate(BearerTokenEvidence evidence) throws RealmUnavailableException {
        Assert.checkNotNullParam("evidence", evidence);
        String jwt = evidence.getToken();
        String[] parts = jwt.split("\\.", -1);
        if (parts.length < 3) {
            throw ElytronMessages.log.tokenRealmJwtInvalidFormat();
        }
        String encodedHeader = parts[0];
        String encodedClaims = parts[1];
        String encodedSignature = parts[2];
        JsonObject claims = this.extractClaims(encodedClaims);
        if (this.verifySignature(encodedHeader, encodedClaims, encodedSignature) && this.hasValidIssuer(claims) && this.hasValidAudience(claims) && this.verifyTimeConstraints(claims)) {
            return JsonUtil.toAttributes(claims);
        }
        return null;
    }

    private boolean verifyTimeConstraints(JsonObject claims) {
        boolean expired;
        int currentTime = this.currentTimeInSeconds();
        boolean bl = expired = currentTime > claims.getInt("exp", -1);
        if (expired) {
            ElytronMessages.log.debug("Token expired");
            return false;
        }
        if (claims.containsKey("nbf")) {
            boolean notBefore;
            boolean bl2 = notBefore = currentTime >= claims.getInt("nbf");
            if (!notBefore) {
                ElytronMessages.log.debugf("Token is before [%s]", (Object)notBefore);
                return false;
            }
        }
        return true;
    }

    private JsonObject extractClaims(String encodedClaims) throws RealmUnavailableException {
        try {
            Base64.Decoder urlDecoder = Base64.getUrlDecoder();
            CodePointIterator decodedClaims = CodePointIterator.ofUtf8Bytes(urlDecoder.decode(encodedClaims));
            return Json.createReader(decodedClaims.asUtf8().asInputStream()).readObject();
        }
        catch (Exception cause) {
            throw ElytronMessages.log.tokenRealmJwtParseFailed(cause);
        }
    }

    private boolean verifySignature(String encodedHeader, String encodedClaims, String encodedSignature) throws RealmUnavailableException {
        if (this.publicKey == null) {
            return true;
        }
        try {
            Base64.Decoder urlDecoder = Base64.getUrlDecoder();
            byte[] decodedSignature = urlDecoder.decode(encodedSignature);
            boolean verify = ByteIterator.ofBytes(decodedSignature).verify(this.createSignature(encodedHeader, encodedClaims));
            if (!verify) {
                ElytronMessages.log.debug("Signature verification failed");
            }
            return verify;
        }
        catch (Exception cause) {
            throw ElytronMessages.log.tokenRealmJwtSignatureCheckFailed(cause);
        }
    }

    private boolean hasValidAudience(JsonObject claims) throws RealmUnavailableException {
        if (this.audiences.isEmpty()) {
            return true;
        }
        JsonValue audience = (JsonValue)claims.get("aud");
        if (audience == null) {
            ElytronMessages.log.debug("Token does not contain an audience claim");
            return false;
        }
        JsonArray audClaimArray = JsonValue.ValueType.STRING.equals((Object)audience.getValueType()) ? Json.createArrayBuilder().add(audience).build() : (JsonArray)audience;
        boolean valid = audClaimArray.stream().map(jsonValue -> (JsonString)jsonValue).anyMatch(audience1 -> this.audiences.contains(audience1.getString()));
        if (!valid) {
            ElytronMessages.log.debugf("Audience check failed. Provided [%s] but was expected [%s].", (Object)audClaimArray.toArray(), (Object)this.audiences);
        }
        return valid;
    }

    private boolean hasValidIssuer(JsonObject claims) throws RealmUnavailableException {
        if (this.issuers.isEmpty()) {
            return true;
        }
        String issuer = claims.getString("iss", null);
        if (issuer == null) {
            ElytronMessages.log.debug("Token does not contain an issuer claim");
            return false;
        }
        boolean valid = this.issuers.contains(issuer);
        if (!valid) {
            ElytronMessages.log.debugf("Issuer check failed. Provided [%s] but was expected [%s].", (Object)issuer, (Object)this.issuers);
        }
        return valid;
    }

    private Signature createSignature(String encodedHeader, String encodedClaims) throws NoSuchAlgorithmException, InvalidKeyException, SignatureException, RealmUnavailableException {
        Signature signature = Signature.getInstance(this.resolveAlgorithm(encodedHeader));
        signature.initVerify(this.publicKey);
        signature.update((encodedHeader + "." + encodedClaims).getBytes());
        return signature;
    }

    private String resolveAlgorithm(String part) throws RealmUnavailableException {
        byte[] headerDecoded = Base64.getUrlDecoder().decode(part);
        JsonObject headers = Json.createReader(ByteIterator.ofBytes(headerDecoded).asInputStream()).readObject();
        JsonString algClaim = (JsonString)headers.get("alg");
        if (algClaim == null) {
            throw ElytronMessages.log.tokenRealmJwtSignatureInvalidAlgorithm("not_provided");
        }
        String algorithm = algClaim.getString();
        ElytronMessages.log.debugf("Token is using algorithm [%s]", (Object)algorithm);
        switch (algorithm) {
            case "RS256": {
                return "SHA256withRSA";
            }
            case "RS384": {
                return "SHA384withRSA";
            }
            case "RS512": {
                return "SHA512withRSA";
            }
        }
        throw ElytronMessages.log.tokenRealmJwtSignatureInvalidAlgorithm(algorithm);
    }

    private int currentTimeInSeconds() {
        return (int)(System.currentTimeMillis() / 1000L);
    }

    public static class Builder {
        private Set<String> issuers = new LinkedHashSet<String>();
        private Set<String> audience = new LinkedHashSet<String>();
        private PublicKey publicKey;

        private Builder() {
        }

        public Builder issuer(String ... issuer) {
            this.issuers.addAll(Arrays.asList(issuer));
            return this;
        }

        public Builder audience(String ... audience) {
            this.audience.addAll(Arrays.asList(audience));
            return this;
        }

        public Builder publicKey(byte[] publicKeyPem) {
            Iterator<PemEntry<?>> pemEntryIterator = Pem.parsePemContent(CodePointIterator.ofUtf8Bytes(publicKeyPem));
            PublicKey publicKey = pemEntryIterator.next().tryCast(PublicKey.class);
            if (publicKey == null) {
                throw ElytronMessages.log.tokenRealmJwtInvalidPublicKeyPem();
            }
            this.publicKey = publicKey;
            return this;
        }

        public Builder publicKey(PublicKey publicKey) {
            this.publicKey = publicKey;
            return this;
        }

        public JwtValidator build() {
            return new JwtValidator(this);
        }
    }
}

