/*
 * Decompiled with CFR 0.152.
 */
package org.wildfly.security.password;

import java.io.ByteArrayInputStream;
import java.nio.charset.StandardCharsets;
import java.security.spec.InvalidKeySpecException;
import java.util.Locale;
import java.util.NoSuchElementException;
import java.util.concurrent.ThreadLocalRandom;
import org.wildfly.security.password.spec.BCryptPasswordSpec;
import org.wildfly.security.password.spec.BSDUnixDESCryptPasswordSpec;
import org.wildfly.security.password.spec.PasswordSpec;
import org.wildfly.security.password.spec.SimpleDigestPasswordSpec;
import org.wildfly.security.password.spec.SunUnixMD5CryptPasswordSpec;
import org.wildfly.security.password.spec.UnixDESCryptPasswordSpec;
import org.wildfly.security.password.spec.UnixMD5CryptPasswordSpec;
import org.wildfly.security.password.spec.UnixSHACryptPasswordSpec;
import org.wildfly.security.util.Alphabet;
import org.wildfly.security.util.ByteIterator;
import org.wildfly.security.util.CodePointIterator;

public final class PasswordUtil {
    private static final int A_CRYPT_MD5 = 1;
    private static final int A_BCRYPT = 2;
    private static final int A_BSD_NT_HASH = 3;
    private static final int A_CRYPT_SHA_256 = 4;
    private static final int A_CRYPT_SHA_512 = 5;
    private static final int A_SUN_CRYPT_MD5 = 6;
    private static final int A_APACHE_HTDIGEST = 7;
    private static final int A_BSD_CRYPT_DES = 8;
    private static final int A_CRYPT_DES = 9;
    private static final int A_DIGEST_MD2 = 10;
    private static final int A_DIGEST_MD5 = 11;
    private static final int A_DIGEST_SHA_1 = 12;
    private static final int A_DIGEST_SHA_256 = 13;
    private static final int A_DIGEST_SHA_384 = 14;
    private static final int A_DIGEST_SHA_512 = 15;
    private static final int A_SUN_CRYPT_MD5_BARE_SALT = 16;
    private static final int[] MD5_IDX = new int[]{12, 6, 0, 13, 7, 1, 14, 8, 2, 15, 9, 3, 5, 10, 4, 11};
    private static final int[] MD5_IDX_REV = PasswordUtil.inverse(MD5_IDX);
    private static final int[] SHA_256_IDX = new int[]{20, 10, 0, 11, 1, 21, 2, 22, 12, 23, 13, 3, 14, 4, 24, 5, 25, 15, 26, 16, 6, 17, 7, 27, 8, 28, 18, 29, 19, 9, 30, 31};
    private static final int[] SHA_256_IDX_REV = PasswordUtil.inverse(SHA_256_IDX);
    private static final int[] SHA_512_IDX = new int[]{42, 21, 0, 1, 43, 22, 23, 2, 44, 45, 24, 3, 4, 46, 25, 26, 5, 47, 48, 27, 6, 7, 49, 28, 29, 8, 50, 51, 30, 9, 10, 52, 31, 32, 11, 53, 54, 33, 12, 13, 55, 34, 35, 14, 56, 57, 36, 15, 16, 58, 37, 38, 17, 59, 60, 39, 18, 19, 61, 40, 41, 20, 62, 63};
    private static final int[] SHA_512_IDX_REV = PasswordUtil.inverse(SHA_512_IDX);

    private PasswordUtil() {
    }

    private static int doIdentifyAlgorithm(char[] chars) {
        if (chars.length < 5) {
            return 0;
        }
        if (chars[0] == '$') {
            if (chars[2] == '$') {
                switch (chars[1]) {
                    case '1': {
                        return 1;
                    }
                    case '2': {
                        return 2;
                    }
                    case '3': {
                        return 3;
                    }
                    case '5': {
                        return 4;
                    }
                    case '6': {
                        return 5;
                    }
                }
                return 0;
            }
            if (chars[3] == '$') {
                if (chars[1] == '2') {
                    if (chars[2] == 'a' || chars[2] == 'x' || chars[2] == 'y') {
                        return 2;
                    }
                    return 0;
                }
                return 0;
            }
            if (chars[4] == '$' || chars[4] == ',') {
                if (chars[1] == 'm' && chars[2] == 'd' && chars[3] == '5') {
                    int idx = PasswordUtil.lastIndexOf(chars, '$');
                    if (idx > 0) {
                        if (chars[idx - 1] == '$') {
                            return 6;
                        }
                        return 16;
                    }
                    return 0;
                }
                return 0;
            }
            if (chars[5] == '$') {
                if (chars[1] == 'a' && chars[2] == 'p' && chars[3] == 'r' && chars[4] == '1') {
                    return 7;
                }
                return 0;
            }
            return 0;
        }
        if (chars[0] == '_') {
            return 8;
        }
        if (chars[0] == '[') {
            int idx = PasswordUtil.indexOf(chars, ']');
            if (idx != -1) {
                switch (new String(chars, 1, idx - 1).toLowerCase(Locale.US)) {
                    case "md2": {
                        return 10;
                    }
                    case "md5": {
                        return 11;
                    }
                    case "sha-1": {
                        return 12;
                    }
                    case "sha-256": {
                        return 13;
                    }
                    case "sha-384": {
                        return 14;
                    }
                    case "sha-512": {
                        return 15;
                    }
                }
                return 0;
            }
            return 0;
        }
        if (chars.length == 13) {
            return 9;
        }
        return 0;
    }

    public static String identifyAlgorithm(char[] chars) {
        return PasswordUtil.getAlgorithmNameString(PasswordUtil.doIdentifyAlgorithm(chars));
    }

    static String getAlgorithmNameString(int id) {
        switch (id) {
            case 1: {
                return "crypt-md5";
            }
            case 2: {
                return "bcrypt";
            }
            case 3: {
                return "bsd-nt-hash";
            }
            case 4: {
                return "crypt-sha-256";
            }
            case 5: {
                return "crypt-sha-512";
            }
            case 6: {
                return "sun-crypt-md5";
            }
            case 7: {
                return "apache-htdigest";
            }
            case 8: {
                return "bsd-crypt-des";
            }
            case 9: {
                return "crypt-des";
            }
            case 10: {
                return "digest-md2";
            }
            case 11: {
                return "digest-md5";
            }
            case 12: {
                return "digest-sha-1";
            }
            case 13: {
                return "digest-sha-256";
            }
            case 14: {
                return "digest-sha-384";
            }
            case 15: {
                return "digest-sha-512";
            }
            case 16: {
                return "sun-crypt-md5-bare-salt";
            }
        }
        return null;
    }

    public static String identifyAlgorithm(String string) {
        return PasswordUtil.identifyAlgorithm(string.toCharArray());
    }

    public static char[] getCryptStringChars(PasswordSpec passwordSpec) throws InvalidKeySpecException {
        StringBuilder b = PasswordUtil.getCryptStringToBuilder(passwordSpec);
        char[] chars = new char[b.length()];
        b.getChars(0, b.length(), chars, 0);
        return chars;
    }

    public static String getCryptString(PasswordSpec passwordSpec) throws InvalidKeySpecException {
        return PasswordUtil.getCryptStringToBuilder(passwordSpec).toString();
    }

    private static StringBuilder getCryptStringToBuilder(PasswordSpec passwordSpec) throws InvalidKeySpecException {
        if (passwordSpec == null) {
            throw new IllegalArgumentException("passwordSpec is null");
        }
        StringBuilder b = new StringBuilder();
        if (passwordSpec instanceof BCryptPasswordSpec) {
            BCryptPasswordSpec spec = (BCryptPasswordSpec)passwordSpec;
            b.append("$2a$");
            if (spec.getIterationCount() < 10) {
                b.append(0);
            }
            b.append(spec.getIterationCount());
            b.append("$");
            ByteIterator.ofBytes(spec.getSalt()).base64Encode(Alphabet.Base64Alphabet.BCRYPT, false).drainTo(b);
            ByteIterator.ofBytes(spec.getHashBytes()).base64Encode(Alphabet.Base64Alphabet.BCRYPT, false).drainTo(b);
        } else if (passwordSpec instanceof BSDUnixDESCryptPasswordSpec) {
            b.append('_');
            BSDUnixDESCryptPasswordSpec spec = (BSDUnixDESCryptPasswordSpec)passwordSpec;
            int iterationCount = spec.getIterationCount();
            b.appendCodePoint(Alphabet.Base64Alphabet.MOD_CRYPT.encode(iterationCount & 0x3F));
            b.appendCodePoint(Alphabet.Base64Alphabet.MOD_CRYPT.encode(iterationCount >> 6 & 0x3F));
            b.appendCodePoint(Alphabet.Base64Alphabet.MOD_CRYPT.encode(iterationCount >> 12 & 0x3F));
            b.appendCodePoint(Alphabet.Base64Alphabet.MOD_CRYPT.encode(iterationCount >> 18 & 0x3F));
            int salt = spec.getSalt();
            b.appendCodePoint(Alphabet.Base64Alphabet.MOD_CRYPT.encode(salt & 0x3F));
            b.appendCodePoint(Alphabet.Base64Alphabet.MOD_CRYPT.encode(salt >> 6 & 0x3F));
            b.appendCodePoint(Alphabet.Base64Alphabet.MOD_CRYPT.encode(salt >> 12 & 0x3F));
            b.appendCodePoint(Alphabet.Base64Alphabet.MOD_CRYPT.encode(salt >> 18 & 0x3F));
            ByteIterator.ofBytes(spec.getHash()).base64Encode(Alphabet.Base64Alphabet.MOD_CRYPT, false).drainTo(b);
        } else if (passwordSpec instanceof SimpleDigestPasswordSpec) {
            SimpleDigestPasswordSpec spec = (SimpleDigestPasswordSpec)passwordSpec;
            String algorithm = spec.getAlgorithm();
            b.append('[').append(algorithm).append(']');
            ByteIterator.ofBytes(spec.getDigest()).base64Encode().drainTo(b);
        } else if (passwordSpec instanceof UnixDESCryptPasswordSpec) {
            UnixDESCryptPasswordSpec spec = (UnixDESCryptPasswordSpec)passwordSpec;
            short salt = spec.getSalt();
            b.appendCodePoint(Alphabet.Base64Alphabet.MOD_CRYPT.encode(salt & 0x3F));
            b.appendCodePoint(Alphabet.Base64Alphabet.MOD_CRYPT.encode(salt >> 6 & 0x3F));
            ByteIterator.ofBytes(spec.getHash()).base64Encode(Alphabet.Base64Alphabet.MOD_CRYPT, false).drainTo(b);
        } else if (passwordSpec instanceof UnixMD5CryptPasswordSpec) {
            byte[] salt;
            b.append("$1$");
            UnixMD5CryptPasswordSpec spec = (UnixMD5CryptPasswordSpec)passwordSpec;
            for (byte sb : salt = spec.getSalt()) {
                b.append((char)(sb & 0xFF));
            }
            b.append('$');
            ByteIterator.ofBytes(spec.getHash(), MD5_IDX).base64Encode(Alphabet.Base64Alphabet.MOD_CRYPT_LE, false).drainTo(b);
        } else if (passwordSpec instanceof SunUnixMD5CryptPasswordSpec) {
            SunUnixMD5CryptPasswordSpec spec = (SunUnixMD5CryptPasswordSpec)passwordSpec;
            int iterationCount = spec.getIterationCount();
            if (iterationCount > 0) {
                b.append("$md5,rounds=").append(iterationCount).append('$');
            } else {
                b.append("$md5$");
            }
            byte[] salt = spec.getSalt();
            for (byte sb : salt) {
                b.append((char)(sb & 0xFF));
            }
            switch (spec.getAlgorithm()) {
                case "sun-crypt-md5": {
                    b.append("$$");
                    break;
                }
                case "sun-crypt-md5-bare-salt": {
                    b.append("$");
                    break;
                }
                default: {
                    throw new InvalidKeySpecException("Unrecognized key spec algorithm");
                }
            }
            ByteIterator.ofBytes(spec.getHash(), MD5_IDX).base64Encode(Alphabet.Base64Alphabet.MOD_CRYPT_LE, false).drainTo(b);
        } else if (passwordSpec instanceof UnixSHACryptPasswordSpec) {
            byte[] salt;
            int[] interleave;
            UnixSHACryptPasswordSpec spec = (UnixSHACryptPasswordSpec)passwordSpec;
            switch (spec.getAlgorithm()) {
                case "crypt-sha-256": {
                    b.append("$5$");
                    interleave = SHA_256_IDX;
                    break;
                }
                case "crypt-sha-512": {
                    b.append("$6$");
                    interleave = SHA_512_IDX;
                    break;
                }
                default: {
                    throw new InvalidKeySpecException("Unrecognized key spec algorithm");
                }
            }
            int iterationCount = spec.getIterationCount();
            if (iterationCount != 5000) {
                b.append("rounds=").append(iterationCount).append('$');
            }
            for (byte sb : salt = spec.getSalt()) {
                b.append((char)(sb & 0xFF));
            }
            b.append('$');
            ByteIterator.ofBytes(spec.getHash(), interleave).base64Encode(Alphabet.Base64Alphabet.MOD_CRYPT_LE, false).drainTo(b);
        } else {
            throw new InvalidKeySpecException("Password spec cannot be rendered as a string");
        }
        return b;
    }

    public static PasswordSpec parseCryptString(String cryptString) throws InvalidKeySpecException {
        if (cryptString == null) {
            throw new IllegalArgumentException("cryptString is null");
        }
        return PasswordUtil.parseCryptString(cryptString.toCharArray());
    }

    public static PasswordSpec parseCryptString(char[] cryptString) throws InvalidKeySpecException {
        if (cryptString == null) {
            throw new IllegalArgumentException("cryptString is null");
        }
        int algorithmId = PasswordUtil.doIdentifyAlgorithm(cryptString);
        switch (algorithmId) {
            case 1: {
                return PasswordUtil.parseUnixMD5CryptPasswordString(cryptString);
            }
            case 2: {
                return PasswordUtil.parseBCryptPasswordString(cryptString);
            }
            case 3: {
                throw new UnsupportedOperationException("not supported yet");
            }
            case 4: {
                return PasswordUtil.parseUnixSHA256CryptPasswordString(cryptString);
            }
            case 5: {
                return PasswordUtil.parseUnixSHA512CryptPasswordString(cryptString);
            }
            case 6: {
                return PasswordUtil.parseSunUnixMD5CryptPasswordString("sun-crypt-md5", cryptString);
            }
            case 16: {
                return PasswordUtil.parseSunUnixMD5CryptPasswordString("sun-crypt-md5-bare-salt", cryptString);
            }
            case 7: {
                throw new UnsupportedOperationException("not supported yet");
            }
            case 8: {
                return PasswordUtil.parseBSDUnixDESCryptPasswordString(cryptString);
            }
            case 9: {
                return PasswordUtil.parseUnixDESCryptPasswordString(cryptString);
            }
            case 10: 
            case 11: 
            case 12: 
            case 13: 
            case 14: 
            case 15: {
                return PasswordUtil.parseSimpleDigestPasswordString(algorithmId, cryptString);
            }
        }
        throw new InvalidKeySpecException("Unknown crypt string algorithm");
    }

    private static int parseModCryptIterationCount(CodePointIterator reader, int minIterations, int maxIterations, int defaultIterations) throws InvalidKeySpecException {
        int iterationCount;
        CodePointIterator dr = reader.delimitedBy(36);
        try {
            if (dr.limitedTo(7).contentEquals(CodePointIterator.ofString("rounds="))) {
                iterationCount = 0;
                while (dr.hasNext()) {
                    int ch = dr.next();
                    if (iterationCount != maxIterations) {
                        if (ch < 48 || ch > 57 || (iterationCount = (iterationCount << 3) + (iterationCount << 1) + ch - 48) <= maxIterations) continue;
                        iterationCount = maxIterations;
                        continue;
                    }
                    throw new InvalidKeySpecException("Invalid character encountered");
                }
                if (!reader.hasNext()) {
                    throw new InvalidKeySpecException("No iteration count terminator given");
                }
                reader.next();
            } else {
                iterationCount = defaultIterations;
            }
        }
        catch (NoSuchElementException ignored) {
            throw new InvalidKeySpecException("Unexpected end of input string");
        }
        return Math.max(minIterations, iterationCount);
    }

    private static int[] inverse(int[] orig) {
        int[] n = new int[orig.length];
        for (int i = 0; i < orig.length; ++i) {
            n[orig[i]] = i;
        }
        return n;
    }

    private static SimpleDigestPasswordSpec parseSimpleDigestPasswordString(int algorithmId, char[] cryptString) throws InvalidKeySpecException {
        int initialLen;
        switch (algorithmId) {
            case 10: 
            case 11: {
                initialLen = "[mdX]".length();
                break;
            }
            case 12: {
                initialLen = "[sha-1]".length();
                break;
            }
            case 13: 
            case 14: 
            case 15: {
                initialLen = "[sha-XXX]".length();
                break;
            }
            default: {
                throw new IllegalStateException();
            }
        }
        byte[] bytes = CodePointIterator.ofChars(cryptString, 0, initialLen).base64Decode().drain();
        return new SimpleDigestPasswordSpec(PasswordUtil.getAlgorithmNameString(algorithmId), bytes);
    }

    private static UnixSHACryptPasswordSpec parseUnixSHA256CryptPasswordString(char[] cryptString) throws InvalidKeySpecException {
        assert (cryptString[0] == '$');
        assert (cryptString[1] == '5');
        assert (cryptString[2] == '$');
        return PasswordUtil.parseUnixSHACryptPasswordSpec(cryptString, SHA_256_IDX_REV, "crypt-sha-256");
    }

    private static UnixSHACryptPasswordSpec parseUnixSHA512CryptPasswordString(char[] cryptString) throws InvalidKeySpecException {
        assert (cryptString[0] == '$');
        assert (cryptString[1] == '6');
        assert (cryptString[2] == '$');
        return PasswordUtil.parseUnixSHACryptPasswordSpec(cryptString, SHA_512_IDX_REV, "crypt-sha-512");
    }

    private static UnixSHACryptPasswordSpec parseUnixSHACryptPasswordSpec(char[] cryptString, int[] table, String algorithm) throws InvalidKeySpecException {
        CodePointIterator r = CodePointIterator.ofChars(cryptString, 3);
        try {
            int iterationCount = PasswordUtil.parseModCryptIterationCount(r, 1000, 999999999, 5000);
            byte[] salt = r.delimitedBy(36).drainToString().getBytes(StandardCharsets.ISO_8859_1);
            if (!r.hasNext()) {
                throw new InvalidKeySpecException("No salt terminator given");
            }
            r.next();
            byte[] decoded = r.base64Decode(Alphabet.Base64Alphabet.MOD_CRYPT_LE, false).limitedTo(table.length).drain();
            if (decoded.length != table.length) {
                throw new IllegalArgumentException("Invalid hash length");
            }
            byte[] hash = ByteIterator.ofBytes(decoded, table).drain();
            return new UnixSHACryptPasswordSpec(algorithm, hash, salt, iterationCount);
        }
        catch (NoSuchElementException ignored) {
            throw new InvalidKeySpecException("Unexpected end of password string", ignored);
        }
    }

    private static UnixMD5CryptPasswordSpec parseUnixMD5CryptPasswordString(char[] cryptString) throws InvalidKeySpecException {
        assert (cryptString[0] == '$');
        assert (cryptString[1] == '1');
        assert (cryptString[2] == '$');
        CodePointIterator r = CodePointIterator.ofChars(cryptString, 3);
        try {
            byte[] salt = r.delimitedBy(36).drainToString().getBytes(StandardCharsets.ISO_8859_1);
            if (!r.hasNext()) {
                throw new InvalidKeySpecException("No salt terminator given");
            }
            r.next();
            byte[] decoded = r.base64Decode(Alphabet.Base64Alphabet.MOD_CRYPT_LE, false).limitedTo(MD5_IDX_REV.length).drain();
            if (decoded.length != MD5_IDX.length) {
                throw new IllegalArgumentException("Invalid hash length");
            }
            byte[] hash = ByteIterator.ofBytes(decoded, MD5_IDX_REV).drain();
            return new UnixMD5CryptPasswordSpec(hash, salt);
        }
        catch (NoSuchElementException ignored) {
            throw new InvalidKeySpecException("Unexpected end of password string");
        }
    }

    private static SunUnixMD5CryptPasswordSpec parseSunUnixMD5CryptPasswordString(String algorithm, char[] cryptString) throws InvalidKeySpecException {
        assert (cryptString[0] == '$');
        assert (cryptString[1] == 'm');
        assert (cryptString[2] == 'd');
        assert (cryptString[3] == '5');
        assert (cryptString[4] == '$' || cryptString[4] == ',');
        CodePointIterator r = CodePointIterator.ofChars(cryptString, 5);
        try {
            byte[] decoded;
            int iterationCount = cryptString[4] == ',' ? PasswordUtil.parseModCryptIterationCount(r, 0, 0x7FFFEFFF, 0) : 0;
            byte[] salt = r.delimitedBy(36).drainToString().getBytes(StandardCharsets.ISO_8859_1);
            if (!r.hasNext()) {
                throw new InvalidKeySpecException("No salt terminator given");
            }
            r.next();
            if (algorithm.equals("sun-crypt-md5") && r.hasNext() && r.peekNext() == 36) {
                r.next();
            }
            if ((decoded = r.base64Decode(Alphabet.Base64Alphabet.MOD_CRYPT_LE, false).limitedTo(MD5_IDX_REV.length).drain()).length != MD5_IDX.length) {
                throw new IllegalArgumentException("Invalid hash length");
            }
            byte[] hash = ByteIterator.ofBytes(decoded, MD5_IDX_REV).drain();
            return new SunUnixMD5CryptPasswordSpec(algorithm, hash, salt, iterationCount);
        }
        catch (NoSuchElementException ignored) {
            throw new InvalidKeySpecException("Unexpected end of password string");
        }
    }

    private static BCryptPasswordSpec parseBCryptPasswordString(char[] cryptString) throws InvalidKeySpecException {
        assert (cryptString[0] == '$');
        assert (cryptString[1] == '2');
        char minor = '\u0000';
        if (cryptString[2] != '$') {
            minor = cryptString[2];
            if (minor != 'a' && minor != 'x' && minor != 'y') {
                throw new InvalidKeySpecException("Invalid minor version");
            }
            assert (cryptString[3] == '$');
        }
        CodePointIterator r = CodePointIterator.ofChars(cryptString, minor == '\u0000' ? 3 : 4);
        try {
            int cost = Integer.parseInt(r.limitedTo(2).drainToString());
            if (r.hasNext() && r.peekNext() != 36) {
                throw new InvalidKeySpecException("Invalid cost: must be a two digit integer");
            }
            if (!r.hasNext()) {
                throw new InvalidKeySpecException("Unexpected end of password string");
            }
            r.next();
            byte[] decodedSalt = r.limitedTo(22).base64Decode(Alphabet.Base64Alphabet.BCRYPT, false).drain();
            byte[] decodedPassword = r.limitedTo(31).base64Decode(Alphabet.Base64Alphabet.BCRYPT, false).drain();
            return new BCryptPasswordSpec(decodedPassword, decodedSalt, cost);
        }
        catch (NoSuchElementException ignored) {
            throw new InvalidKeySpecException("Unexpected end of password string");
        }
    }

    private static UnixDESCryptPasswordSpec parseUnixDESCryptPasswordString(char[] cryptString) throws InvalidKeySpecException {
        assert (cryptString.length == 13);
        CodePointIterator r = CodePointIterator.ofChars(cryptString);
        int s0 = Alphabet.Base64Alphabet.MOD_CRYPT.decode(r.next());
        int s1 = Alphabet.Base64Alphabet.MOD_CRYPT.decode(r.next());
        short salt = (short)(s0 | s1 << 6);
        byte[] hash = r.base64Decode(Alphabet.Base64Alphabet.MOD_CRYPT, false).limitedTo(8).drain();
        return new UnixDESCryptPasswordSpec(hash, salt);
    }

    private static BSDUnixDESCryptPasswordSpec parseBSDUnixDESCryptPasswordString(char[] cryptString) throws InvalidKeySpecException {
        assert (cryptString.length == 20);
        assert (cryptString[0] == '_');
        CodePointIterator r = CodePointIterator.ofChars(cryptString, 1);
        int s0 = Alphabet.Base64Alphabet.MOD_CRYPT.decode(r.next());
        int s1 = Alphabet.Base64Alphabet.MOD_CRYPT.decode(r.next());
        int s2 = Alphabet.Base64Alphabet.MOD_CRYPT.decode(r.next());
        int s3 = Alphabet.Base64Alphabet.MOD_CRYPT.decode(r.next());
        int iterationCount = s0 | s1 << 6 | s2 << 12 | s3 << 18;
        s0 = Alphabet.Base64Alphabet.MOD_CRYPT.decode(r.next());
        s1 = Alphabet.Base64Alphabet.MOD_CRYPT.decode(r.next());
        s2 = Alphabet.Base64Alphabet.MOD_CRYPT.decode(r.next());
        s3 = Alphabet.Base64Alphabet.MOD_CRYPT.decode(r.next());
        int salt = s0 | s1 << 6 | s2 << 12 | s3 << 18;
        byte[] hash = r.base64Decode(Alphabet.Base64Alphabet.MOD_CRYPT, false).limitedTo(11).drain();
        return new BSDUnixDESCryptPasswordSpec(hash, salt, iterationCount);
    }

    private static int indexOf(char[] chars, char c) {
        for (int i = 0; i < chars.length; ++i) {
            if (chars[i] != c) continue;
            return i;
        }
        return -1;
    }

    private static int lastIndexOf(char[] chars, char c) {
        for (int i = chars.length - 1; i >= 0; --i) {
            if (chars[i] != c) continue;
            return i;
        }
        return -1;
    }

    public static byte[] generateRandomSalt(int saltSize) {
        byte[] randomSalt = new byte[saltSize];
        ThreadLocalRandom.current().nextBytes(randomSalt);
        return randomSalt;
    }

    static class IByteArrayInputStream
    extends ByteArrayInputStream {
        private final int[] interleave;

        IByteArrayInputStream(byte[] buf, int[] interleave) {
            super(buf);
            this.interleave = interleave;
        }

        IByteArrayInputStream(byte[] buf, int offset, int length, int[] interleave) {
            super(buf, offset, length);
            this.interleave = interleave;
        }

        @Override
        public synchronized int read() {
            return this.pos < this.count ? this.buf[this.interleave[this.pos++]] & 0xFF : -1;
        }
    }
}

