/*
 * Decompiled with CFR 0.152.
 */
package org.wildfly.security.auth.provider.jdbc.mapper;

import java.nio.charset.StandardCharsets;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.spec.InvalidKeySpecException;
import java.sql.ResultSet;
import java.sql.SQLException;
import org.wildfly.common.Assert;
import org.wildfly.security._private.ElytronMessages;
import org.wildfly.security.auth.provider.jdbc.KeyMapper;
import org.wildfly.security.auth.server.CredentialSupport;
import org.wildfly.security.password.PasswordFactory;
import org.wildfly.security.password.PasswordUtil;
import org.wildfly.security.password.interfaces.BCryptPassword;
import org.wildfly.security.password.interfaces.BSDUnixDESCryptPassword;
import org.wildfly.security.password.interfaces.ClearPassword;
import org.wildfly.security.password.interfaces.DigestPassword;
import org.wildfly.security.password.interfaces.SaltedSimpleDigestPassword;
import org.wildfly.security.password.interfaces.ScramDigestPassword;
import org.wildfly.security.password.interfaces.SimpleDigestPassword;
import org.wildfly.security.password.interfaces.SunUnixMD5CryptPassword;
import org.wildfly.security.password.interfaces.UnixDESCryptPassword;
import org.wildfly.security.password.interfaces.UnixMD5CryptPassword;
import org.wildfly.security.password.spec.ClearPasswordSpec;
import org.wildfly.security.password.spec.SaltedSimpleDigestPasswordSpec;
import org.wildfly.security.password.spec.ScramDigestPasswordSpec;
import org.wildfly.security.password.spec.SimpleDigestPasswordSpec;
import org.wildfly.security.util.CodePointIterator;

public class PasswordKeyMapper
implements KeyMapper {
    private final int hash;
    private final String algorithm;
    private int salt = -1;
    private int iterationCount = -1;
    private final Class<?> passwordType;

    public PasswordKeyMapper(String algorithm, int hash) throws InvalidKeyException {
        Assert.checkNotNullParam((String)"algorithm", (Object)algorithm);
        Assert.checkMinimumParameter((String)"hash", (int)1, (int)hash);
        this.algorithm = algorithm;
        this.passwordType = this.toPasswordType(algorithm);
        this.hash = hash;
    }

    public PasswordKeyMapper(String algorithm, int hash, int salt) throws InvalidKeyException {
        this(algorithm, hash);
        Assert.checkMinimumParameter((String)"salt", (int)1, (int)salt);
        this.salt = salt;
    }

    public PasswordKeyMapper(String algorithm, int hash, int salt, int iterationCount) throws InvalidKeyException {
        this(algorithm, hash, salt);
        Assert.checkMinimumParameter((String)"iterationCount", (int)1, (int)iterationCount);
        this.iterationCount = iterationCount;
    }

    @Override
    public Class<?> getKeyType() {
        return this.passwordType;
    }

    @Override
    public CredentialSupport getCredentialSupport(ResultSet resultSet) {
        Object map = this.map(resultSet);
        if (map != null) {
            return CredentialSupport.FULLY_SUPPORTED;
        }
        return CredentialSupport.UNSUPPORTED;
    }

    public String getAlgorithm() {
        return this.algorithm;
    }

    public int getHash() {
        return this.hash;
    }

    public int getSalt() {
        return this.salt;
    }

    public int getIterationCount() {
        return this.iterationCount;
    }

    @Override
    public Object map(ResultSet resultSet) {
        byte[] hash = null;
        byte[] salt = null;
        int iterationCount = 0;
        try {
            if (resultSet.next()) {
                hash = this.toByteArray(resultSet.getObject(this.getHash()));
                if (this.getSalt() > 0) {
                    salt = this.toByteArray(resultSet.getObject(this.getSalt()));
                }
                if (this.getIterationCount() > 0) {
                    iterationCount = resultSet.getInt(this.getIterationCount());
                }
            }
        }
        catch (SQLException e) {
            throw ElytronMessages.log.couldNotObtainCredentialWithCause(e);
        }
        if (hash != null) {
            PasswordFactory passwordFactory = this.getPasswordFactory(this.getAlgorithm());
            try {
                Class<?> credentialType = this.getKeyType();
                if (ClearPassword.class.equals(credentialType)) {
                    return this.toClearPassword(hash, passwordFactory);
                }
                if (BCryptPassword.class.equals(credentialType)) {
                    return PasswordUtil.parseCryptString(this.toCharArray(hash));
                }
                if (SaltedSimpleDigestPassword.class.equals(credentialType)) {
                    return this.toSaltedSimpleDigestPassword(hash, salt, passwordFactory);
                }
                if (SimpleDigestPassword.class.equals(credentialType)) {
                    return this.toSimpleDigestPassword(hash, passwordFactory);
                }
                if (ScramDigestPassword.class.equals(credentialType)) {
                    return this.toScramDigestPassword(hash, salt, iterationCount, passwordFactory);
                }
            }
            catch (InvalidKeySpecException e) {
                throw ElytronMessages.log.invalidPasswordKeySpecificationForAlgorithm(this.algorithm, e);
            }
        }
        return null;
    }

    private Object toScramDigestPassword(byte[] hash, byte[] salt, int iterationCount, PasswordFactory passwordFactory) throws InvalidKeySpecException {
        if (salt == null) {
            throw ElytronMessages.log.saltIsExpectedWhenCreatingPasswords(ScramDigestPassword.class.getSimpleName());
        }
        ScramDigestPasswordSpec saltedSimpleDigestPasswordSpec = new ScramDigestPasswordSpec(hash, salt, iterationCount);
        return passwordFactory.generatePassword(saltedSimpleDigestPasswordSpec);
    }

    private Object toSimpleDigestPassword(byte[] hash, PasswordFactory passwordFactory) throws InvalidKeySpecException {
        SimpleDigestPasswordSpec simpleDigestPasswordSpec = new SimpleDigestPasswordSpec(hash);
        return passwordFactory.generatePassword(simpleDigestPasswordSpec);
    }

    private Object toSaltedSimpleDigestPassword(byte[] hash, byte[] salt, PasswordFactory passwordFactory) throws InvalidKeySpecException {
        if (salt == null) {
            throw ElytronMessages.log.saltIsExpectedWhenCreatingPasswords(SaltedSimpleDigestPassword.class.getSimpleName());
        }
        SaltedSimpleDigestPasswordSpec saltedSimpleDigestPasswordSpec = new SaltedSimpleDigestPasswordSpec(hash, salt);
        return passwordFactory.generatePassword(saltedSimpleDigestPasswordSpec);
    }

    private Object toClearPassword(byte[] hash, PasswordFactory passwordFactory) throws InvalidKeySpecException {
        return passwordFactory.generatePassword(new ClearPasswordSpec(this.toCharArray(hash)));
    }

    private PasswordFactory getPasswordFactory(String algorithm) {
        try {
            return PasswordFactory.getInstance(algorithm);
        }
        catch (NoSuchAlgorithmException e) {
            throw ElytronMessages.log.couldNotObtainPasswordFactoryForAlgorithm(algorithm, e);
        }
    }

    private byte[] toByteArray(Object value) {
        if (String.class.isInstance(value)) {
            return value.toString().getBytes(StandardCharsets.UTF_8);
        }
        if (byte[].class.isInstance(value)) {
            return (byte[])value;
        }
        return new byte[0];
    }

    private char[] toCharArray(byte[] hash) {
        return CodePointIterator.ofUtf8Bytes(hash).drainToString().toCharArray();
    }

    private Class<?> toPasswordType(String algorithm) throws InvalidKeyException {
        switch (algorithm) {
            case "clear": {
                return ClearPassword.class;
            }
            case "bcrypt": {
                return BCryptPassword.class;
            }
            case "crypt-md5": {
                return UnixMD5CryptPassword.class;
            }
            case "sun-crypt-md5": 
            case "sun-crypt-md5-bare-salt": {
                return SunUnixMD5CryptPassword.class;
            }
            case "crypt-sha-256": 
            case "crypt-sha-512": {
                return ClearPassword.class;
            }
            case "digest-md5": 
            case "digest-sha": 
            case "digest-sha-256": 
            case "digest-sha-512": {
                return DigestPassword.class;
            }
            case "simple-digest-md2": 
            case "simple-digest-md5": 
            case "simple-digest-sha-1": 
            case "simple-digest-sha-256": 
            case "simple-digest-sha-384": 
            case "simple-digest-sha-512": {
                return SimpleDigestPassword.class;
            }
            case "password-salt-digest-md5": 
            case "password-salt-digest-sha-1": 
            case "password-salt-digest-sha-256": 
            case "password-salt-digest-sha-384": 
            case "password-salt-digest-sha-512": 
            case "salt-password-digest-md5": 
            case "salt-password-digest-sha-1": 
            case "salt-password-digest-sha-256": 
            case "salt-password-digest-sha-384": 
            case "salt-password-digest-sha-512": {
                return SaltedSimpleDigestPassword.class;
            }
            case "crypt-des": {
                return UnixDESCryptPassword.class;
            }
            case "bsd-crypt-des": {
                return BSDUnixDESCryptPassword.class;
            }
            case "scram-sha-1": 
            case "scram-sha-256": {
                return ScramDigestPassword.class;
            }
        }
        throw ElytronMessages.log.unknownPasswordTypeOrAlgorithm(algorithm);
    }
}

