/*
 * Decompiled with CFR 0.152.
 */
package org.infinispan.util.concurrent;

import java.util.ArrayDeque;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.AbstractExecutorService;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import org.infinispan.util.TimeService;
import org.infinispan.util.concurrent.BlockingRunnable;
import org.infinispan.util.concurrent.BlockingTaskAwareExecutorService;
import org.infinispan.util.logging.Log;
import org.infinispan.util.logging.LogFactory;

public class BlockingTaskAwareExecutorServiceImpl
extends AbstractExecutorService
implements BlockingTaskAwareExecutorService {
    private static final Log log = LogFactory.getLog(BlockingTaskAwareExecutorServiceImpl.class);
    private final Queue<BlockingRunnable> blockedTasks = new ConcurrentLinkedQueue<BlockingRunnable>();
    private final ExecutorService executorService;
    private final TimeService timeService;
    private final ControllerThread controllerThread;
    private volatile boolean shutdown;

    public BlockingTaskAwareExecutorServiceImpl(String controllerThreadName, ExecutorService executorService, TimeService timeService) {
        this.executorService = executorService;
        this.timeService = timeService;
        this.shutdown = false;
        this.controllerThread = new ControllerThread(controllerThreadName);
        this.controllerThread.start();
    }

    @Override
    public final void execute(BlockingRunnable runnable) {
        if (this.shutdown) {
            throw new RejectedExecutionException("Executor Service is already shutdown");
        }
        if (runnable.isReady()) {
            this.doExecute(runnable);
            if (log.isTraceEnabled()) {
                log.tracef("Added a new task directly: %s task(s) are waiting", this.blockedTasks.size());
            }
        } else {
            this.blockedTasks.offer(runnable);
            this.controllerThread.checkForReadyTask();
            if (log.isTraceEnabled()) {
                log.tracef("Added a new task to the queue: %s task(s) are waiting", this.blockedTasks.size());
            }
        }
    }

    @Override
    public void shutdown() {
        this.shutdown = true;
    }

    @Override
    public List<Runnable> shutdownNow() {
        this.shutdown = true;
        this.controllerThread.interrupt();
        LinkedList<Runnable> runnableList = new LinkedList<Runnable>();
        runnableList.addAll(this.executorService.shutdownNow());
        runnableList.addAll(this.blockedTasks);
        return runnableList;
    }

    @Override
    public boolean isShutdown() {
        return this.shutdown;
    }

    @Override
    public boolean isTerminated() {
        return this.shutdown && this.blockedTasks.isEmpty() && this.executorService.isTerminated();
    }

    @Override
    public boolean awaitTermination(long timeout, TimeUnit unit) throws InterruptedException {
        long endTime = this.timeService.expectedEndTime(timeout, unit);
        long waitTime = this.timeService.remainingTime(endTime, TimeUnit.MILLISECONDS);
        while (!this.blockedTasks.isEmpty() && waitTime > 0L) {
            Thread.sleep(waitTime);
            waitTime = this.timeService.remainingTime(endTime, TimeUnit.MILLISECONDS);
        }
        return this.isTerminated();
    }

    @Override
    public final void checkForReadyTasks() {
        this.controllerThread.checkForReadyTask();
    }

    @Override
    public void execute(Runnable command) {
        if (this.shutdown) {
            throw new RejectedExecutionException("Executor Service is already shutdown");
        }
        if (command instanceof BlockingRunnable) {
            this.execute((BlockingRunnable)command);
        } else {
            this.execute(new RunnableWrapper(command));
        }
    }

    private void doExecute(BlockingRunnable runnable) {
        try {
            this.executorService.execute(runnable);
        }
        catch (RejectedExecutionException rejected) {
            this.blockedTasks.offer(runnable);
        }
    }

    private static class RunnableWrapper
    implements BlockingRunnable {
        private final Runnable runnable;

        private RunnableWrapper(Runnable runnable) {
            this.runnable = runnable;
        }

        @Override
        public boolean isReady() {
            return true;
        }

        @Override
        public void run() {
            this.runnable.run();
        }
    }

    private class ControllerThread
    extends Thread {
        private final Semaphore semaphore;
        private volatile boolean interrupted;

        public ControllerThread(String controllerThreadName) {
            super(controllerThreadName);
            this.semaphore = new Semaphore(0);
        }

        public void checkForReadyTask() {
            this.semaphore.release();
        }

        @Override
        public void interrupt() {
            this.interrupted = true;
            super.interrupt();
            this.semaphore.release();
        }

        @Override
        public void run() {
            while (!this.interrupted) {
                BlockingRunnable runnable;
                try {
                    this.semaphore.acquire();
                }
                catch (InterruptedException e) {
                    return;
                }
                this.semaphore.drainPermits();
                int size = BlockingTaskAwareExecutorServiceImpl.this.blockedTasks.size();
                if (size == 0) continue;
                ArrayDeque<BlockingRunnable> readyList = new ArrayDeque<BlockingRunnable>(size);
                Iterator iterator = BlockingTaskAwareExecutorServiceImpl.this.blockedTasks.iterator();
                while (iterator.hasNext()) {
                    BlockingRunnable runnable2 = (BlockingRunnable)iterator.next();
                    if (!runnable2.isReady()) continue;
                    iterator.remove();
                    readyList.addLast(runnable2);
                }
                if (log.isTraceEnabled()) {
                    log.tracef("Tasks to be executed=%s, still pending=~%s", readyList.size(), size);
                }
                while ((runnable = (BlockingRunnable)readyList.pollFirst()) != null) {
                    BlockingTaskAwareExecutorServiceImpl.this.doExecute(runnable);
                }
            }
        }
    }
}

