/*
 * Decompiled with CFR 0.152.
 */
package org.apache.activemq.artemis.core.paging.impl;

import java.nio.file.FileStore;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.activemq.artemis.api.core.SimpleString;
import org.apache.activemq.artemis.core.paging.PageTransactionInfo;
import org.apache.activemq.artemis.core.paging.PagingManager;
import org.apache.activemq.artemis.core.paging.PagingStore;
import org.apache.activemq.artemis.core.paging.PagingStoreFactory;
import org.apache.activemq.artemis.core.server.ActiveMQScheduledComponent;
import org.apache.activemq.artemis.core.server.ActiveMQServerLogger;
import org.apache.activemq.artemis.core.server.files.FileStoreMonitor;
import org.apache.activemq.artemis.core.settings.HierarchicalRepository;
import org.apache.activemq.artemis.core.settings.impl.AddressSettings;
import org.apache.activemq.artemis.utils.collections.ConcurrentHashSet;
import org.apache.activemq.artemis.utils.runnables.AtomicRunnable;
import org.jboss.logging.Logger;

public final class PagingManagerImpl
implements PagingManager {
    private static final int ARTEMIS_DEBUG_PAGING_INTERVAL = Integer.valueOf(System.getProperty("artemis.debug.paging.interval", "0"));
    private static final Logger logger = Logger.getLogger(PagingManagerImpl.class);
    private volatile boolean started = false;
    private final ReentrantReadWriteLock syncLock = new ReentrantReadWriteLock();
    private final Set<PagingStore> blockedStored = new ConcurrentHashSet();
    private final ConcurrentMap<SimpleString, PagingStore> stores = new ConcurrentHashMap<SimpleString, PagingStore>();
    private final HierarchicalRepository<AddressSettings> addressSettingsRepository;
    private PagingStoreFactory pagingStoreFactory;
    private final AtomicLong globalSizeBytes = new AtomicLong(0L);
    private final AtomicLong numberOfMessages = new AtomicLong(0L);
    private final long maxSize;
    private volatile boolean cleanupEnabled = true;
    private volatile boolean diskFull = false;
    private final Executor memoryExecutor;
    private final Queue<Runnable> memoryCallback = new ConcurrentLinkedQueue<Runnable>();
    private final ConcurrentMap<Long, PageTransactionInfo> transactions = new ConcurrentHashMap<Long, PageTransactionInfo>();
    private ActiveMQScheduledComponent scheduledComponent = null;

    public void replacePageStoreFactory(PagingStoreFactory factory) {
        this.pagingStoreFactory = factory;
    }

    public PagingStoreFactory getPagingStoreFactory() {
        return this.pagingStoreFactory;
    }

    public PagingManagerImpl(PagingStoreFactory pagingSPI, HierarchicalRepository<AddressSettings> addressSettingsRepository, long maxSize) {
        this.pagingStoreFactory = pagingSPI;
        this.addressSettingsRepository = addressSettingsRepository;
        addressSettingsRepository.registerListener(this);
        this.maxSize = maxSize;
        this.memoryExecutor = pagingSPI.newExecutor();
    }

    public PagingManagerImpl(PagingStoreFactory pagingSPI, HierarchicalRepository<AddressSettings> addressSettingsRepository) {
        this(pagingSPI, addressSettingsRepository, -1L);
    }

    @Override
    public void addBlockedStore(PagingStore store) {
        this.blockedStored.add(store);
    }

    @Override
    public void onChange() {
        this.reapplySettings();
    }

    private void reapplySettings() {
        for (PagingStore store : this.stores.values()) {
            AddressSettings settings = this.addressSettingsRepository.getMatch(store.getAddress().toString());
            store.applySetting(settings);
        }
    }

    @Override
    public PagingManagerImpl addSize(int size) {
        if (size > 0) {
            this.numberOfMessages.incrementAndGet();
        } else {
            this.numberOfMessages.decrementAndGet();
        }
        long newSize = this.globalSizeBytes.addAndGet(size);
        if (newSize < 0L) {
            ActiveMQServerLogger.LOGGER.negativeGlobalAddressSize(newSize);
        }
        if (size < 0) {
            this.checkMemoryRelease();
        }
        return this;
    }

    @Override
    public long getGlobalSize() {
        return this.globalSizeBytes.get();
    }

    protected void checkMemoryRelease() {
        if (!(this.diskFull || this.maxSize >= 0L && this.globalSizeBytes.get() >= this.maxSize || this.blockedStored.isEmpty())) {
            if (!this.memoryCallback.isEmpty()) {
                if (this.memoryExecutor != null) {
                    this.memoryExecutor.execute(this::memoryReleased);
                } else {
                    this.memoryReleased();
                }
            }
            Iterator<PagingStore> storeIterator = this.blockedStored.iterator();
            while (storeIterator.hasNext()) {
                PagingStore store = storeIterator.next();
                if (!store.checkReleasedMemory()) continue;
                storeIterator.remove();
            }
        }
    }

    @Override
    public void injectMonitor(FileStoreMonitor monitor) throws Exception {
        this.pagingStoreFactory.injectMonitor(monitor);
        monitor.addCallback(new LocalMonitor());
    }

    @Override
    public boolean isDiskFull() {
        return this.diskFull;
    }

    @Override
    public boolean isUsingGlobalSize() {
        return this.maxSize > 0L;
    }

    @Override
    public void checkMemory(Runnable runWhenAvailable) {
        if (this.isGlobalFull()) {
            this.memoryCallback.add(AtomicRunnable.checkAtomic((Runnable)runWhenAvailable));
            return;
        }
        runWhenAvailable.run();
    }

    private void memoryReleased() {
        Runnable runnable;
        while ((runnable = this.memoryCallback.poll()) != null) {
            runnable.run();
        }
    }

    @Override
    public boolean isGlobalFull() {
        return this.diskFull || this.maxSize > 0L && this.globalSizeBytes.get() > this.maxSize;
    }

    @Override
    public void disableCleanup() {
        if (!this.cleanupEnabled) {
            return;
        }
        this.lock();
        try {
            this.cleanupEnabled = false;
            for (PagingStore store : this.stores.values()) {
                store.disableCleanup();
            }
        }
        finally {
            this.unlock();
        }
    }

    @Override
    public void resumeCleanup() {
        if (this.cleanupEnabled) {
            return;
        }
        this.lock();
        try {
            this.cleanupEnabled = true;
            for (PagingStore store : this.stores.values()) {
                store.enableCleanup();
            }
        }
        finally {
            this.unlock();
        }
    }

    @Override
    public SimpleString[] getStoreNames() {
        Set names = this.stores.keySet();
        return names.toArray(new SimpleString[names.size()]);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void reloadStores() throws Exception {
        this.lock();
        try {
            List<PagingStore> reloadedStores = this.pagingStoreFactory.reloadStores(this.addressSettingsRepository);
            for (PagingStore store : reloadedStores) {
                PagingStore oldStore = (PagingStore)this.stores.remove(store.getStoreName());
                if (oldStore != null) {
                    oldStore.stop();
                    oldStore = null;
                }
                store.start();
                this.stores.put(store.getStoreName(), store);
            }
        }
        finally {
            this.unlock();
        }
    }

    @Override
    public void deletePageStore(SimpleString storeName) throws Exception {
        this.syncLock.readLock().lock();
        try {
            PagingStore store = (PagingStore)this.stores.remove(storeName);
            if (store != null) {
                store.stop();
            }
        }
        finally {
            this.syncLock.readLock().unlock();
        }
    }

    @Override
    public PagingStore getPageStore(SimpleString storeName) throws Exception {
        PagingStore store = (PagingStore)this.stores.get(storeName);
        if (store != null) {
            return store;
        }
        return this.newStore(storeName);
    }

    @Override
    public void addTransaction(PageTransactionInfo pageTransaction) {
        if (logger.isTraceEnabled()) {
            logger.trace((Object)("Adding pageTransaction " + pageTransaction.getTransactionID()));
        }
        this.transactions.put(pageTransaction.getTransactionID(), pageTransaction);
    }

    @Override
    public void removeTransaction(long id) {
        if (logger.isTraceEnabled()) {
            logger.trace((Object)("Removing pageTransaction " + id));
        }
        this.transactions.remove(id);
    }

    @Override
    public PageTransactionInfo getTransaction(long id) {
        if (logger.isTraceEnabled()) {
            logger.trace((Object)("looking up pageTX = " + id));
        }
        return (PageTransactionInfo)this.transactions.get(id);
    }

    @Override
    public Map<Long, PageTransactionInfo> getTransactions() {
        return this.transactions;
    }

    public boolean isStarted() {
        return this.started;
    }

    public void start() throws Exception {
        this.lock();
        try {
            if (this.started) {
                return;
            }
            this.pagingStoreFactory.setPagingManager(this);
            this.reloadStores();
            if (ARTEMIS_DEBUG_PAGING_INTERVAL > 0) {
                this.scheduledComponent = new ActiveMQScheduledComponent(this.pagingStoreFactory.getScheduledExecutor(), this.pagingStoreFactory.newExecutor(), ARTEMIS_DEBUG_PAGING_INTERVAL, TimeUnit.SECONDS, false){

                    public void run() {
                        PagingManagerImpl.this.debug();
                    }
                };
                this.scheduledComponent.start();
            }
            this.started = true;
        }
        finally {
            this.unlock();
        }
    }

    public void debug() {
        logger.info((Object)("size = " + this.globalSizeBytes + " bytes, messages = " + this.numberOfMessages));
    }

    public synchronized void stop() throws Exception {
        if (!this.started) {
            return;
        }
        this.started = false;
        if (this.scheduledComponent != null) {
            this.scheduledComponent.stop();
            this.scheduledComponent = null;
        }
        this.lock();
        try {
            for (PagingStore store : this.stores.values()) {
                store.stop();
            }
            this.pagingStoreFactory.stop();
        }
        finally {
            this.unlock();
        }
    }

    @Override
    public void processReload() throws Exception {
        for (PagingStore store : this.stores.values()) {
            store.processReload();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private PagingStore newStore(SimpleString address) throws Exception {
        this.syncLock.readLock().lock();
        try {
            PagingStore store = (PagingStore)this.stores.get(address);
            if (store == null) {
                store = this.pagingStoreFactory.newStore(address, this.addressSettingsRepository.getMatch(address.toString()));
                store.start();
                if (!this.cleanupEnabled) {
                    store.disableCleanup();
                }
                this.stores.put(address, store);
            }
            PagingStore pagingStore = store;
            return pagingStore;
        }
        finally {
            this.syncLock.readLock().unlock();
        }
    }

    @Override
    public void unlock() {
        this.syncLock.writeLock().unlock();
    }

    @Override
    public void lock() {
        this.syncLock.writeLock().lock();
    }

    class LocalMonitor
    implements FileStoreMonitor.Callback {
        LocalMonitor() {
        }

        @Override
        public void tick(FileStore store, double usage) {
            logger.tracef("Tick from store:: %s, usage at %f", (Object)store, (Object)usage);
        }

        @Override
        public void over(FileStore store, double usage) {
            if (!PagingManagerImpl.this.diskFull) {
                ActiveMQServerLogger.LOGGER.diskBeyondCapacity();
                PagingManagerImpl.this.diskFull = true;
            }
        }

        @Override
        public void under(FileStore store, double usage) {
            if (PagingManagerImpl.this.diskFull || !PagingManagerImpl.this.blockedStored.isEmpty() || !PagingManagerImpl.this.memoryCallback.isEmpty()) {
                ActiveMQServerLogger.LOGGER.diskCapacityRestored();
                PagingManagerImpl.this.diskFull = false;
                PagingManagerImpl.this.checkMemoryRelease();
            }
        }
    }
}

