/*
 * Decompiled with CFR 0.152.
 */
package org.wildfly.security.password.util;

import java.nio.charset.StandardCharsets;
import java.security.spec.InvalidKeySpecException;
import java.util.NoSuchElementException;
import org.wildfly.common.Assert;
import org.wildfly.common.codec.Base64Alphabet;
import org.wildfly.common.iteration.ByteIterator;
import org.wildfly.common.iteration.CodePointIterator;
import org.wildfly.security._private.ElytronMessages;
import org.wildfly.security.password.Password;
import org.wildfly.security.password.interfaces.BCryptPassword;
import org.wildfly.security.password.interfaces.BSDUnixDESCryptPassword;
import org.wildfly.security.password.interfaces.SunUnixMD5CryptPassword;
import org.wildfly.security.password.interfaces.UnixDESCryptPassword;
import org.wildfly.security.password.interfaces.UnixMD5CryptPassword;
import org.wildfly.security.password.interfaces.UnixSHACryptPassword;

public final class ModularCrypt {
    private static final int A_CRYPT_MD5 = 1;
    private static final int A_BCRYPT = 2;
    private static final int A_BSD_NT_HASH = 3;
    private static final int A_CRYPT_SHA_256 = 4;
    private static final int A_CRYPT_SHA_512 = 5;
    private static final int A_SUN_CRYPT_MD5 = 6;
    private static final int A_APACHE_HTDIGEST = 7;
    private static final int A_BSD_CRYPT_DES = 8;
    private static final int A_CRYPT_DES = 9;
    private static final int A_SUN_CRYPT_MD5_BARE_SALT = 10;
    private static final int[] MD5_IDX = new int[]{12, 6, 0, 13, 7, 1, 14, 8, 2, 15, 9, 3, 5, 10, 4, 11};
    private static final int[] MD5_IDX_REV = ModularCrypt.inverse(MD5_IDX);
    private static final int[] SHA_256_IDX = new int[]{20, 10, 0, 11, 1, 21, 2, 22, 12, 23, 13, 3, 14, 4, 24, 5, 25, 15, 26, 16, 6, 17, 7, 27, 8, 28, 18, 29, 19, 9, 30, 31};
    private static final int[] SHA_256_IDX_REV = ModularCrypt.inverse(SHA_256_IDX);
    private static final int[] SHA_512_IDX = new int[]{42, 21, 0, 1, 43, 22, 23, 2, 44, 45, 24, 3, 4, 46, 25, 26, 5, 47, 48, 27, 6, 7, 49, 28, 29, 8, 50, 51, 30, 9, 10, 52, 31, 32, 11, 53, 54, 33, 12, 13, 55, 34, 35, 14, 56, 57, 36, 15, 16, 58, 37, 38, 17, 59, 60, 39, 18, 19, 61, 40, 41, 20, 62, 63};
    private static final int[] SHA_512_IDX_REV = ModularCrypt.inverse(SHA_512_IDX);
    public static final Base64Alphabet MOD_CRYPT = new ModCryptBase64Alphabet(false);
    public static final Base64Alphabet MOD_CRYPT_LE = new ModCryptBase64Alphabet(true);
    public static final Base64Alphabet BCRYPT = new Base64Alphabet(false){

        public int encode(int val) {
            if (val == 0) {
                return 46;
            }
            if (val == 1) {
                return 47;
            }
            if (val <= 27) {
                return 65 + val - 2;
            }
            if (val <= 53) {
                return 97 + val - 28;
            }
            assert (val < 64);
            return 48 + val - 54;
        }

        public int decode(int codePoint) {
            if (codePoint == 46) {
                return 0;
            }
            if (codePoint == 47) {
                return 1;
            }
            if (65 <= codePoint && codePoint <= 90) {
                return codePoint - 65 + 2;
            }
            if (97 <= codePoint && codePoint <= 122) {
                return codePoint - 97 + 28;
            }
            if (48 <= codePoint && codePoint <= 57) {
                return codePoint - 48 + 54;
            }
            return -1;
        }
    };

    private ModularCrypt() {
    }

    private static int doIdentifyAlgorithm(char[] chars) {
        if (chars.length < 5) {
            return 0;
        }
        if (chars[0] == '$') {
            if (chars[2] == '$') {
                switch (chars[1]) {
                    case '1': {
                        return 1;
                    }
                    case '2': {
                        return 2;
                    }
                    case '3': {
                        return 3;
                    }
                    case '5': {
                        return 4;
                    }
                    case '6': {
                        return 5;
                    }
                }
                return 0;
            }
            if (chars[3] == '$') {
                if (chars[1] == '2') {
                    if (chars[2] == 'a' || chars[2] == 'x' || chars[2] == 'y') {
                        return 2;
                    }
                    return 0;
                }
                return 0;
            }
            if (chars[4] == '$' || chars[4] == ',') {
                if (chars[1] == 'm' && chars[2] == 'd' && chars[3] == '5') {
                    int idx = ModularCrypt.lastIndexOf(chars, '$');
                    if (idx > 0) {
                        if (chars[idx - 1] == '$') {
                            return 6;
                        }
                        return 10;
                    }
                    return 0;
                }
                return 0;
            }
            if (chars[5] == '$') {
                if (chars[1] == 'a' && chars[2] == 'p' && chars[3] == 'r' && chars[4] == '1') {
                    return 7;
                }
                return 0;
            }
            return 0;
        }
        if (chars[0] == '_') {
            return 8;
        }
        if (chars.length == 13) {
            return 9;
        }
        return 0;
    }

    public static String identifyAlgorithm(char[] chars) {
        return ModularCrypt.getAlgorithmNameString(ModularCrypt.doIdentifyAlgorithm(chars));
    }

    static String getAlgorithmNameString(int id) {
        switch (id) {
            case 1: {
                return "crypt-md5";
            }
            case 2: {
                return "bcrypt";
            }
            case 3: {
                return "bsd-nt-hash";
            }
            case 4: {
                return "crypt-sha-256";
            }
            case 5: {
                return "crypt-sha-512";
            }
            case 6: {
                return "sun-crypt-md5";
            }
            case 7: {
                return "apache-htdigest";
            }
            case 8: {
                return "bsd-crypt-des";
            }
            case 9: {
                return "crypt-des";
            }
            case 10: {
                return "sun-crypt-md5-bare-salt";
            }
        }
        return null;
    }

    public static char[] encode(Password password) throws InvalidKeySpecException {
        StringBuilder b = ModularCrypt.getCryptStringToBuilder(password);
        char[] chars = new char[b.length()];
        b.getChars(0, b.length(), chars, 0);
        return chars;
    }

    public static String encodeAsString(Password password) throws InvalidKeySpecException {
        return ModularCrypt.getCryptStringToBuilder(password).toString();
    }

    private static StringBuilder getCryptStringToBuilder(Password password) throws InvalidKeySpecException {
        Assert.checkNotNullParam((String)"password", (Object)password);
        StringBuilder b = new StringBuilder();
        if (password instanceof BCryptPassword) {
            BCryptPassword spec = (BCryptPassword)password;
            b.append("$2a$");
            if (spec.getIterationCount() < 10) {
                b.append(0);
            }
            b.append(spec.getIterationCount());
            b.append("$");
            ByteIterator.ofBytes((byte[])spec.getSalt()).base64Encode(BCRYPT, false).drainTo(b);
            ByteIterator.ofBytes((byte[])spec.getHash()).base64Encode(BCRYPT, false).drainTo(b);
        } else if (password instanceof BSDUnixDESCryptPassword) {
            b.append('_');
            BSDUnixDESCryptPassword spec = (BSDUnixDESCryptPassword)password;
            int iterationCount = spec.getIterationCount();
            b.appendCodePoint(MOD_CRYPT.encode(iterationCount & 0x3F));
            b.appendCodePoint(MOD_CRYPT.encode(iterationCount >> 6 & 0x3F));
            b.appendCodePoint(MOD_CRYPT.encode(iterationCount >> 12 & 0x3F));
            b.appendCodePoint(MOD_CRYPT.encode(iterationCount >> 18 & 0x3F));
            int salt = spec.getSalt();
            b.appendCodePoint(MOD_CRYPT.encode(salt & 0x3F));
            b.appendCodePoint(MOD_CRYPT.encode(salt >> 6 & 0x3F));
            b.appendCodePoint(MOD_CRYPT.encode(salt >> 12 & 0x3F));
            b.appendCodePoint(MOD_CRYPT.encode(salt >> 18 & 0x3F));
            ByteIterator.ofBytes((byte[])spec.getHash()).base64Encode(MOD_CRYPT, false).drainTo(b);
        } else if (password instanceof UnixDESCryptPassword) {
            UnixDESCryptPassword spec = (UnixDESCryptPassword)password;
            short salt = spec.getSalt();
            b.appendCodePoint(MOD_CRYPT.encode(salt & 0x3F));
            b.appendCodePoint(MOD_CRYPT.encode(salt >> 6 & 0x3F));
            ByteIterator.ofBytes((byte[])spec.getHash()).base64Encode(MOD_CRYPT, false).drainTo(b);
        } else if (password instanceof UnixMD5CryptPassword) {
            byte[] salt;
            b.append("$1$");
            UnixMD5CryptPassword spec = (UnixMD5CryptPassword)password;
            for (byte sb : salt = spec.getSalt()) {
                b.append((char)(sb & 0xFF));
            }
            b.append('$');
            ByteIterator.ofBytes((byte[])spec.getHash(), (int[])MD5_IDX).base64Encode(MOD_CRYPT_LE, false).drainTo(b);
        } else if (password instanceof SunUnixMD5CryptPassword) {
            SunUnixMD5CryptPassword spec = (SunUnixMD5CryptPassword)password;
            int iterationCount = spec.getIterationCount();
            if (iterationCount > 0) {
                b.append("$md5,rounds=").append(iterationCount).append('$');
            } else {
                b.append("$md5$");
            }
            byte[] salt = spec.getSalt();
            for (byte sb : salt) {
                b.append((char)(sb & 0xFF));
            }
            switch (spec.getAlgorithm()) {
                case "sun-crypt-md5": {
                    b.append("$$");
                    break;
                }
                case "sun-crypt-md5-bare-salt": {
                    b.append("$");
                    break;
                }
                default: {
                    throw ElytronMessages.log.invalidKeySpecUnrecognizedKeySpecAlgorithm();
                }
            }
            ByteIterator.ofBytes((byte[])spec.getHash(), (int[])MD5_IDX).base64Encode(MOD_CRYPT_LE, false).drainTo(b);
        } else if (password instanceof UnixSHACryptPassword) {
            byte[] salt;
            int[] interleave;
            UnixSHACryptPassword spec = (UnixSHACryptPassword)password;
            switch (spec.getAlgorithm()) {
                case "crypt-sha-256": {
                    b.append("$5$");
                    interleave = SHA_256_IDX;
                    break;
                }
                case "crypt-sha-512": {
                    b.append("$6$");
                    interleave = SHA_512_IDX;
                    break;
                }
                default: {
                    throw ElytronMessages.log.invalidKeySpecUnrecognizedKeySpecAlgorithm();
                }
            }
            int iterationCount = spec.getIterationCount();
            if (iterationCount != 5000) {
                b.append("rounds=").append(iterationCount).append('$');
            }
            for (byte sb : salt = spec.getSalt()) {
                b.append((char)(sb & 0xFF));
            }
            b.append('$');
            ByteIterator.ofBytes((byte[])spec.getHash(), (int[])interleave).base64Encode(MOD_CRYPT_LE, false).drainTo(b);
        } else {
            throw ElytronMessages.log.invalidKeySpecPasswordSpecCannotBeRenderedAsString();
        }
        return b;
    }

    public static Password decode(String cryptString) throws InvalidKeySpecException {
        Assert.checkNotNullParam((String)"cryptString", (Object)cryptString);
        return ModularCrypt.decode(cryptString.toCharArray());
    }

    public static Password decode(char[] cryptString) throws InvalidKeySpecException {
        Assert.checkNotNullParam((String)"cryptString", (Object)cryptString);
        int algorithmId = ModularCrypt.doIdentifyAlgorithm(cryptString);
        switch (algorithmId) {
            case 1: {
                return ModularCrypt.parseUnixMD5CryptPasswordString(cryptString);
            }
            case 2: {
                return ModularCrypt.parseBCryptPasswordString(cryptString);
            }
            case 3: {
                throw new UnsupportedOperationException("not supported yet");
            }
            case 4: {
                return ModularCrypt.parseUnixSHA256CryptPasswordString(cryptString);
            }
            case 5: {
                return ModularCrypt.parseUnixSHA512CryptPasswordString(cryptString);
            }
            case 6: {
                return ModularCrypt.parseSunUnixMD5CryptPasswordString("sun-crypt-md5", cryptString);
            }
            case 10: {
                return ModularCrypt.parseSunUnixMD5CryptPasswordString("sun-crypt-md5-bare-salt", cryptString);
            }
            case 7: {
                throw new UnsupportedOperationException("not supported yet");
            }
            case 8: {
                return ModularCrypt.parseBSDUnixDESCryptPasswordString(cryptString);
            }
            case 9: {
                return ModularCrypt.parseUnixDESCryptPasswordString(cryptString);
            }
        }
        throw ElytronMessages.log.invalidKeySpecUnknownCryptStringAlgorithm();
    }

    private static int parseModCryptIterationCount(CodePointIterator reader, int minIterations, int maxIterations, int defaultIterations) throws InvalidKeySpecException {
        int iterationCount;
        CodePointIterator dr = reader.delimitedBy(new int[]{36});
        try {
            if (dr.limitedTo(7L).contentEquals(CodePointIterator.ofString((String)"rounds="))) {
                iterationCount = 0;
                while (dr.hasNext()) {
                    int ch = dr.next();
                    if (iterationCount != maxIterations) {
                        if (ch < 48 || ch > 57 || (iterationCount = (iterationCount << 3) + (iterationCount << 1) + ch - 48) <= maxIterations) continue;
                        iterationCount = maxIterations;
                        continue;
                    }
                    throw ElytronMessages.log.invalidKeySpecInvalidCharacterEncountered();
                }
                if (!reader.hasNext()) {
                    throw ElytronMessages.log.invalidKeySpecNoIterationCountTerminatorGiven();
                }
                reader.next();
            } else {
                iterationCount = defaultIterations;
            }
        }
        catch (NoSuchElementException ignored) {
            throw ElytronMessages.log.invalidKeySpecUnexpectedEndOfInputString();
        }
        return Math.max(minIterations, iterationCount);
    }

    private static int[] inverse(int[] orig) {
        int[] n = new int[orig.length];
        for (int i = 0; i < orig.length; ++i) {
            n[orig[i]] = i;
        }
        return n;
    }

    private static Password parseUnixSHA256CryptPasswordString(char[] cryptString) throws InvalidKeySpecException {
        assert (cryptString[0] == '$');
        assert (cryptString[1] == '5');
        assert (cryptString[2] == '$');
        return ModularCrypt.parseUnixSHACryptPassword(cryptString, SHA_256_IDX_REV, "crypt-sha-256");
    }

    private static Password parseUnixSHA512CryptPasswordString(char[] cryptString) throws InvalidKeySpecException {
        assert (cryptString[0] == '$');
        assert (cryptString[1] == '6');
        assert (cryptString[2] == '$');
        return ModularCrypt.parseUnixSHACryptPassword(cryptString, SHA_512_IDX_REV, "crypt-sha-512");
    }

    private static Password parseUnixSHACryptPassword(char[] cryptString, int[] table, String algorithm) throws InvalidKeySpecException {
        CodePointIterator r = CodePointIterator.ofChars((char[])cryptString, (int)3);
        try {
            int iterationCount = ModularCrypt.parseModCryptIterationCount(r, 1000, 999999999, 5000);
            byte[] salt = r.delimitedBy(new int[]{36}).drainToString().getBytes(StandardCharsets.ISO_8859_1);
            if (!r.hasNext()) {
                throw ElytronMessages.log.invalidKeySpecNoSaltTerminatorGiven();
            }
            r.next();
            byte[] decoded = r.base64Decode(MOD_CRYPT_LE, false).limitedTo(table.length).drain();
            if (decoded.length != table.length) {
                throw ElytronMessages.log.invalidHashLength();
            }
            byte[] hash = ByteIterator.ofBytes((byte[])decoded, (int[])table).drain();
            return UnixSHACryptPassword.createRaw(algorithm, salt, hash, iterationCount);
        }
        catch (NoSuchElementException e) {
            throw ElytronMessages.log.invalidKeySpecUnexpectedEndOfPasswordStringWithCause(e);
        }
    }

    private static Password parseUnixMD5CryptPasswordString(char[] cryptString) throws InvalidKeySpecException {
        assert (cryptString[0] == '$');
        assert (cryptString[1] == '1');
        assert (cryptString[2] == '$');
        CodePointIterator r = CodePointIterator.ofChars((char[])cryptString, (int)3);
        try {
            byte[] salt = r.delimitedBy(new int[]{36}).drainToString().getBytes(StandardCharsets.ISO_8859_1);
            if (!r.hasNext()) {
                throw ElytronMessages.log.invalidKeySpecNoSaltTerminatorGiven();
            }
            r.next();
            byte[] decoded = r.base64Decode(MOD_CRYPT_LE, false).limitedTo(MD5_IDX_REV.length).drain();
            if (decoded.length != MD5_IDX.length) {
                throw ElytronMessages.log.invalidHashLength();
            }
            byte[] hash = ByteIterator.ofBytes((byte[])decoded, (int[])MD5_IDX_REV).drain();
            return UnixMD5CryptPassword.createRaw("crypt-md5", salt, hash);
        }
        catch (NoSuchElementException e) {
            throw ElytronMessages.log.invalidKeySpecUnexpectedEndOfPasswordStringWithCause(e);
        }
    }

    private static Password parseSunUnixMD5CryptPasswordString(String algorithm, char[] cryptString) throws InvalidKeySpecException {
        assert (cryptString[0] == '$');
        assert (cryptString[1] == 'm');
        assert (cryptString[2] == 'd');
        assert (cryptString[3] == '5');
        assert (cryptString[4] == '$' || cryptString[4] == ',');
        CodePointIterator r = CodePointIterator.ofChars((char[])cryptString, (int)5);
        try {
            byte[] decoded;
            int iterationCount = cryptString[4] == ',' ? ModularCrypt.parseModCryptIterationCount(r, 0, 0x7FFFEFFF, 0) : 0;
            byte[] salt = r.delimitedBy(new int[]{36}).drainToString().getBytes(StandardCharsets.ISO_8859_1);
            if (!r.hasNext()) {
                throw ElytronMessages.log.invalidKeySpecNoSaltTerminatorGiven();
            }
            r.next();
            if (algorithm.equals("sun-crypt-md5") && r.hasNext() && r.peekNext() == 36) {
                r.next();
            }
            if ((decoded = r.base64Decode(MOD_CRYPT_LE, false).limitedTo(MD5_IDX_REV.length).drain()).length != MD5_IDX.length) {
                throw ElytronMessages.log.invalidHashLength();
            }
            byte[] hash = ByteIterator.ofBytes((byte[])decoded, (int[])MD5_IDX_REV).drain();
            return SunUnixMD5CryptPassword.createRaw(algorithm, salt, hash, iterationCount);
        }
        catch (NoSuchElementException e) {
            throw ElytronMessages.log.invalidKeySpecUnexpectedEndOfPasswordStringWithCause(e);
        }
    }

    private static Password parseBCryptPasswordString(char[] cryptString) throws InvalidKeySpecException {
        assert (cryptString[0] == '$');
        assert (cryptString[1] == '2');
        char minor = '\u0000';
        if (cryptString[2] != '$') {
            minor = cryptString[2];
            if (minor != 'a' && minor != 'x' && minor != 'y') {
                throw ElytronMessages.log.invalidKeySpecInvalidMinorVersion();
            }
            assert (cryptString[3] == '$');
        }
        CodePointIterator r = CodePointIterator.ofChars((char[])cryptString, (int)(minor == '\u0000' ? 3 : 4));
        try {
            int cost = Integer.parseInt(r.limitedTo(2L).drainToString());
            if (r.hasNext() && r.peekNext() != 36) {
                throw ElytronMessages.log.invalidKeySpecCostMustBeTwoDigitInteger();
            }
            if (!r.hasNext()) {
                throw ElytronMessages.log.invalidKeySpecUnexpectedEndOfPasswordString();
            }
            r.next();
            byte[] decodedSalt = r.limitedTo(22L).base64Decode(BCRYPT, false).drain();
            byte[] decodedPassword = r.limitedTo(31L).base64Decode(BCRYPT, false).drain();
            return BCryptPassword.createRaw("bcrypt", decodedPassword, decodedSalt, cost);
        }
        catch (NoSuchElementException e) {
            throw ElytronMessages.log.invalidKeySpecUnexpectedEndOfPasswordStringWithCause(e);
        }
    }

    private static Password parseUnixDESCryptPasswordString(char[] cryptString) throws InvalidKeySpecException {
        assert (cryptString.length == 13);
        CodePointIterator r = CodePointIterator.ofChars((char[])cryptString);
        int s0 = MOD_CRYPT.decode(r.next());
        int s1 = MOD_CRYPT.decode(r.next());
        short salt = (short)(s0 | s1 << 6);
        byte[] hash = r.base64Decode(MOD_CRYPT, false).limitedTo(8).drain();
        return UnixDESCryptPassword.createRaw("crypt-des", salt, hash);
    }

    private static Password parseBSDUnixDESCryptPasswordString(char[] cryptString) throws InvalidKeySpecException {
        assert (cryptString.length == 20);
        assert (cryptString[0] == '_');
        CodePointIterator r = CodePointIterator.ofChars((char[])cryptString, (int)1);
        int s0 = MOD_CRYPT.decode(r.next());
        int s1 = MOD_CRYPT.decode(r.next());
        int s2 = MOD_CRYPT.decode(r.next());
        int s3 = MOD_CRYPT.decode(r.next());
        int iterationCount = s0 | s1 << 6 | s2 << 12 | s3 << 18;
        s0 = MOD_CRYPT.decode(r.next());
        s1 = MOD_CRYPT.decode(r.next());
        s2 = MOD_CRYPT.decode(r.next());
        s3 = MOD_CRYPT.decode(r.next());
        int salt = s0 | s1 << 6 | s2 << 12 | s3 << 18;
        byte[] hash = r.base64Decode(MOD_CRYPT, false).limitedTo(11).drain();
        return BSDUnixDESCryptPassword.createRaw("bsd-crypt-des", hash, salt, iterationCount);
    }

    private static int lastIndexOf(char[] chars, char c) {
        for (int i = chars.length - 1; i >= 0; --i) {
            if (chars[i] != c) continue;
            return i;
        }
        return -1;
    }

    static class ModCryptBase64Alphabet
    extends Base64Alphabet {
        ModCryptBase64Alphabet(boolean littleEndian) {
            super(littleEndian);
        }

        public int encode(int val) {
            if (val == 0) {
                return 46;
            }
            if (val == 1) {
                return 47;
            }
            if (val <= 11) {
                return 48 + val - 2;
            }
            if (val <= 37) {
                return 65 + val - 12;
            }
            assert (val < 64);
            return 97 + val - 38;
        }

        public int decode(int codePoint) throws IllegalArgumentException {
            if (codePoint == 46) {
                return 0;
            }
            if (codePoint == 47) {
                return 1;
            }
            if (48 <= codePoint && codePoint <= 57) {
                return codePoint - 48 + 2;
            }
            if (65 <= codePoint && codePoint <= 90) {
                return codePoint - 65 + 12;
            }
            if (97 <= codePoint && codePoint <= 122) {
                return codePoint - 97 + 38;
            }
            return -1;
        }
    }
}

