/*
 * Decompiled with CFR 0.152.
 */
package io.smallrye.faulttolerance.metrics;

import com.netflix.hystrix.HystrixCircuitBreaker;
import com.netflix.hystrix.HystrixCommandMetrics;
import com.netflix.hystrix.HystrixEventType;
import com.netflix.hystrix.HystrixThreadPoolKey;
import com.netflix.hystrix.HystrixThreadPoolMetrics;
import com.netflix.hystrix.exception.HystrixRuntimeException;
import io.smallrye.faulttolerance.DefaultHystrixConcurrencyStrategy;
import io.smallrye.faulttolerance.HystrixCommandInterceptor;
import io.smallrye.faulttolerance.RetryContext;
import io.smallrye.faulttolerance.SimpleCommand;
import io.smallrye.faulttolerance.SynchronousCircuitBreaker;
import io.smallrye.faulttolerance.config.FaultToleranceOperation;
import io.smallrye.faulttolerance.metrics.MetricNames;
import io.smallrye.faulttolerance.metrics.MetricsCollector;
import java.util.function.Supplier;
import javax.enterprise.context.ApplicationScoped;
import javax.inject.Inject;
import org.eclipse.microprofile.config.inject.ConfigProperty;
import org.eclipse.microprofile.faulttolerance.exceptions.TimeoutException;
import org.eclipse.microprofile.metrics.Counter;
import org.eclipse.microprofile.metrics.Gauge;
import org.eclipse.microprofile.metrics.Histogram;
import org.eclipse.microprofile.metrics.Metadata;
import org.eclipse.microprofile.metrics.Metric;
import org.eclipse.microprofile.metrics.MetricID;
import org.eclipse.microprofile.metrics.MetricRegistry;
import org.eclipse.microprofile.metrics.MetricType;
import org.jboss.logging.Logger;

@ApplicationScoped
public class MetricsCollectorFactory {
    private static final Logger LOGGER = Logger.getLogger(DefaultHystrixConcurrencyStrategy.class);
    @Inject
    MetricRegistry registry;
    @Inject
    @ConfigProperty(name="MP_Fault_Tolerance_Metrics_Enabled", defaultValue="true")
    Boolean metricsEnabled;

    public MetricsCollector createCollector(FaultToleranceOperation operation, RetryContext retryContext, HystrixThreadPoolKey threadPoolKey) {
        if (this.metricsEnabled.booleanValue()) {
            return new MetricsCollectorImpl(operation, retryContext, threadPoolKey);
        }
        return MetricsCollector.NOOP;
    }

    public MetricRegistry getRegistry() {
        return this.registry;
    }

    public boolean isMetricsEnabled() {
        return this.metricsEnabled;
    }

    private void runSafely(Runnable runnable) {
        try {
            runnable.run();
        }
        catch (RuntimeException any) {
            LOGGER.warn((Object)"Collecting metrics failed", (Throwable)any);
        }
    }

    public static Metadata metadataOf(String name, MetricType metricType) {
        return Metadata.builder().withName(name).withType(metricType).reusable().build();
    }

    class MetricsCollectorImpl
    implements MetricsCollector {
        private final FaultToleranceOperation operation;
        private final String metricsPrefix;
        private final RetryContext retryContext;
        private final HystrixThreadPoolKey threadPoolKey;
        private boolean isCircuitBreakerOpenBeforeExceptionProcessing;
        private long start;

        MetricsCollectorImpl(FaultToleranceOperation operation, RetryContext retryContext, HystrixThreadPoolKey threadPoolKey) {
            this.operation = operation;
            this.retryContext = retryContext;
            this.threadPoolKey = threadPoolKey;
            this.metricsPrefix = MetricNames.metricsPrefix(operation.getMethod());
        }

        @Override
        public void init(SynchronousCircuitBreaker circuitBreaker) {
            MetricsCollectorFactory.this.runSafely(() -> {
                this.counterInc(this.metricsPrefix + ".invocations.total");
                this.start = 0L;
                if (circuitBreaker != null) {
                    this.gaugeRegister(this.metricsPrefix + ".circuitbreaker.open.total", circuitBreaker::getOpenTotal);
                    this.gaugeRegister(this.metricsPrefix + ".circuitbreaker.closed.total", circuitBreaker::getClosedTotal);
                    this.gaugeRegister(this.metricsPrefix + ".circuitbreaker.halfOpen.total", circuitBreaker::getHalfOpenTotal);
                }
            });
        }

        @Override
        public void beforeExecute(SimpleCommand command) {
            MetricsCollectorFactory.this.runSafely(() -> {
                this.start = System.nanoTime();
                if (this.retryContext != null && this.retryContext.hasBeenRetried()) {
                    this.counterInc(this.metricsPrefix + ".retry.retries.total");
                }
                if (this.operation.hasBulkhead()) {
                    if (this.operation.isAsync()) {
                        HystrixThreadPoolMetrics threadPoolMetrics = HystrixThreadPoolMetrics.getInstance((HystrixThreadPoolKey)this.threadPoolKey);
                        this.gaugeRegister(this.metricsPrefix + ".bulkhead.waitingQueue.population", () -> threadPoolMetrics.getCurrentQueueSize().longValue());
                    }
                    HystrixCommandMetrics hcm = command.getMetrics();
                    this.gaugeRegister(this.metricsPrefix + ".bulkhead.concurrentExecutions", () -> hcm.getCurrentConcurrentExecutionCount());
                }
            });
        }

        @Override
        public void afterSuccess(SimpleCommand command) {
            MetricsCollectorFactory.this.runSafely(() -> {
                if (this.retryContext != null) {
                    if (command.isResponseFromFallback()) {
                        this.counterInc(this.metricsPrefix + ".retry.callsFailed.total");
                    } else if (this.retryContext.hasBeenRetried()) {
                        this.counterInc(this.metricsPrefix + ".retry.callsSucceededRetried.total");
                    } else {
                        this.counterInc(this.metricsPrefix + ".retry.callsSucceededNotRetried.total");
                    }
                }
                if (this.operation.hasTimeout()) {
                    this.counterInc(this.metricsPrefix + ".timeout.callsNotTimedOut.total");
                }
                if (this.operation.hasCircuitBreaker()) {
                    this.counterInc(this.metricsPrefix + ".circuitbreaker.callsSucceeded.total");
                }
                if (this.operation.hasBulkhead()) {
                    this.counterInc(this.metricsPrefix + ".bulkhead.callsAccepted.total");
                    if (this.start != 0L) {
                        this.histogramUpdate(this.metricsPrefix + ".bulkhead.executionDuration", System.nanoTime() - this.start);
                    }
                }
            });
        }

        @Override
        public void onError(SimpleCommand command, HystrixRuntimeException e) {
            MetricsCollectorFactory.this.runSafely(() -> {
                if (this.operation.hasBulkhead() && (HystrixRuntimeException.FailureType.REJECTED_THREAD_EXECUTION == e.getFailureType() || HystrixRuntimeException.FailureType.REJECTED_SEMAPHORE_EXECUTION == e.getFailureType())) {
                    this.counterInc(this.metricsPrefix + ".bulkhead.callsRejected.total");
                }
                if (this.operation.hasCircuitBreaker()) {
                    if (e.getFailureType() == HystrixRuntimeException.FailureType.SHORTCIRCUIT) {
                        this.counterInc(this.metricsPrefix + ".circuitbreaker.callsPrevented.total");
                    } else if (this.circuitBreakerFailsOn(e, command)) {
                        this.counterInc(this.metricsPrefix + ".circuitbreaker.callsFailed.total");
                    } else {
                        this.counterInc(this.metricsPrefix + ".circuitbreaker.callsSucceeded.total");
                    }
                    this.isCircuitBreakerOpenBeforeExceptionProcessing = command.getCircuitBreaker().isOpen();
                }
                if (this.retryContext != null) {
                    if (this.retryContext.isLastAttempt() && !this.operation.hasFallback()) {
                        this.counterInc(this.metricsPrefix + ".retry.callsFailed.total");
                    }
                } else {
                    this.counterInc(this.metricsPrefix + ".invocations.failed.total");
                }
            });
        }

        private boolean circuitBreakerFailsOn(HystrixRuntimeException e, SimpleCommand command) {
            HystrixCircuitBreaker circuitBreaker = command.getCircuitBreaker();
            if (circuitBreaker instanceof SynchronousCircuitBreaker) {
                Exception cause = HystrixCommandInterceptor.getCause(e);
                SynchronousCircuitBreaker cb = (SynchronousCircuitBreaker)circuitBreaker;
                return cb.failsOn(cause);
            }
            return true;
        }

        @Override
        public void onProcessedError(SimpleCommand command, Exception exception) {
            MetricsCollectorFactory.this.runSafely(() -> {
                HystrixCircuitBreaker cb = command.getCircuitBreaker();
                if (cb != null && cb.isOpen() && !this.isCircuitBreakerOpenBeforeExceptionProcessing) {
                    this.counterInc(this.metricsPrefix + ".circuitbreaker.opened.total");
                }
                if (exception != null && TimeoutException.class.equals(exception.getClass())) {
                    this.counterInc(this.metricsPrefix + ".timeout.callsTimedOut.total");
                }
            });
        }

        @Override
        public void afterExecute(SimpleCommand command) {
            MetricsCollectorFactory.this.runSafely(() -> {
                if (this.start != 0L && this.operation.hasTimeout()) {
                    this.histogramUpdate(this.metricsPrefix + ".timeout.executionDuration", System.nanoTime() - this.start);
                }
                if (command.getEventCounts().contains(HystrixEventType.FALLBACK_SUCCESS) || command.getEventCounts().contains(HystrixEventType.FALLBACK_FAILURE)) {
                    this.counterInc(this.metricsPrefix + ".fallback.calls.total");
                }
                if (this.retryContext != null && command.isFailedExecution() && !command.isResponseFromFallback()) {
                    this.counterInc(this.metricsPrefix + ".invocations.failed.total");
                }
            });
        }

        private void counterInc(String name) {
            this.counterOf(name).inc();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void gaugeRegister(String name, Supplier<Long> supplier) {
            MetricID metricID = new MetricID(name);
            Gauge gauge = (Gauge)MetricsCollectorFactory.this.registry.getGauges().get(metricID);
            if (gauge == null) {
                FaultToleranceOperation faultToleranceOperation = this.operation;
                synchronized (faultToleranceOperation) {
                    gauge = (Gauge)MetricsCollectorFactory.this.registry.getGauges().get(metricID);
                    if (gauge == null) {
                        MetricsCollectorFactory.this.registry.register(name, (Metric)((Gauge)supplier::get));
                    }
                }
            }
        }

        private void histogramUpdate(String name, long value) {
            this.histogramOf(name).update(value);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private Counter counterOf(String name) {
            MetricID metricID = new MetricID(name);
            Counter counter = (Counter)MetricsCollectorFactory.this.registry.getCounters().get(metricID);
            if (counter == null) {
                FaultToleranceOperation faultToleranceOperation = this.operation;
                synchronized (faultToleranceOperation) {
                    counter = (Counter)MetricsCollectorFactory.this.registry.getCounters().get(metricID);
                    if (counter == null) {
                        counter = MetricsCollectorFactory.this.registry.counter(MetricsCollectorFactory.metadataOf(name, MetricType.COUNTER));
                    }
                }
            }
            return counter;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private Histogram histogramOf(String name) {
            MetricID metricID = new MetricID(name);
            Histogram histogram = (Histogram)MetricsCollectorFactory.this.registry.getHistograms().get(metricID);
            if (histogram == null) {
                FaultToleranceOperation faultToleranceOperation = this.operation;
                synchronized (faultToleranceOperation) {
                    histogram = (Histogram)MetricsCollectorFactory.this.registry.getHistograms().get(metricID);
                    if (histogram == null) {
                        histogram = MetricsCollectorFactory.this.registry.histogram(MetricsCollectorFactory.metadataOf(name, MetricType.HISTOGRAM));
                    }
                }
            }
            return histogram;
        }
    }
}

