/*
 * Decompiled with CFR 0.152.
 */
package io.smallrye.faulttolerance.core.circuit.breaker;

import io.smallrye.faulttolerance.core.FaultToleranceStrategy;
import io.smallrye.faulttolerance.core.InvocationContext;
import io.smallrye.faulttolerance.core.circuit.breaker.CircuitBreakerEvents;
import io.smallrye.faulttolerance.core.circuit.breaker.CircuitBreakerListener;
import io.smallrye.faulttolerance.core.circuit.breaker.RollingWindow;
import io.smallrye.faulttolerance.core.stopwatch.RunningStopwatch;
import io.smallrye.faulttolerance.core.stopwatch.Stopwatch;
import io.smallrye.faulttolerance.core.util.Preconditions;
import io.smallrye.faulttolerance.core.util.SetOfThrowables;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;
import org.eclipse.microprofile.faulttolerance.exceptions.CircuitBreakerOpenException;

public class CircuitBreaker<V>
implements FaultToleranceStrategy<V> {
    static final int STATE_CLOSED = 0;
    static final int STATE_OPEN = 1;
    static final int STATE_HALF_OPEN = 2;
    final FaultToleranceStrategy<V> delegate;
    final String description;
    final SetOfThrowables failOn;
    final SetOfThrowables skipOn;
    final long delayInMillis;
    final int rollingWindowSize;
    final int failureThreshold;
    final int successThreshold;
    final Stopwatch stopwatch;
    final List<CircuitBreakerListener> listeners = new CopyOnWriteArrayList<CircuitBreakerListener>();
    final AtomicReference<State> state;
    final MetricsRecorder metricsRecorder;
    final AtomicLong previousHalfOpenTime = new AtomicLong();
    volatile long halfOpenStart;
    final AtomicLong previousClosedTime = new AtomicLong();
    volatile long closedStart;
    final AtomicLong previousOpenTime = new AtomicLong();
    volatile long openStart;

    public CircuitBreaker(FaultToleranceStrategy<V> delegate, String description, SetOfThrowables failOn, SetOfThrowables skipOn, long delayInMillis, int requestVolumeThreshold, double failureRatio, int successThreshold, Stopwatch stopwatch, MetricsRecorder metricsRecorder) {
        this.delegate = Preconditions.checkNotNull(delegate, "Circuit breaker delegate must be set");
        this.description = Preconditions.checkNotNull(description, "Circuit breaker description must be set");
        this.failOn = Preconditions.checkNotNull(failOn, "Set of fail-on throwables must be set");
        this.skipOn = Preconditions.checkNotNull(skipOn, "Set of skip-on throwables must be set");
        this.delayInMillis = Preconditions.check(delayInMillis, delayInMillis >= 0L, "Circuit breaker delay must be >= 0");
        this.successThreshold = Preconditions.check(successThreshold, successThreshold > 0, "Circuit breaker success threshold must be > 0");
        this.stopwatch = Preconditions.checkNotNull(stopwatch, "Stopwatch must be set");
        this.failureThreshold = Preconditions.check((int)Math.ceil(failureRatio * (double)requestVolumeThreshold), failureRatio >= 0.0 && failureRatio <= 1.0, "Circuit breaker rolling window failure ratio must be >= 0 && <= 1");
        this.rollingWindowSize = Preconditions.check(requestVolumeThreshold, requestVolumeThreshold > 0, "Circuit breaker rolling window size must be > 0");
        this.state = new AtomicReference<State>(State.closed(this.rollingWindowSize, this.failureThreshold));
        this.metricsRecorder = metricsRecorder == null ? MetricsRecorder.NO_OP : metricsRecorder;
        this.closedStart = System.nanoTime();
        this.metricsRecorder.circuitBreakerClosedTimeProvider(() -> this.getTime(0, this.closedStart, this.previousClosedTime));
        this.metricsRecorder.circuitBreakerHalfOpenTimeProvider(() -> this.getTime(2, this.halfOpenStart, this.previousHalfOpenTime));
        this.metricsRecorder.circuitBreakerOpenTimeProvider(() -> this.getTime(1, this.openStart, this.previousOpenTime));
    }

    private Long getTime(int measuredState, long measuredStateStart, AtomicLong prevMeasuredStateTime) {
        return this.state.get().id == measuredState ? prevMeasuredStateTime.get() + System.nanoTime() - measuredStateStart : prevMeasuredStateTime.get();
    }

    @Override
    public V apply(InvocationContext<V> ctx) throws Exception {
        State state = this.state.get();
        switch (state.id) {
            case 0: {
                return this.inClosed(ctx, state);
            }
            case 1: {
                return this.inOpen(ctx, state);
            }
            case 2: {
                return this.inHalfOpen(ctx, state);
            }
        }
        throw new AssertionError((Object)("Invalid circuit breaker state: " + state.id));
    }

    boolean isConsideredSuccess(Throwable e) {
        return this.skipOn.includes(e.getClass()) || !this.failOn.includes(e.getClass());
    }

    private V inClosed(InvocationContext<V> ctx, State state) throws Exception {
        try {
            V result = this.delegate.apply(ctx);
            this.metricsRecorder.circuitBreakerSucceeded();
            boolean failureThresholdReached = state.rollingWindow.recordSuccess();
            if (failureThresholdReached) {
                this.toOpen(state, ctx);
            }
            this.listeners.forEach(CircuitBreakerListener::succeeded);
            return result;
        }
        catch (Throwable e) {
            boolean failureThresholdReached;
            boolean isFailure;
            boolean bl = isFailure = !this.isConsideredSuccess(e);
            if (isFailure) {
                this.listeners.forEach(CircuitBreakerListener::failed);
                this.metricsRecorder.circuitBreakerFailed();
            } else {
                this.listeners.forEach(CircuitBreakerListener::succeeded);
                this.metricsRecorder.circuitBreakerSucceeded();
            }
            boolean bl2 = failureThresholdReached = isFailure ? state.rollingWindow.recordFailure() : state.rollingWindow.recordSuccess();
            if (failureThresholdReached) {
                long now;
                this.openStart = now = System.nanoTime();
                this.previousClosedTime.addAndGet(now - this.closedStart);
                this.metricsRecorder.circuitBreakerClosedToOpen();
                this.toOpen(state, ctx);
            }
            throw e;
        }
    }

    private V inOpen(InvocationContext<V> ctx, State state) throws Exception {
        long now;
        if (state.runningStopwatch.elapsedTimeInMillis() < this.delayInMillis) {
            this.metricsRecorder.circuitBreakerRejected();
            this.listeners.forEach(CircuitBreakerListener::rejected);
            throw new CircuitBreakerOpenException(this.description + " circuit breaker is open");
        }
        this.halfOpenStart = now = System.nanoTime();
        this.previousOpenTime.addAndGet(now - this.openStart);
        this.toHalfOpen(state, ctx);
        return this.apply(ctx);
    }

    private V inHalfOpen(InvocationContext<V> ctx, State state) throws Exception {
        try {
            V result = this.delegate.apply(ctx);
            this.metricsRecorder.circuitBreakerSucceeded();
            int successes = state.consecutiveSuccesses.incrementAndGet();
            if (successes >= this.successThreshold) {
                long now;
                this.closedStart = now = System.nanoTime();
                this.previousHalfOpenTime.addAndGet(now - this.halfOpenStart);
                this.toClosed(state, ctx);
            }
            this.listeners.forEach(CircuitBreakerListener::succeeded);
            return result;
        }
        catch (Throwable e) {
            this.metricsRecorder.circuitBreakerFailed();
            this.listeners.forEach(CircuitBreakerListener::failed);
            this.toOpen(state, ctx);
            throw e;
        }
    }

    void toClosed(State state, InvocationContext<V> ctx) {
        State newState = State.closed(this.rollingWindowSize, this.failureThreshold);
        boolean moved = this.state.compareAndSet(state, newState);
        if (moved) {
            ctx.fireEvent(CircuitBreakerEvents.StateTransition.TO_CLOSED);
        }
    }

    void toOpen(State state, InvocationContext<V> ctx) {
        State newState = State.open(this.stopwatch);
        boolean moved = this.state.compareAndSet(state, newState);
        if (moved) {
            ctx.fireEvent(CircuitBreakerEvents.StateTransition.TO_OPEN);
        }
    }

    void toHalfOpen(State state, InvocationContext<V> ctx) {
        State newState = State.halfOpen();
        boolean moved = this.state.compareAndSet(state, newState);
        if (moved) {
            ctx.fireEvent(CircuitBreakerEvents.StateTransition.TO_HALF_OPEN);
        }
    }

    public void addListener(CircuitBreakerListener listener) {
        this.listeners.add(listener);
    }

    public static interface MetricsRecorder {
        public static final MetricsRecorder NO_OP = new MetricsRecorder(){

            @Override
            public void circuitBreakerRejected() {
            }

            @Override
            public void circuitBreakerOpenTimeProvider(Supplier<Long> supplier) {
            }

            @Override
            public void circuitBreakerHalfOpenTimeProvider(Supplier<Long> supplier) {
            }

            @Override
            public void circuitBreakerClosedTimeProvider(Supplier<Long> supplier) {
            }

            @Override
            public void circuitBreakerClosedToOpen() {
            }

            @Override
            public void circuitBreakerFailed() {
            }

            @Override
            public void circuitBreakerSucceeded() {
            }
        };

        public void circuitBreakerRejected();

        public void circuitBreakerOpenTimeProvider(Supplier<Long> var1);

        public void circuitBreakerHalfOpenTimeProvider(Supplier<Long> var1);

        public void circuitBreakerClosedTimeProvider(Supplier<Long> var1);

        public void circuitBreakerClosedToOpen();

        public void circuitBreakerFailed();

        public void circuitBreakerSucceeded();
    }

    static final class State {
        final int id;
        RollingWindow rollingWindow;
        RunningStopwatch runningStopwatch;
        AtomicInteger consecutiveSuccesses;

        private State(int id) {
            this.id = id;
        }

        static State closed(int rollingWindowSize, int failureThreshold) {
            State result = new State(0);
            result.rollingWindow = RollingWindow.create(rollingWindowSize, failureThreshold);
            return result;
        }

        static State open(Stopwatch stopwatch) {
            State result = new State(1);
            result.runningStopwatch = stopwatch.start();
            return result;
        }

        static State halfOpen() {
            State result = new State(2);
            result.consecutiveSuccesses = new AtomicInteger(0);
            return result;
        }
    }
}

