/*
 * Decompiled with CFR 0.152.
 */
package io.smallrye.faulttolerance.core.timer;

import io.smallrye.faulttolerance.core.timer.TimerLogger;
import io.smallrye.faulttolerance.core.timer.TimerTask;
import io.smallrye.faulttolerance.core.util.Preconditions;
import io.smallrye.faulttolerance.core.util.RunnableWrapper;
import java.util.Comparator;
import java.util.NoSuchElementException;
import java.util.SortedSet;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.LockSupport;

public final class Timer {
    private static final AtomicInteger COUNTER = new AtomicInteger(0);
    private static final Comparator<TimerTask> TIMER_TASK_COMPARATOR = (o1, o2) -> {
        if (o1 == o2) {
            return 0;
        }
        return o1.startTime <= o2.startTime ? -1 : 1;
    };
    private final String name;
    private final SortedSet<TimerTask> tasks;
    private final Thread thread;
    private final AtomicBoolean running = new AtomicBoolean(true);

    public Timer(Executor executor) {
        Preconditions.checkNotNull(executor, "Executor must be set");
        this.name = "SmallRye Fault Tolerance Timer " + COUNTER.incrementAndGet();
        TimerLogger.LOG.createdTimer(this.name);
        this.tasks = new ConcurrentSkipListSet<TimerTask>(TIMER_TASK_COMPARATOR);
        this.thread = new Thread(() -> {
            while (this.running.get()) {
                try {
                    TimerTask task;
                    if (this.tasks.isEmpty()) {
                        LockSupport.park();
                        continue;
                    }
                    try {
                        task = this.tasks.first();
                    }
                    catch (NoSuchElementException e) {
                        continue;
                    }
                    long currentTime = System.nanoTime();
                    long taskStartTime = task.startTime;
                    if (taskStartTime - currentTime <= 0L) {
                        this.tasks.remove(task);
                        if (!task.state.compareAndSet(0, 1)) continue;
                        Executor executorForTask = task.executorOverride;
                        if (executorForTask == null) {
                            executorForTask = executor;
                        }
                        executorForTask.execute(() -> {
                            TimerLogger.LOG.runningTimerTask(task);
                            try {
                                task.runnable.run();
                            }
                            finally {
                                task.state.set(2);
                            }
                        });
                        continue;
                    }
                    LockSupport.parkNanos(taskStartTime - currentTime);
                }
                catch (Exception e) {
                    TimerLogger.LOG.unexpectedExceptionInTimerLoop(e);
                }
            }
        }, this.name);
        this.thread.start();
    }

    public TimerTask schedule(long delayInMillis, Runnable task) {
        return this.schedule(delayInMillis, task, null);
    }

    public TimerTask schedule(long delayInMillis, Runnable task, Executor executor) {
        long startTime = System.nanoTime() + TimeUnit.MILLISECONDS.toNanos(delayInMillis);
        TimerTask timerTask = new TimerTask(startTime, RunnableWrapper.INSTANCE.wrap(task), this.tasks::remove, executor);
        this.tasks.add(timerTask);
        LockSupport.unpark(this.thread);
        TimerLogger.LOG.scheduledTimerTask(timerTask, delayInMillis);
        return timerTask;
    }

    public void shutdown() throws InterruptedException {
        if (this.running.compareAndSet(true, false)) {
            TimerLogger.LOG.shutdownTimer(this.name);
            this.thread.interrupt();
            this.thread.join();
        }
    }
}

