/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2022, Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.wildfly.clustering.ee;

import java.util.UUID;
import java.util.concurrent.ThreadLocalRandom;
import java.util.function.Supplier;

/**
 * UUID factory implementations.
 * @author Paul Ferraro
 */
public enum UUIDFactory implements Supplier<UUID> {
    /**
     * UUID factory that uses a {@link ThreadLocalRandom}.
     * UUIDs generated by this factory are <em>not</em> cryptographically secure.
     */
    INSECURE() {
        @Override
        public java.util.UUID get() {
            byte[] data = new byte[16];
            ThreadLocalRandom.current().nextBytes(data);
            data[6] &= 0x0f; /* clear version */
            data[6] |= 0x40; /* set to version 4 */
            data[8] &= 0x3f; /* clear variant */
            data[8] |= 0x80; /* set to IETF variant */
            long msb = 0;
            long lsb = 0;
            for (int i = 0; i < 8; i++) {
               msb = (msb << 8) | (data[i] & 0xff);
            }
            for (int i = 8; i < 16; i++) {
               lsb = (lsb << 8) | (data[i] & 0xff);
            }
            return new UUID(msb, lsb);
        }
    },
    /**
     * UUID factory that uses a {@link java.security.SecureRandom}.
     * UUIDs generated by this factory are cryptographically secure.
     */
    SECURE() {
        @Override
        public java.util.UUID get() {
            return UUID.randomUUID();
        }
    },
    ;
}
