/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2021, Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.wildfly.clustering.infinispan.marshalling;

import java.util.List;
import java.util.function.BiFunction;
import java.util.function.UnaryOperator;
import java.util.stream.Collectors;

import org.infinispan.commons.marshall.Marshaller;
import org.infinispan.commons.util.AggregatedClassLoader;
import org.infinispan.protostream.ProtobufUtil;
import org.infinispan.protostream.SerializationContext;
import org.infinispan.protostream.SerializationContextInitializer;
import org.jboss.marshalling.ModularClassResolver;
import org.jboss.modules.Module;
import org.jboss.modules.ModuleLoader;
import org.wildfly.clustering.infinispan.marshalling.jboss.JBossMarshaller;
import org.wildfly.clustering.infinispan.marshalling.protostream.ProtoStreamMarshaller;
import org.wildfly.clustering.marshalling.protostream.ModuleClassLoaderMarshaller;
import org.wildfly.clustering.marshalling.protostream.SerializationContextBuilder;

/**
 * @author Paul Ferraro
 */
public enum MarshallerFactory implements BiFunction<ModuleLoader, List<Module>, Marshaller> {

    DEFAULT() {
        @Override
        public Marshaller apply(ModuleLoader moduleLoader, List<Module> modules) {
            SerializationContext context = ProtobufUtil.newSerializationContext();
            for (Module module : modules) {
                for (SerializationContextInitializer initializer : module.loadService(SerializationContextInitializer.class)) {
                    initializer.registerSchema(context);
                    initializer.registerMarshallers(context);
                }
            }
            return new org.infinispan.commons.marshall.ProtoStreamMarshaller(context);
        }
    },
    JBOSS() {
        @Override
        public Marshaller apply(ModuleLoader moduleLoader, List<Module> modules) {
            ClassLoader classLoader = modules.size() > 1 ? new AggregatedClassLoader(modules.stream().map(Module::getClassLoader).collect(Collectors.toList())) : modules.get(0).getClassLoader();
            return new JBossMarshaller(ModularClassResolver.getInstance(moduleLoader), classLoader);
        }
    },
    PROTOSTREAM() {
        @Override
        public Marshaller apply(ModuleLoader moduleLoader, List<Module> modules) {
            return new ProtoStreamMarshaller(new ModuleClassLoaderMarshaller(moduleLoader), new UnaryOperator<SerializationContextBuilder>() {
                @Override
                public SerializationContextBuilder apply(SerializationContextBuilder builder) {
                    for (Module module : modules) {
                        builder.load(module.getClassLoader());
                    }
                    return builder;
                }
            });
        }
    },
    ;
}
