/*
 * Copyright The WildFly Authors
 * SPDX-License-Identifier: Apache-2.0
 */
package org.jboss.as.test.multinode.clientinterceptor.secured;

import static org.jboss.as.test.shared.PermissionUtils.createPermissionsXmlAsset;

import java.io.File;
import java.io.FilePermission;
import java.security.SecurityPermission;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.Callable;
import jakarta.ejb.EJBAccessException;
import javax.security.auth.AuthPermission;

import org.jboss.arquillian.container.test.api.Deployment;
import org.jboss.arquillian.container.test.api.OperateOnDeployment;
import org.jboss.arquillian.container.test.api.TargetsContainer;
import org.jboss.arquillian.junit.Arquillian;
import org.jboss.arquillian.junit.InSequence;
import org.jboss.as.arquillian.api.ServerSetup;
import org.jboss.as.test.integration.security.common.AbstractSecurityDomainSetup;
import org.jboss.as.test.shared.integration.ejb.security.Util;
import org.jboss.as.test.shared.integration.interceptor.clientside.AbstractClientInterceptorsSetupTask;
import org.jboss.as.test.shared.integration.interceptor.clientside.InterceptorModule;
import org.jboss.as.test.shared.util.ClientInterceptorUtil;
import org.jboss.shrinkwrap.api.Archive;
import org.jboss.shrinkwrap.api.ShrinkWrap;
import org.jboss.shrinkwrap.api.asset.StringAsset;
import org.jboss.shrinkwrap.api.spec.JavaArchive;
import org.jboss.shrinkwrap.api.spec.WebArchive;
import org.junit.Assert;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.wildfly.security.permission.ElytronPermission;

/**
 * A test case verifying client-side interceptor execution while accessing an EJB with @RolesAllowed, @DenyAll and @PermitAll security annotations.
 * See https://issues.jboss.org/browse/WFLY-6144 for more details.
 *
 * @author <a href="mailto:szhantem@redhat.com">Sultan Zhantemirov</a> (c) 2019 Red Hat, inc.
 */
@RunWith(Arquillian.class)
@ServerSetup({SecuredBeanClientInterceptorTestCase.SetupTask.class, EjbSecurityDomainSetup.class})
public class SecuredBeanClientInterceptorTestCase {

    private static final String ARCHIVE_NAME_CLIENT = "secured-bean-test-client";
    private static final String ARCHIVE_NAME_SERVER = "secured-bean-test-server";

    private static final String moduleName = "secured-bean-interceptor-module";

    @Deployment(name = "server")
    @TargetsContainer("multinode-server")
    public static Archive<?> deployment0() {
        JavaArchive jar = ShrinkWrap.create(JavaArchive.class, ARCHIVE_NAME_SERVER + ".jar");
        jar.addClasses(Secured.class, SecuredBean.class);
        return jar;
    }

    @Deployment(name = "client")
    @TargetsContainer("multinode-client")
    public static Archive<?> deployment1() {
        final Package currentPackage = SecuredBeanClientInterceptorTestCase.class.getPackage();

        WebArchive war = ShrinkWrap.create(WebArchive.class, ARCHIVE_NAME_CLIENT + ".war");
        war.addClasses(Util.class, ClientInterceptorUtil.class);
        war.addClasses(AbstractSecurityDomainSetup.class, EjbSecurityDomainSetup.class);
        war.addClasses(SecuredBeanClientInterceptorTestCase.class, SampleSecureInterceptor.class, Secured.class, SecuredBean.class);
        war.addPackage(AbstractClientInterceptorsSetupTask.class.getPackage());
        war.addAsManifestResource("META-INF/jboss-ejb-client-receivers.xml", "jboss-ejb-client.xml");
        war.addAsWebInfResource(currentPackage, "jboss-web.xml", "jboss-web.xml");
        war.addAsManifestResource(new StringAsset("Manifest-Version: 1.0\nDependencies: org.jboss.as.controller-client,org.jboss.dmr\n"), "MANIFEST.MF");
        war.addAsManifestResource(
                createPermissionsXmlAsset(
                        new SecurityPermission("putProviderProperty.WildFlyElytron"),
                        new ElytronPermission("getSecurityDomain"),
                        new ElytronPermission("authenticate"),
                        new RuntimePermission("getProtectionDomain"),
                        new AuthPermission("modifyPrincipals"),
                        new FilePermission(System.getProperty("jboss.home") + File.separatorChar + "standalone" +
                                           File.separatorChar + "tmp" + File.separatorChar + "auth" + File.separatorChar + "-", "read")),
                "permissions.xml");
        return war;
    }

    @Test
    @InSequence(1)
    @OperateOnDeployment("client")
    public void securedBeanAccessCheck() throws Exception {
        Secured bean = ClientInterceptorUtil.lookupStatelessRemote(ARCHIVE_NAME_SERVER, SecuredBean.class, Secured.class);
        Assert.assertNotNull(bean);

        String result = bean.permitAll("permitAll");
        Assert.assertNotNull(result);

        try {
            bean.denyAll("denyAll");
            Assert.fail("it was supposed to throw EJBAccessException");
        }
        catch (EJBAccessException ie){
            // expected
        }
    }

    @Test
    @InSequence(2)
    @OperateOnDeployment("client")
    public void securedBeanRoleCheck() throws Exception {
        final Callable<Void> callable = () -> {
            Secured bean = ClientInterceptorUtil.lookupStatelessRemote(ARCHIVE_NAME_SERVER, SecuredBean.class, Secured.class);

            try {
                bean.roleEcho("role1access");
            } catch (EJBAccessException ignored) {
            }

            try {
                bean.role2Echo("access");
            } catch (EJBAccessException e) {
                // expected
            }
            Assert.assertEquals(0, SampleSecureInterceptor.latch.getCount());
            return null;
        };
        Util.switchIdentity("user1", "password1", callable);
    }

    static class SetupTask extends AbstractClientInterceptorsSetupTask.SetupTask {
        @Override
        public List<InterceptorModule> getModules() {
            return Collections.singletonList(new InterceptorModule(
                    SampleSecureInterceptor.class,
                    moduleName,
                    "module.xml",
                    SecuredBeanClientInterceptorTestCase.class.getResource("module.xml"),
                    "client-side-interceptor-secured.jar"
            ));
        }
    }
}
