/*
 * Copyright The WildFly Authors
 * SPDX-License-Identifier: Apache-2.0
 */
package org.wildfly.test.integration.elytron.ejb;

import static org.junit.Assert.assertEquals;

import java.io.File;
import java.io.IOException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Properties;

import javax.naming.Context;
import javax.naming.InitialContext;


import org.jboss.arquillian.container.test.api.Deployment;
import org.jboss.arquillian.container.test.api.RunAsClient;
import org.jboss.arquillian.junit.Arquillian;
import org.jboss.arquillian.test.api.ArquillianResource;
import org.jboss.as.arquillian.api.ServerSetup;
import org.jboss.as.arquillian.container.ManagementClient;
import org.jboss.as.test.integration.security.common.Utils;
import org.jboss.as.test.categories.CommonCriteria;
import org.jboss.as.test.shared.TestSuiteEnvironment;
import org.jboss.ejb.client.RequestSendFailedException;
import org.jboss.shrinkwrap.api.ShrinkWrap;
import org.jboss.shrinkwrap.api.spec.JavaArchive;
import org.junit.Assert;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.junit.runner.RunWith;
import org.wildfly.test.security.common.elytron.EjbElytronDomainSetup;
import org.wildfly.test.security.common.elytron.ElytronDomainSetup;
import org.wildfly.test.security.common.elytron.ServletElytronDomainSetup;

/**
 * Test authentication with the use of a source address role decoder where the IP address of the remote
 * client matches the address configured on the decoder.
 *
 * @author <a href="mailto:fjuma@redhat.com">Farah Juma</a>
 */
@RunWith(Arquillian.class)
@RunAsClient
@ServerSetup({ AuthenticationWithSourceAddressRoleDecoderMatchTestCase.ElytronDomainSetupOverride.class, EjbElytronDomainSetup.class, ServletElytronDomainSetup.class })
@Category(CommonCriteria.class)
public class AuthenticationWithSourceAddressRoleDecoderMatchTestCase {

    private static final String APPLICATION_NAME = "authentication-with-source-address-role-decoder";

    @ArquillianResource
    private ManagementClient mgmtClient;

    @Deployment
    public static JavaArchive createDeployment() throws IOException {
        final JavaArchive jar = ShrinkWrap.create(JavaArchive.class, APPLICATION_NAME + ".jar");
        jar.addClasses(SecurityInformation.class, SecuredBean.class);
        return jar;
    }

    /*
     The EJB being used in this test class is secured using the "elytron-tests" security domain. This security
     domain is configured with:
      1) a source-address-role-decoder that assigns the "admin" role if the IP address of the remote client is TestSuiteEnvironment.getServerAddress()
      2) a permission-mapper that assigns the "LoginPermission" if the identity has the "admin" role unless the principal
         is "user2"
     */

    private static String getIPAddress() throws IllegalStateException {
        try {
            return InetAddress.getByName(TestSuiteEnvironment.getServerAddress()).getHostAddress();
        } catch (UnknownHostException e) {
            throw new IllegalStateException(e);
        }
    }

    @Test
    public void testAuthenticationIPAddressAndPermissionMapperMatch() throws Exception {
        Properties ejbClientConfiguration = createEjbClientConfiguration(Utils.getHost(mgmtClient), "user1", "password1");
        SecurityInformation targetBean = lookupEJB(SecuredBean.class, SecurityInformation.class, ejbClientConfiguration);
        assertEquals("user1", targetBean.getPrincipalName());

        ejbClientConfiguration = createEjbClientConfiguration(Utils.getHost(mgmtClient), "admin", "admin");
        targetBean = lookupEJB(SecuredBean.class, SecurityInformation.class, ejbClientConfiguration);
        assertEquals("admin", targetBean.getPrincipalName());
    }

    @Test
    public void testAuthenticationIPAddressMatchAndPermissionMapperMismatch() throws Exception {
        final Properties ejbClientConfiguration = createEjbClientConfiguration(Utils.getHost(mgmtClient), "user2", "password2");
        final SecurityInformation targetBean = lookupEJB(SecuredBean.class, SecurityInformation.class, ejbClientConfiguration);
        try {
            targetBean.getPrincipalName();
            Assert.fail("Expected RequestSendFailedException not thrown");
        } catch (RequestSendFailedException expected) {
        }
    }

    @Test
    public void testAuthenticationInvalidCredentials() throws Exception {
        Properties ejbClientConfiguration = createEjbClientConfiguration(Utils.getHost(mgmtClient), "user1", "badpassword");
        SecurityInformation targetBean = lookupEJB(SecuredBean.class, SecurityInformation.class, ejbClientConfiguration);
        try {
            targetBean.getPrincipalName();
            Assert.fail("Expected RequestSendFailedException not thrown");
        } catch (RequestSendFailedException expected) {
        }

        ejbClientConfiguration = createEjbClientConfiguration(Utils.getHost(mgmtClient), "user2", "badpassword");
        targetBean = lookupEJB(SecuredBean.class, SecurityInformation.class, ejbClientConfiguration);
        try {
            targetBean.getPrincipalName();
            Assert.fail("Expected RequestSendFailedException not thrown");
        } catch (RequestSendFailedException expected) {
        }
    }


    private static Properties createEjbClientConfiguration(String hostName, String user, String password) {
        final Properties pr = new Properties();
        pr.put("remote.connectionprovider.create.options.org.xnio.Options.SSL_ENABLED", "false");
        pr.put("remote.connection.default.connect.options.org.xnio.Options.SASL_DISALLOWED_MECHANISMS", "JBOSS-LOCAL-USER");
        pr.put("remote.connections", "default");
        pr.put("remote.connection.default.host", hostName);
        pr.put("remote.connection.default.port", "8080");
        pr.put("remote.connection.default.username", user);
        pr.put("remote.connection.default.password", password);
        return pr;
    }

    private static <T> T lookupEJB(Class<? extends T> beanImplClass, Class<T> remoteInterface, Properties ejbProperties) throws Exception {
        final Properties jndiProperties = new Properties();
        jndiProperties.putAll(ejbProperties);
        jndiProperties.put(Context.URL_PKG_PREFIXES, "org.jboss.ejb.client.naming");
        final Context context = new InitialContext(jndiProperties);

        return (T) context.lookup("ejb:/" + APPLICATION_NAME + "/" + beanImplClass.getSimpleName() + "!"
                + remoteInterface.getName());
    }

    static class ElytronDomainSetupOverride extends ElytronDomainSetup {
        public ElytronDomainSetupOverride() {
            super(new File(AuthenticationWithSourceAddressRoleDecoderMatchTestCase.class.getResource("users.properties").getFile()).getAbsolutePath(),
                    new File(AuthenticationWithSourceAddressRoleDecoderMatchTestCase.class.getResource("roles.properties").getFile()).getAbsolutePath(),
                    "elytron-tests",
                    "ipPermissionMapper",
                    getIPAddress());
        }
    }

}
