/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2011, Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.as.clustering.infinispan.subsystem;

import java.util.EnumSet;
import java.util.HashMap;
import java.util.Map;

import org.jboss.as.clustering.controller.Attribute;
import org.jboss.as.clustering.infinispan.subsystem.remote.ConnectionPoolResourceDefinition;
import org.jboss.as.clustering.infinispan.subsystem.remote.RemoteCacheContainerResourceDefinition;
import org.jboss.as.clustering.infinispan.subsystem.remote.RemoteClusterResourceDefinition;
import org.jboss.as.controller.PathElement;
import org.jboss.as.controller.descriptions.ModelDescriptionConstants;

/**
 * Enumerates the elements used in the Infinispan subsystem schema.
 *
 * @author Paul Ferraro
 * @author Richard Achmatowicz (c) 2011 RedHat Inc.
 * @author Tristan Tarrant
 */
public enum XMLElement {
    // must be first
    UNKNOWN((String) null),

    @Deprecated ASYNC_OPERATIONS_THREAD_POOL("async-operations-thread-pool"),
    BACKUP(BackupResourceDefinition.WILDCARD_PATH),
    @Deprecated BACKUP_FOR("backup-for"),
    BACKUPS(BackupsResourceDefinition.PATH),
    BINARY_KEYED_TABLE("binary-keyed-table"),
    BLOCKING_THREAD_POOL("blocking-thread-pool"),
    @Deprecated BUCKET_TABLE("bucket-table"),
    CACHE_CONTAINER(CacheContainerResourceDefinition.WILDCARD_PATH),
    DATA_COLUMN(TableResourceDefinition.ColumnAttribute.DATA),
    DISTRIBUTED_CACHE(DistributedCacheResourceDefinition.WILDCARD_PATH),
    ENTRIES("entries"),
    @Deprecated ENTRY_TABLE("entry-table"),
    @Deprecated EVICTION("eviction"),
    @Deprecated BINARY_MEMORY("binary-memory"),
    HEAP_MEMORY("heap-memory"),
    @Deprecated OBJECT_MEMORY("object-memory"),
    OFF_HEAP_MEMORY("off-heap-memory"),
    EXPIRATION(ExpirationResourceDefinition.PATH),
    EXPIRATION_THREAD_POOL("expiration-thread-pool"),
    FILE_STORE("file-store"),
    ID_COLUMN(TableResourceDefinition.ColumnAttribute.ID),
    INVALIDATION_CACHE(InvalidationCacheResourceDefinition.WILDCARD_PATH),
    LISTENER_THREAD_POOL("listener-thread-pool"),
    JDBC_STORE("jdbc-store"),
    STRING_KEYED_JDBC_STORE("string-keyed-jdbc-store"),
    BINARY_KEYED_JDBC_STORE("binary-keyed-jdbc-store"),
    MIXED_KEYED_JDBC_STORE("mixed-keyed-jdbc-store"),
    @Deprecated INDEXING("indexing"),
    LOCAL_CACHE(LocalCacheResourceDefinition.WILDCARD_PATH),
    LOCKING(LockingResourceDefinition.PATH),
    NON_BLOCKING_THREAD_POOL("non-blocking-thread-pool"),
    PARTITION_HANDLING(PartitionHandlingResourceDefinition.PATH),
    @Deprecated PERSISTENCE_THREAD_POOL("persistence-thread-pool"),
    @Deprecated REMOTE_COMMAND_THREAD_POOL("remote-command-thread-pool"),
    PROPERTY(ModelDescriptionConstants.PROPERTY),
    REMOTE_SERVER("remote-server"),
    REMOTE_STORE("remote-store"),
    REPLICATED_CACHE(ReplicatedCacheResourceDefinition.WILDCARD_PATH),
    @Deprecated SCATTERED_CACHE(ScatteredCacheResourceDefinition.WILDCARD_PATH),
    SEGMENT_COLUMN(TableResourceDefinition.ColumnAttribute.SEGMENT),
    SIZE("size"),
    STATE_TRANSFER(StateTransferResourceDefinition.PATH),
    @Deprecated STATE_TRANSFER_THREAD_POOL("state-transfer-thread-pool"),
    STORE(StoreResourceDefinition.WILDCARD_PATH),
    STRING_KEYED_TABLE("string-keyed-table"),
    TABLE(TableResourceDefinition.WILDCARD_PATH),
    TAKE_OFFLINE("take-offline"),
    TIMESTAMP_COLUMN(TableResourceDefinition.ColumnAttribute.TIMESTAMP),
    TRANSACTION(TransactionResourceDefinition.PATH),
    TRANSPORT(TransportResourceDefinition.WILDCARD_PATH),
    @Deprecated TRANSPORT_THREAD_POOL("transport-thread-pool"),
    WRITE_BEHIND("write-behind"),

    // remote-cache-container
    REMOTE_CACHE_CONTAINER(RemoteCacheContainerResourceDefinition.WILDCARD_PATH),
    ASYNC_THREAD_POOL("async-thread-pool"),
    CONNECTION_POOL(ConnectionPoolResourceDefinition.PATH),
    INVALIDATION_NEAR_CACHE("invalidation-near-cache"),
    REMOTE_CLUSTERS("remote-clusters"),
    REMOTE_CLUSTER(RemoteClusterResourceDefinition.WILDCARD_PATH),
    SECURITY("security"),
    HOTROD_STORE("hotrod-store"),
    ;

    private final String name;

    XMLElement(Attribute attribute) {
        this(attribute.getDefinition().getXmlName());
    }

    XMLElement(PathElement path) {
        this.name = path.isWildcard() ? path.getKey() : path.getValue();
    }

    XMLElement(String name) {
        this.name = name;
    }

    /**
     * Get the local name of this element.
     *
     * @return the local name
     */
    public String getLocalName() {
        return this.name;
    }

    private static final Map<String, XMLElement> elements;

    static {
        final Map<String, XMLElement> map = new HashMap<>();
        for (XMLElement element : EnumSet.allOf(XMLElement.class)) {
            final String name = element.getLocalName();
            if (name != null) {
                assert !map.containsKey(name) : element;
                map.put(name, element);
            }
        }
        elements = map;
    }

    public static XMLElement forName(String localName) {
        final XMLElement element = elements.get(localName);
        return element == null ? UNKNOWN : element;
    }
}
