/*
 * Decompiled with CFR 0.152.
 */
package org.apache.activemq.artemis.core.server.impl.quorum;

import java.util.Objects;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.apache.activemq.artemis.api.core.ActiveMQException;
import org.apache.activemq.artemis.core.server.NodeManager;
import org.apache.activemq.artemis.quorum.DistributedLock;
import org.apache.activemq.artemis.quorum.DistributedPrimitiveManager;
import org.apache.activemq.artemis.quorum.MutableLong;
import org.apache.activemq.artemis.quorum.UnavailableStateException;
import org.jboss.logging.Logger;

public final class ActivationSequenceStateMachine {
    private static final long CHECK_ACTIVATION_SEQUENCE_WAIT_MILLIS = 200L;
    private static final long LIVE_LOCK_ACQUIRE_TIMEOUT_MILLIS = 2000L;

    private ActivationSequenceStateMachine() {
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static DistributedLock tryActivate(NodeManager nodeManager, DistributedPrimitiveManager distributedManager, Logger logger2) throws InterruptedException, ExecutionException, TimeoutException, UnavailableStateException {
        Objects.requireNonNull(nodeManager);
        Objects.requireNonNull(distributedManager);
        Objects.requireNonNull(logger2);
        String nodeId = nodeManager.getNodeId() == null ? null : nodeManager.getNodeId().toString();
        Objects.requireNonNull(nodeId);
        long nodeActivationSequence = nodeManager.getNodeActivationSequence();
        if (nodeActivationSequence < 0L) {
            throw new IllegalArgumentException("nodeActivationSequence must be > 0");
        }
        DistributedLock activationLock = distributedManager.getDistributedLock(nodeId);
        try (MutableLong coordinatedNodeSequence = distributedManager.getMutableLong(nodeId);){
            while (true) {
                long timeout = 0L;
                switch (ActivationSequenceStateMachine.validateActivationSequence(coordinatedNodeSequence, activationLock, nodeId, nodeActivationSequence, logger2)) {
                    case Stale: {
                        activationLock.close();
                        DistributedLock distributedLock = null;
                        return distributedLock;
                    }
                    case InSync: {
                        timeout = 2000L;
                        break;
                    }
                }
                if (!activationLock.tryLock(timeout, TimeUnit.MILLISECONDS)) {
                    logger2.debugf("Candidate for Node ID = %s, with local activation sequence: %d, cannot acquire live lock within %dms; retrying", (Object)nodeId, (Object)nodeActivationSequence, (Object)timeout);
                    if (timeout != 0L) continue;
                    Thread.sleep(200L);
                    continue;
                }
                switch (ActivationSequenceStateMachine.validateActivationSequence(coordinatedNodeSequence, activationLock, nodeId, nodeActivationSequence, logger2)) {
                    case Stale: {
                        activationLock.close();
                        DistributedLock distributedLock = null;
                        return distributedLock;
                    }
                    case InSync: {
                        logger2.infof("Assuming live role for NodeID = %s, local activation sequence %d matches current coordinated activation sequence %d", (Object)nodeId, (Object)nodeActivationSequence, (Object)nodeActivationSequence);
                        DistributedLock distributedLock = activationLock;
                        return distributedLock;
                    }
                    case SelfRepair: {
                        logger2.infof("Assuming live role for NodeID = %s: local activation sequence %d matches claimed coordinated activation sequence %d. Repairing sequence", (Object)nodeId, (Object)nodeActivationSequence, (Object)nodeActivationSequence);
                        try {
                            ActivationSequenceStateMachine.repairActivationSequence(nodeManager, coordinatedNodeSequence, nodeActivationSequence, true);
                            DistributedLock distributedLock = activationLock;
                            return distributedLock;
                        }
                        catch (NodeManager.NodeManagerException | UnavailableStateException ex) {
                            activationLock.close();
                            throw ex;
                        }
                    }
                    case MaybeInSync: {
                        logger2.warnf("Assuming live role for NodeID = %s: repairing claimed activation sequence", (Object)nodeId);
                        try {
                            ActivationSequenceStateMachine.repairActivationSequence(nodeManager, coordinatedNodeSequence, nodeActivationSequence, false);
                            DistributedLock ex = activationLock;
                            return ex;
                        }
                        catch (NodeManager.NodeManagerException | UnavailableStateException ex) {
                            activationLock.close();
                            throw ex;
                        }
                    }
                }
            }
        }
    }

    private static void repairActivationSequence(NodeManager nodeManager, MutableLong coordinatedNodeSequence, long nodeActivationSequence, boolean selfHeal) throws UnavailableStateException {
        long coordinatedNodeActivationSequence;
        long l = coordinatedNodeActivationSequence = selfHeal ? nodeActivationSequence : nodeActivationSequence + 1L;
        if (!selfHeal) {
            nodeManager.writeNodeActivationSequence(coordinatedNodeActivationSequence);
        }
        coordinatedNodeSequence.set(coordinatedNodeActivationSequence);
    }

    private static ValidationResult validateActivationSequence(MutableLong coordinatedNodeSequence, DistributedLock activationLock, String lockAndLongId, long nodeActivationSequence, Logger logger2) throws UnavailableStateException {
        assert (coordinatedNodeSequence.getMutableLongId().equals(lockAndLongId));
        assert (activationLock.getLockId().equals(lockAndLongId));
        long currentCoordinatedNodeSequence = coordinatedNodeSequence.get();
        if (nodeActivationSequence == currentCoordinatedNodeSequence) {
            return ValidationResult.InSync;
        }
        if (currentCoordinatedNodeSequence > 0L) {
            logger2.infof("Not a candidate for NodeID = %s activation, local activation sequence %d does not match coordinated activation sequence %d", (Object)lockAndLongId, (Object)nodeActivationSequence, (Object)currentCoordinatedNodeSequence);
            return ValidationResult.Stale;
        }
        long claimedCoordinatedNodeSequence = -currentCoordinatedNodeSequence;
        long sequenceGap = claimedCoordinatedNodeSequence - nodeActivationSequence;
        if (sequenceGap == 0L) {
            return ValidationResult.SelfRepair;
        }
        if (sequenceGap == 1L) {
            return ValidationResult.MaybeInSync;
        }
        assert (sequenceGap > 1L);
        logger2.infof("Not a candidate for NodeID = %s activation, local activation sequence %d does not match coordinated activation sequence %d", (Object)lockAndLongId, (Object)nodeActivationSequence, (Object)claimedCoordinatedNodeSequence);
        return ValidationResult.Stale;
    }

    public static boolean awaitNextCommittedActivationSequence(DistributedPrimitiveManager distributedManager, String coordinatedLockAndNodeId, long activationSequence, long timeoutMills, Logger logger2) throws ExecutionException, InterruptedException, TimeoutException, UnavailableStateException {
        long elapsedNs;
        Objects.requireNonNull(distributedManager);
        Objects.requireNonNull(logger2);
        Objects.requireNonNull(coordinatedLockAndNodeId);
        if (activationSequence < 0L) {
            throw new IllegalArgumentException("activationSequence must be >= 0, while is " + activationSequence);
        }
        if (!distributedManager.isStarted()) {
            throw new IllegalStateException("manager must be started");
        }
        MutableLong coordinatedActivationSequence = distributedManager.getMutableLong(coordinatedLockAndNodeId);
        boolean anyNext = false;
        long timeoutNs = TimeUnit.MILLISECONDS.toNanos(timeoutMills);
        long started = System.nanoTime();
        do {
            long coordinatedValue;
            if ((coordinatedValue = coordinatedActivationSequence.get()) > activationSequence) {
                logger2.infof("Detected a new activation sequence with NodeID = %s: and sequence: %d", (Object)coordinatedLockAndNodeId, (Object)coordinatedValue);
                anyNext = true;
                break;
            }
            if (coordinatedValue < 0L) {
                long claimedSequence = -coordinatedValue;
                long activationsGap = claimedSequence - activationSequence;
                if (activationsGap > 1L) {
                    logger2.infof("Detected furthers sequential server activations from sequence %d, with NodeID = %s: and claimed sequence: %d", (Object)activationSequence, (Object)coordinatedLockAndNodeId, (Object)claimedSequence);
                    anyNext = true;
                    break;
                }
                logger2.debugf("Detected claiming of activation sequence = %d for NodeID = %s", claimedSequence, (Object)coordinatedLockAndNodeId);
            }
            try {
                TimeUnit.MILLISECONDS.sleep(200L);
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        } while ((elapsedNs = System.nanoTime() - started) < timeoutNs);
        return anyNext;
    }

    public static void ensureSequentialAccessToNodeData(String serverDescription, NodeManager nodeManager, DistributedPrimitiveManager distributedManager, Logger logger2) throws ActiveMQException, InterruptedException, UnavailableStateException, ExecutionException, TimeoutException {
        long nextActivationSequence;
        Objects.requireNonNull(serverDescription);
        Objects.requireNonNull(nodeManager);
        Objects.requireNonNull(distributedManager);
        Objects.requireNonNull(logger2);
        String lockAndLongId = nodeManager.getNodeId() == null ? null : nodeManager.getNodeId().toString();
        Objects.requireNonNull(lockAndLongId);
        long nodeActivationSequence = nodeManager.getNodeActivationSequence();
        if (nodeActivationSequence < 0L) {
            throw new IllegalArgumentException("nodeActivationSequence must be >= 0");
        }
        DistributedLock liveLock = distributedManager.getDistributedLock(lockAndLongId);
        if (!liveLock.isHeldByCaller()) {
            String message = String.format("Server [%s], live lock for NodeID = %s, not held, activation sequence cannot be safely changed", serverDescription, lockAndLongId);
            logger2.info((Object)message);
            throw new UnavailableStateException(message);
        }
        MutableLong coordinatedNodeActivationSequence = distributedManager.getMutableLong(lockAndLongId);
        if (!coordinatedNodeActivationSequence.compareAndSet(nodeActivationSequence, -(nextActivationSequence = nodeActivationSequence + 1L))) {
            String message = String.format("Server [%s], cannot assume live role for NodeID = %s, activation sequence claim failed, local activation sequence %d no longer matches current coordinated sequence %d", serverDescription, lockAndLongId, nodeActivationSequence, coordinatedNodeActivationSequence.get());
            logger2.infof(message, new Object[0]);
            throw new ActiveMQException(message);
        }
        try {
            nodeManager.writeNodeActivationSequence(nextActivationSequence);
        }
        catch (NodeManager.NodeManagerException fatal) {
            logger2.errorf("Server [%s] failed to set local activation sequence to: %d for NodeId =%s. Cannot continue committing coordinated activation sequence: REQUIRES ADMIN INTERVENTION", (Object)serverDescription, (Object)nextActivationSequence, (Object)lockAndLongId);
            throw new UnavailableStateException((Throwable)fatal);
        }
        logger2.infof("Server [%s], incremented local activation sequence to: %d for NodeId = %s", (Object)serverDescription, (Object)nextActivationSequence, (Object)lockAndLongId);
        if (!coordinatedNodeActivationSequence.compareAndSet(-nextActivationSequence, nextActivationSequence)) {
            String message = String.format("Server [%s], cannot assume live role for NodeID = %s, activation sequence commit failed, local activation sequence %d no longer matches current coordinated sequence %d", serverDescription, lockAndLongId, nodeActivationSequence, coordinatedNodeActivationSequence.get());
            logger2.infof(message, new Object[0]);
            throw new ActiveMQException(message);
        }
        logger2.infof("Server [%s], incremented coordinated activation sequence to: %d for NodeId = %s", (Object)serverDescription, (Object)nextActivationSequence, (Object)lockAndLongId);
    }

    private static enum ValidationResult {
        Stale,
        InSync,
        MaybeInSync,
        SelfRepair;

    }
}

