/*
 * Copyright The WildFly Authors
 * SPDX-License-Identifier: Apache-2.0
 */

package org.jboss.as.test.integration.jaxrs.client;

import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import jakarta.annotation.Resource;
import jakarta.enterprise.concurrent.ManagedExecutorService;
import jakarta.enterprise.concurrent.ManagedScheduledExecutorService;
import jakarta.enterprise.concurrent.ManagedThreadFactory;
import jakarta.ws.rs.GET;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.PathParam;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.WebApplicationException;
import jakarta.ws.rs.core.Context;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.UriInfo;

import org.jboss.as.test.shared.TimeoutUtil;

/**
 * @author <a href="mailto:jperkins@redhat.com">James R. Perkins</a>
 */
@Path("/context")
public class ClientThreadContextResource {

    @Resource
    private ManagedExecutorService executor;

    @Resource
    private ManagedScheduledExecutorService scheduledExecutor;

    @Resource
    private ManagedThreadFactory threadFactory;

    @GET
    @Path("/async")
    @Produces(MediaType.TEXT_PLAIN)
    public CompletionStage<String> async(@Context final UriInfo uriInfo) {
        final CompletableFuture<String> cs = new CompletableFuture<>();
        executor.execute(() -> {
            try {
                cs.complete(uriInfo.getPath());
            } catch (Exception e) {
                cs.completeExceptionally(e);
            }
        });
        return cs;
    }

    @GET
    @Path("/async/delayed")
    @Produces(MediaType.TEXT_PLAIN)
    @SuppressWarnings("MagicNumber")
    public CompletionStage<String> delayedAsync(@Context final UriInfo uriInfo) {
        final CompletableFuture<String> cs = new CompletableFuture<>();
        scheduledExecutor.schedule(() -> {
            try {
                cs.complete(uriInfo.getPath());
            } catch (Exception e) {
                cs.completeExceptionally(e);
            }
        }, 200, TimeUnit.MILLISECONDS);
        return cs;
    }

    @GET
    @Path("/async/thread-factory")
    @Produces(MediaType.TEXT_PLAIN)
    public CompletionStage<String> threadFactory(@Context final UriInfo uriInfo) {
        final ExecutorService executor = Executors.newSingleThreadExecutor(threadFactory);
        final CompletableFuture<String> cs = new CompletableFuture<>();
        executor.submit(() -> {
            try {
                cs.complete(uriInfo.getPath());
            } catch (Exception e) {
                cs.completeExceptionally(e);
            } finally {
                executor.shutdownNow();
            }
        });
        return cs;
    }

    @GET
    @Path("/async/scheduled/{count}")
    @Produces(MediaType.APPLICATION_JSON)
    @SuppressWarnings("MagicNumber")
    public Set<String> scheduled(@Context final UriInfo uriInfo, @PathParam("count") final int count) throws Exception {
        final Set<String> collected = new ConcurrentSkipListSet<>(String::compareTo);
        final ScheduledFuture<?> sf = scheduledExecutor.scheduleAtFixedRate(() -> {
            final long current = collected.size();
            if (current <= count) {
                collected.add(uriInfo.getPath() + "-" + current);
            }
        }, 0L, TimeoutUtil.adjust(600), TimeUnit.MILLISECONDS);
        try {
            // Wait no more than 5 seconds
            long timeout = TimeUnit.SECONDS.toMillis(TimeoutUtil.adjust(5));
            while (timeout > 0) {
                if (collected.size() == 3) {
                    break;
                }
                TimeUnit.MILLISECONDS.sleep(100L);
                timeout -= 100L;
            }
            if (timeout < 0) {
                throw new WebApplicationException("Scheduled tasks did not complete within 5 seconds");
            }
        } finally {
            sf.cancel(true);
        }
        return collected;
    }
}
