/*
 * Decompiled with CFR 0.152.
 */
package org.apache.activemq.artemis.tests.integration.stomp.v11;

import jakarta.jms.BytesMessage;
import jakarta.jms.Destination;
import jakarta.jms.Message;
import jakarta.jms.MessageConsumer;
import jakarta.jms.MessageProducer;
import jakarta.jms.TextMessage;
import java.io.IOException;
import java.lang.invoke.MethodHandles;
import java.net.URI;
import java.nio.channels.ClosedChannelException;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Collection;
import java.util.UUID;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import org.apache.activemq.artemis.api.core.RoutingType;
import org.apache.activemq.artemis.api.core.SimpleString;
import org.apache.activemq.artemis.core.protocol.stomp.StompConnection;
import org.apache.activemq.artemis.core.protocol.stomp.v11.StompFrameHandlerV11;
import org.apache.activemq.artemis.core.server.Queue;
import org.apache.activemq.artemis.spi.core.protocol.RemotingConnection;
import org.apache.activemq.artemis.tests.extensions.parameterized.ParameterizedTestExtension;
import org.apache.activemq.artemis.tests.extensions.parameterized.Parameters;
import org.apache.activemq.artemis.tests.integration.stomp.StompTestBase;
import org.apache.activemq.artemis.tests.integration.stomp.util.ClientStompFrame;
import org.apache.activemq.artemis.tests.integration.stomp.util.StompClientConnection;
import org.apache.activemq.artemis.tests.integration.stomp.util.StompClientConnectionFactory;
import org.apache.activemq.artemis.tests.integration.stomp.util.StompClientConnectionV11;
import org.apache.activemq.artemis.tests.util.Wait;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.TestTemplate;
import org.junit.jupiter.api.extension.ExtendWith;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ExtendWith(value={ParameterizedTestExtension.class})
public class StompV11Test
extends StompTestBase {
    private static final Logger logger = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    public static final String CLIENT_ID = "myclientid";
    private StompClientConnection conn;
    private URI v10Uri;

    @Parameters(name="{0}")
    public static Collection<Object[]> data() {
        return Arrays.asList({"ws+v11.stomp"}, {"tcp+v11.stomp"});
    }

    @Override
    @BeforeEach
    public void setUp() throws Exception {
        super.setUp();
        this.v10Uri = new URI(this.uri.toString().replace("v11", "v10"));
        this.conn = StompClientConnectionFactory.createClientConnection(this.uri);
    }

    @Override
    @AfterEach
    public void tearDown() throws Exception {
        try {
            boolean connected;
            boolean bl = connected = this.conn != null && this.conn.isConnected();
            if (connected) {
                this.conn.disconnect();
            }
        }
        finally {
            super.tearDown();
            this.conn.closeTransport();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @TestTemplate
    public void testConnection() throws Exception {
        this.server.getSecurityStore().setSecurityEnabled(true);
        StompClientConnection connection = StompClientConnectionFactory.createClientConnection(this.v10Uri);
        try {
            connection.connect(this.defUser, this.defPass);
            Assertions.assertTrue((boolean)connection.isConnected());
            Assertions.assertEquals((Object)"1.0", (Object)connection.getVersion());
            connection.disconnect();
            connection.closeTransport();
            connection = StompClientConnectionFactory.createClientConnection(this.uri);
            connection.connect(this.defUser, this.defPass);
            Assertions.assertTrue((boolean)connection.isConnected());
            Assertions.assertEquals((Object)"1.1", (Object)connection.getVersion());
            connection.disconnect();
            connection.closeTransport();
            connection = StompClientConnectionFactory.createClientConnection(this.uri);
            connection.connect();
            Assertions.assertFalse((boolean)connection.isConnected());
            StompClientConnectionV11 conn = (StompClientConnectionV11)StompClientConnectionFactory.createClientConnection(this.uri);
            conn.connect1(this.defUser, this.defPass);
            Assertions.assertTrue((boolean)conn.isConnected());
            conn.disconnect();
            conn.closeTransport();
            conn = (StompClientConnectionV11)StompClientConnectionFactory.createClientConnection(this.uri);
            ClientStompFrame frame = conn.connect("invaliduser", this.defPass);
            Assertions.assertFalse((boolean)conn.isConnected());
            Assertions.assertTrue((boolean)"ERROR".equals(frame.getCommand()));
            Assertions.assertTrue((boolean)frame.getBody().contains("Security Error occurred"));
            conn.closeTransport();
        }
        finally {
            connection.closeTransport();
        }
    }

    @TestTemplate
    public void testNegotiation() throws Exception {
        ClientStompFrame frame = this.conn.createFrame("CONNECT").addHeader("host", "127.0.0.1").addHeader("login", this.defUser).addHeader("passcode", this.defPass);
        ClientStompFrame reply = this.conn.sendFrame(frame);
        Assertions.assertEquals((Object)"CONNECTED", (Object)reply.getCommand());
        Assertions.assertNull((Object)reply.getHeader("version"));
        this.conn.disconnect();
        this.conn = StompClientConnectionFactory.createClientConnection(this.uri);
        frame = this.conn.createFrame("CONNECT").addHeader("accept-version", "1.0").addHeader("host", "127.0.0.1").addHeader("login", this.defUser).addHeader("passcode", this.defPass);
        reply = this.conn.sendFrame(frame);
        Assertions.assertEquals((Object)"CONNECTED", (Object)reply.getCommand());
        Assertions.assertEquals((Object)"1.0", (Object)reply.getHeader("version"));
        this.conn.disconnect();
        this.conn = StompClientConnectionFactory.createClientConnection(this.uri);
        frame = this.conn.createFrame("CONNECT").addHeader("accept-version", "1.1").addHeader("host", "127.0.0.1").addHeader("login", this.defUser).addHeader("passcode", this.defPass);
        reply = this.conn.sendFrame(frame);
        Assertions.assertEquals((Object)"CONNECTED", (Object)reply.getCommand());
        Assertions.assertEquals((Object)"1.1", (Object)reply.getHeader("version"));
        this.conn.disconnect();
        this.conn = StompClientConnectionFactory.createClientConnection(this.uri);
        frame = this.conn.createFrame("CONNECT").addHeader("accept-version", "1.0,1.1,1.3").addHeader("host", "127.0.0.1").addHeader("login", this.defUser).addHeader("passcode", this.defPass);
        reply = this.conn.sendFrame(frame);
        Assertions.assertEquals((Object)"CONNECTED", (Object)reply.getCommand());
        Assertions.assertEquals((Object)"1.1", (Object)reply.getHeader("version"));
        this.conn.disconnect();
        this.conn = StompClientConnectionFactory.createClientConnection(this.uri);
        frame = this.conn.createFrame("CONNECT").addHeader("accept-version", "1.3").addHeader("host", "127.0.0.1").addHeader("login", this.defUser).addHeader("passcode", this.defPass);
        reply = this.conn.sendFrame(frame);
        Assertions.assertEquals((Object)"ERROR", (Object)reply.getCommand());
        logger.debug("Got error frame {}", (Object)reply);
    }

    @TestTemplate
    public void testServerFrame() throws Exception {
        ClientStompFrame frame = this.conn.connect(this.defUser, this.defPass);
        this.conn.disconnect();
        Assertions.assertTrue((frame.getHeader("server") != null ? (byte)1 : 0) != 0);
        this.server.getRemotingService().destroyAcceptor("stomp");
        this.server.getRemotingService().createAcceptor("stomp", "tcp://localhost:61613?disableStompServerHeader=false").start();
        this.conn = StompClientConnectionFactory.createClientConnection(this.uri);
        frame = this.conn.connect(this.defUser, this.defPass);
        this.conn.disconnect();
        Assertions.assertTrue((frame.getHeader("server") != null ? (byte)1 : 0) != 0);
        this.server.getRemotingService().destroyAcceptor("stomp");
        this.server.getRemotingService().createAcceptor("stomp", "tcp://localhost:61613?disableStompServerHeader=true").start();
        this.conn = StompClientConnectionFactory.createClientConnection(this.uri);
        frame = this.conn.connect(this.defUser, this.defPass);
        this.conn.disconnect();
        Assertions.assertTrue((frame.getHeader("server") == null ? (byte)1 : 0) != 0);
    }

    @TestTemplate
    public void testSendAndReceive() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        ClientStompFrame response = StompV11Test.send(this.conn, StompV11Test.getQueuePrefix() + StompV11Test.getQueueName(), "text/plain", "Hello World 1!");
        Assertions.assertNull((Object)response);
        String uuid = UUID.randomUUID().toString();
        response = StompV11Test.send(this.conn, StompV11Test.getQueuePrefix() + StompV11Test.getQueueName(), "text/plain", "Hello World 2!", true);
        StompClientConnection newConn = StompClientConnectionFactory.createClientConnection(this.uri);
        newConn.connect(this.defUser, this.defPass);
        StompV11Test.subscribe(newConn, "a-sub");
        ClientStompFrame frame = newConn.receiveFrame();
        logger.debug("received {}", (Object)frame);
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assertions.assertEquals((Object)"a-sub", (Object)frame.getHeader("subscription"));
        Assertions.assertNotNull((Object)frame.getHeader("message-id"));
        Assertions.assertEquals((Object)(StompV11Test.getQueuePrefix() + StompV11Test.getQueueName()), (Object)frame.getHeader("destination"));
        Assertions.assertEquals((Object)"Hello World 1!", (Object)frame.getBody());
        frame = newConn.receiveFrame();
        logger.debug("received {}", (Object)frame);
        StompV11Test.unsubscribe(newConn, "a-sub");
        newConn.disconnect();
    }

    @TestTemplate
    public void testHeaderContentType() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.send(this.conn, StompV11Test.getQueuePrefix() + StompV11Test.getQueueName(), "application/xml", "Hello World 1!");
        StompClientConnection newConn = StompClientConnectionFactory.createClientConnection(this.uri);
        newConn.connect(this.defUser, this.defPass);
        StompV11Test.subscribe(newConn, "a-sub");
        ClientStompFrame frame = newConn.receiveFrame();
        logger.debug("received {}", (Object)frame);
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assertions.assertEquals((Object)"application/xml", (Object)frame.getHeader("content-type"));
        StompV11Test.unsubscribe(newConn, "a-sub");
        newConn.disconnect();
    }

    @TestTemplate
    public void testHeaderContentLength() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        String body = "Hello World 1!";
        String cLen = String.valueOf(body.getBytes(StandardCharsets.UTF_8).length);
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompV11Test.getQueuePrefix() + StompV11Test.getQueueName()).addHeader("content-type", "application/xml").addHeader("content-length", cLen).setBody(body + "extra");
        this.conn.sendFrame(frame);
        StompClientConnection newConn = StompClientConnectionFactory.createClientConnection(this.uri);
        newConn.connect(this.defUser, this.defPass);
        StompV11Test.subscribe(newConn, "a-sub");
        frame = newConn.receiveFrame();
        logger.debug("received {}", (Object)frame);
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assertions.assertEquals((Object)cLen, (Object)frame.getHeader("content-length"));
        StompV11Test.unsubscribe(newConn, "a-sub");
        newConn.disconnect();
    }

    @TestTemplate
    public void testHeaderEncoding() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        String body = "Hello World 1!";
        String cLen = String.valueOf(body.getBytes(StandardCharsets.UTF_8).length);
        String hKey = "special-header\\\\\\n\\c";
        String hVal = "\\c\\\\\\ngood";
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompV11Test.getQueuePrefix() + StompV11Test.getQueueName()).addHeader("content-type", "application/xml").addHeader("content-length", cLen).addHeader(hKey, hVal);
        logger.debug("key: |{}| val: |{}|", (Object)hKey, (Object)hVal);
        frame.setBody(body);
        this.conn.sendFrame(frame);
        StompClientConnection newConn = StompClientConnectionFactory.createClientConnection(this.uri);
        newConn.connect(this.defUser, this.defPass);
        StompV11Test.subscribe(newConn, "a-sub");
        frame = newConn.receiveFrame();
        logger.debug("received {}", (Object)frame);
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        String value = frame.getHeader("special-header\\\n:");
        Assertions.assertEquals((Object)":\\\ngood", (Object)value);
        StompV11Test.unsubscribe(newConn, "a-sub");
        newConn.disconnect();
    }

    @TestTemplate
    public void testHeaderUndefinedEscape() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        ClientStompFrame frame = this.conn.createFrame("SEND");
        String body = "Hello World 1!";
        String cLen = String.valueOf(body.getBytes(StandardCharsets.UTF_8).length);
        frame.addHeader("destination", StompV11Test.getQueuePrefix() + StompV11Test.getQueueName());
        frame.addHeader("content-type", "text/plain");
        frame.addHeader("content-length", cLen);
        String hKey = "undefined-escape";
        String hVal = "is\\ttab";
        frame.addHeader(hKey, hVal);
        logger.debug("key: |{}| val: |{}|", (Object)hKey, (Object)hVal);
        frame.setBody(body);
        this.conn.sendFrame(frame);
        ClientStompFrame error = this.conn.receiveFrame();
        logger.debug("received {}", (Object)error);
        String desc = "Should have received an ERROR for undefined escape sequence";
        Assertions.assertNotNull((Object)error, (String)desc);
        Assertions.assertEquals((Object)"ERROR", (Object)error.getCommand(), (String)desc);
    }

    @TestTemplate
    public void testHeartBeat() throws Exception {
        ClientStompFrame frame = this.conn.createFrame("CONNECT").addHeader("host", "127.0.0.1").addHeader("login", this.defUser).addHeader("passcode", this.defPass);
        ClientStompFrame reply = this.conn.sendFrame(frame);
        Assertions.assertEquals((Object)"CONNECTED", (Object)reply.getCommand());
        Thread.sleep(5000L);
        Assertions.assertEquals((int)0, (int)this.conn.getFrameQueueSize());
        this.conn.disconnect();
        this.conn = StompClientConnectionFactory.createClientConnection(this.uri);
        frame = this.conn.createFrame("CONNECT").addHeader("host", "127.0.0.1").addHeader("login", this.defUser).addHeader("passcode", this.defPass).addHeader("heart-beat", "0,0").addHeader("accept-version", "1.0,1.1");
        reply = this.conn.sendFrame(frame);
        Assertions.assertEquals((Object)"CONNECTED", (Object)reply.getCommand());
        Assertions.assertEquals((Object)"0,30000", (Object)reply.getHeader("heart-beat"));
        Thread.sleep(5000L);
        Assertions.assertEquals((int)0, (int)this.conn.getFrameQueueSize());
        this.conn.disconnect();
        this.conn = StompClientConnectionFactory.createClientConnection(this.uri);
        frame = this.conn.createFrame("CONNECT").addHeader("host", "127.0.0.1").addHeader("login", this.defUser).addHeader("passcode", this.defPass).addHeader("heart-beat", "1,0").addHeader("accept-version", "1.0,1.1");
        reply = this.conn.sendFrame(frame);
        Assertions.assertEquals((Object)"CONNECTED", (Object)reply.getCommand());
        Assertions.assertEquals((Object)"0,500", (Object)reply.getHeader("heart-beat"));
        Thread.sleep(2000L);
        try {
            StompV11Test.send(this.conn, StompV11Test.getQueuePrefix() + StompV11Test.getQueueName(), "text/plain", "Hello World");
            Assertions.fail((String)"connection should have been destroyed by now");
        }
        catch (IOException iOException) {
            // empty catch block
        }
        this.conn = StompClientConnectionFactory.createClientConnection(this.uri);
        frame = this.conn.createFrame("CONNECT").addHeader("host", "127.0.0.1").addHeader("login", this.defUser).addHeader("passcode", this.defPass).addHeader("heart-beat", "1,0").addHeader("accept-version", "1.0,1.1");
        reply = this.conn.sendFrame(frame);
        Assertions.assertEquals((Object)"CONNECTED", (Object)reply.getCommand());
        Assertions.assertEquals((Object)"0,500", (Object)reply.getHeader("heart-beat"));
        logger.debug("========== start pinger!");
        this.conn.startPinger(500L);
        Thread.sleep(2000L);
        StompV11Test.send(this.conn, StompV11Test.getQueuePrefix() + StompV11Test.getQueueName(), "text/plain", "Hello World");
        this.conn.stopPinger();
        this.conn.disconnect();
    }

    @TestTemplate
    public void testHeartBeat2() throws Exception {
        ClientStompFrame frame = this.conn.createFrame("CONNECT").addHeader("host", "127.0.0.1").addHeader("login", this.defUser).addHeader("passcode", this.defPass).addHeader("heart-beat", "1,1").addHeader("accept-version", "1.0,1.1");
        ClientStompFrame reply = this.conn.sendFrame(frame);
        Assertions.assertEquals((Object)"CONNECTED", (Object)reply.getCommand());
        Assertions.assertEquals((Object)"500,500", (Object)reply.getHeader("heart-beat"));
        this.conn.disconnect();
        this.conn = StompClientConnectionFactory.createClientConnection(this.uri);
        frame = this.conn.createFrame("CONNECT").addHeader("host", "127.0.0.1").addHeader("login", this.defUser).addHeader("passcode", this.defPass).addHeader("heart-beat", "500,1000").addHeader("accept-version", "1.0,1.1");
        reply = this.conn.sendFrame(frame);
        Assertions.assertEquals((Object)"CONNECTED", (Object)reply.getCommand());
        Assertions.assertEquals((Object)"1000,500", (Object)reply.getHeader("heart-beat"));
        logger.debug("========== start pinger!");
        this.conn.startPinger(500L);
        Thread.sleep(10000L);
        int size = this.conn.getServerPingNumber();
        logger.debug("ping received: {}", (Object)size);
        Assertions.assertTrue((size > 5 ? (byte)1 : 0) != 0);
        StompV11Test.send(this.conn, StompV11Test.getQueuePrefix() + StompV11Test.getQueueName(), "text/plain", "Hello World");
        this.conn.disconnect();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @TestTemplate
    public void testSendWithHeartBeatsAndReceive() throws Exception {
        StompClientConnection newConn = null;
        try {
            ClientStompFrame frame = this.conn.createFrame("CONNECT").addHeader("host", "127.0.0.1").addHeader("login", this.defUser).addHeader("passcode", this.defPass).addHeader("heart-beat", "500,1000").addHeader("accept-version", "1.0,1.1");
            this.conn.sendFrame(frame);
            this.conn.startPinger(500L);
            for (int i = 0; i < 10; ++i) {
                StompV11Test.send(this.conn, StompV11Test.getQueuePrefix() + StompV11Test.getQueueName(), "text/plain", "Hello World " + i + "!");
                Thread.sleep(500L);
            }
            newConn = StompClientConnectionFactory.createClientConnection(this.uri);
            newConn.connect(this.defUser, this.defPass);
            StompV11Test.subscribe(newConn, "a-sub");
            int cnt = 0;
            frame = newConn.receiveFrame();
            while (frame != null) {
                ++cnt;
                Thread.sleep(500L);
                frame = newConn.receiveFrame(5000L);
            }
            Assertions.assertEquals((int)10, (int)cnt);
            StompV11Test.unsubscribe(newConn, "a-sub");
        }
        finally {
            if (newConn != null) {
                newConn.disconnect();
            }
            this.conn.disconnect();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @TestTemplate
    public void testSendAndReceiveWithHeartBeats() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        for (int i = 0; i < 10; ++i) {
            StompV11Test.send(this.conn, StompV11Test.getQueuePrefix() + StompV11Test.getQueueName(), "text/plain", "Hello World " + i + "!");
            Thread.sleep(500L);
        }
        StompClientConnection newConn = StompClientConnectionFactory.createClientConnection(this.uri);
        try {
            ClientStompFrame frame = newConn.createFrame("CONNECT").addHeader("host", "127.0.0.1").addHeader("login", this.defUser).addHeader("passcode", this.defPass).addHeader("heart-beat", "500,1000").addHeader("accept-version", "1.0,1.1");
            newConn.sendFrame(frame);
            newConn.startPinger(500L);
            Thread.sleep(500L);
            StompV11Test.subscribe(newConn, "a-sub");
            int cnt = 0;
            frame = newConn.receiveFrame();
            while (frame != null) {
                ++cnt;
                Thread.sleep(500L);
                frame = newConn.receiveFrame(5000L);
            }
            Assertions.assertEquals((int)10, (int)cnt);
            StompV11Test.unsubscribe(newConn, "a-sub");
        }
        finally {
            newConn.disconnect();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @TestTemplate
    public void testSendWithHeartBeatsAndReceiveWithHeartBeats() throws Exception {
        StompClientConnection newConn = null;
        try {
            ClientStompFrame frame = this.conn.createFrame("CONNECT").addHeader("host", "127.0.0.1").addHeader("login", this.defUser).addHeader("passcode", this.defPass).addHeader("heart-beat", "500,1000").addHeader("accept-version", "1.0,1.1");
            this.conn.sendFrame(frame);
            this.conn.startPinger(500L);
            for (int i = 0; i < 10; ++i) {
                StompV11Test.send(this.conn, StompV11Test.getQueuePrefix() + StompV11Test.getQueueName(), "text/plain", "Hello World " + i + "!");
                Thread.sleep(500L);
            }
            newConn = StompClientConnectionFactory.createClientConnection(this.uri);
            frame = newConn.createFrame("CONNECT").addHeader("host", "127.0.0.1").addHeader("login", this.defUser).addHeader("passcode", this.defPass).addHeader("heart-beat", "500,1000").addHeader("accept-version", "1.0,1.1");
            newConn.sendFrame(frame);
            newConn.startPinger(500L);
            Thread.sleep(500L);
            StompV11Test.subscribe(newConn, "a-sub");
            int cnt = 0;
            frame = newConn.receiveFrame();
            while (frame != null) {
                ++cnt;
                Thread.sleep(500L);
                frame = newConn.receiveFrame(5000L);
            }
            Assertions.assertEquals((int)10, (int)cnt);
            StompV11Test.unsubscribe(newConn, "a-sub");
        }
        finally {
            if (newConn != null) {
                newConn.disconnect();
            }
            this.conn.disconnect();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @TestTemplate
    public void testHeartBeatToTTL() throws Exception {
        int port = 61614;
        this.uri = StompV11Test.createStompClientUri(this.scheme, this.hostname, port);
        this.server.getRemotingService().createAcceptor("test", "tcp://127.0.0.1:" + port + "?connectionTtl=1000&connectionTtlMin=5000&connectionTtlMax=10000").start();
        StompClientConnection connection = StompClientConnectionFactory.createClientConnection(this.uri);
        ClientStompFrame frame = connection.createFrame("CONNECT").addHeader("host", "127.0.0.1").addHeader("login", this.defUser).addHeader("passcode", this.defPass);
        ClientStompFrame reply = connection.sendFrame(frame);
        Assertions.assertEquals((Object)"CONNECTED", (Object)reply.getCommand());
        Thread.sleep(3000L);
        Assertions.assertEquals((Object)"ERROR", (Object)connection.receiveFrame(1000L).getCommand());
        Assertions.assertEquals((int)0, (int)connection.getFrameQueueSize());
        try {
            Assertions.assertFalse((boolean)connection.isConnected());
        }
        catch (Exception exception) {
        }
        finally {
            connection.closeTransport();
        }
        connection = StompClientConnectionFactory.createClientConnection(this.uri);
        frame = connection.createFrame("CONNECT").addHeader("host", "127.0.0.1").addHeader("login", this.defUser).addHeader("passcode", this.defPass).addHeader("heart-beat", "0,0").addHeader("accept-version", "1.0,1.1");
        reply = connection.sendFrame(frame);
        logger.debug("Reply: {}", (Object)reply);
        Assertions.assertEquals((Object)"CONNECTED", (Object)reply.getCommand());
        Assertions.assertEquals((Object)"0,500", (Object)reply.getHeader("heart-beat"));
        Thread.sleep(3000L);
        Assertions.assertEquals((Object)"ERROR", (Object)connection.receiveFrame(1000L).getCommand());
        Assertions.assertEquals((int)0, (int)connection.getFrameQueueSize());
        try {
            Assertions.assertFalse((boolean)connection.isConnected());
        }
        catch (Exception exception) {
        }
        finally {
            connection.closeTransport();
        }
        connection = StompClientConnectionFactory.createClientConnection(this.uri);
        frame = connection.createFrame("CONNECT").addHeader("host", "127.0.0.1").addHeader("login", this.defUser).addHeader("passcode", this.defPass).addHeader("heart-beat", "1,0").addHeader("accept-version", "1.0,1.1");
        reply = connection.sendFrame(frame);
        Assertions.assertEquals((Object)"CONNECTED", (Object)reply.getCommand());
        Assertions.assertEquals((Object)"0,2500", (Object)reply.getHeader("heart-beat"));
        Thread.sleep(7000L);
        try {
            Assertions.assertFalse((boolean)connection.isConnected());
        }
        catch (Exception exception) {
        }
        finally {
            connection.closeTransport();
        }
        connection = StompClientConnectionFactory.createClientConnection(this.uri);
        frame = connection.createFrame("CONNECT").addHeader("host", "127.0.0.1").addHeader("login", this.defUser).addHeader("passcode", this.defPass).addHeader("heart-beat", "1,0").addHeader("accept-version", "1.0,1.1");
        reply = connection.sendFrame(frame);
        Assertions.assertEquals((Object)"CONNECTED", (Object)reply.getCommand());
        Assertions.assertEquals((Object)"0,2500", (Object)reply.getHeader("heart-beat"));
        logger.debug("========== start pinger!");
        connection.startPinger(2500L);
        Thread.sleep(7000L);
        StompV11Test.send(connection, StompV11Test.getQueuePrefix() + StompV11Test.getQueueName(), "text/plain", "Hello World");
        connection.stopPinger();
        connection.disconnect();
        connection = StompClientConnectionFactory.createClientConnection(this.uri);
        frame = connection.createFrame("CONNECT").addHeader("host", "127.0.0.1").addHeader("login", this.defUser).addHeader("passcode", this.defPass).addHeader("heart-beat", "20000,0").addHeader("accept-version", "1.0,1.1");
        reply = connection.sendFrame(frame);
        Assertions.assertEquals((Object)"CONNECTED", (Object)reply.getCommand());
        Assertions.assertEquals((Object)"0,5000", (Object)reply.getHeader("heart-beat"));
        Thread.sleep(12000L);
        try {
            Assertions.assertFalse((boolean)connection.isConnected());
        }
        catch (Exception exception) {
        }
        finally {
            connection.closeTransport();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @TestTemplate
    public void testHeartBeatToConnectionTTLModifier() throws Exception {
        int port = 61614;
        this.server.getRemotingService().createAcceptor("test", "tcp://127.0.0.1:" + port + "?heartBeatToConnectionTtlModifier=1").start();
        StompClientConnection connection = StompClientConnectionFactory.createClientConnection(this.uri);
        ClientStompFrame frame = connection.createFrame("CONNECT").addHeader("host", "127.0.0.1").addHeader("login", this.defUser).addHeader("passcode", this.defPass).addHeader("heart-beat", "5000,0").addHeader("accept-version", "1.0,1.1");
        ClientStompFrame reply = connection.sendFrame(frame);
        Assertions.assertEquals((Object)"CONNECTED", (Object)reply.getCommand());
        Assertions.assertEquals((Object)"0,5000", (Object)reply.getHeader("heart-beat"));
        Thread.sleep(6000L);
        try {
            Assertions.assertFalse((boolean)connection.isConnected());
        }
        finally {
            connection.closeTransport();
        }
        this.server.getRemotingService().destroyAcceptor("test");
        this.server.getRemotingService().createAcceptor("test", "tcp://127.0.0.1:" + port + "?heartBeatToConnectionTtlModifier=1.5").start();
        connection = StompClientConnectionFactory.createClientConnection(this.uri);
        frame = connection.createFrame("CONNECT").addHeader("host", "127.0.0.1").addHeader("login", this.defUser).addHeader("passcode", this.defPass).addHeader("heart-beat", "5000,0").addHeader("accept-version", "1.0,1.1");
        reply = connection.sendFrame(frame);
        Assertions.assertEquals((Object)"CONNECTED", (Object)reply.getCommand());
        Assertions.assertEquals((Object)"0,5000", (Object)reply.getHeader("heart-beat"));
        Thread.sleep(6000L);
        connection.disconnect();
    }

    @TestTemplate
    public void testNack() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.subscribe(this.conn, "sub1", "client");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame();
        String messageID = frame.getHeader("message-id");
        StompV11Test.nack(this.conn, "sub1", messageID);
        StompV11Test.unsubscribe(this.conn, "sub1");
        this.conn.disconnect();
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Message message = consumer.receive(100L);
        Assertions.assertNull((Object)message);
    }

    @TestTemplate
    public void testNackWithWrongSubId() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.subscribe(this.conn, "sub1", "client");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame();
        String messageID = frame.getHeader("message-id");
        StompV11Test.nack(this.conn, "sub2", messageID);
        ClientStompFrame error = this.conn.receiveFrame();
        logger.debug("Receiver error: {}", (Object)error);
        StompV11Test.unsubscribe(this.conn, "sub1");
        this.conn.disconnect();
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Message message = consumer.receive(1000L);
        Assertions.assertNotNull((Object)message);
    }

    @TestTemplate
    public void testNackWithWrongMessageId() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.subscribe(this.conn, "sub1", "client");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame();
        frame.getHeader("message-id");
        StompV11Test.nack(this.conn, "sub2", "someother");
        ClientStompFrame error = this.conn.receiveFrame();
        logger.debug("Receiver error: {}", (Object)error);
        StompV11Test.unsubscribe(this.conn, "sub1");
        this.conn.disconnect();
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Message message = consumer.receive(1000L);
        Assertions.assertNotNull((Object)message);
    }

    @TestTemplate
    public void testAck() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.subscribe(this.conn, "sub1", "client");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame();
        String messageID = frame.getHeader("message-id");
        StompV11Test.ack(this.conn, "sub1", messageID, null);
        StompV11Test.unsubscribe(this.conn, "sub1");
        this.conn.disconnect();
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Message message = consumer.receive(100L);
        Assertions.assertNull((Object)message);
    }

    @TestTemplate
    public void testAckWithWrongSubId() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.subscribe(this.conn, "sub1", "client");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame();
        String messageID = frame.getHeader("message-id");
        StompV11Test.ack(this.conn, "sub2", messageID, null);
        ClientStompFrame error = this.conn.receiveFrame();
        logger.debug("Receiver error: {}", (Object)error);
        StompV11Test.unsubscribe(this.conn, "sub1");
        this.conn.disconnect();
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Message message = consumer.receive(1000L);
        Assertions.assertNotNull((Object)message);
    }

    @TestTemplate
    public void testAckWithWrongMessageId() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.subscribe(this.conn, "sub1", "client");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame();
        frame.getHeader("message-id");
        StompV11Test.ack(this.conn, "sub2", "someother", null);
        ClientStompFrame error = this.conn.receiveFrame();
        logger.debug("Receiver error: {}", (Object)error);
        StompV11Test.unsubscribe(this.conn, "sub1");
        this.conn.disconnect();
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Message message = consumer.receive(1000L);
        Assertions.assertNotNull((Object)message);
    }

    @TestTemplate
    public void testErrorWithReceipt() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.subscribe(this.conn, "sub1", "client");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame();
        String messageID = frame.getHeader("message-id");
        ClientStompFrame ackFrame = this.conn.createFrame("ACK").addHeader("subscription", "sub2").addHeader("message-id", messageID).addHeader("receipt", "answer-me");
        ClientStompFrame error = this.conn.sendFrame(ackFrame);
        logger.debug("Receiver error: {}", (Object)error);
        Assertions.assertEquals((Object)"ERROR", (Object)error.getCommand());
        Assertions.assertEquals((Object)"answer-me", (Object)error.getHeader("receipt-id"));
        StompV11Test.unsubscribe(this.conn, "sub1");
        this.conn.disconnect();
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Message message = consumer.receive(1000L);
        Assertions.assertNotNull((Object)message);
    }

    @TestTemplate
    public void testErrorWithReceipt2() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.subscribe(this.conn, "sub1", "client");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame();
        String messageID = frame.getHeader("message-id");
        ClientStompFrame ackFrame = this.conn.createFrame("ACK").addHeader("subscription", "sub1").addHeader("message-id", messageID + "1").addHeader("receipt", "answer-me");
        ClientStompFrame error = this.conn.sendFrame(ackFrame);
        logger.debug("Receiver error: {}", (Object)error);
        Assertions.assertEquals((Object)"ERROR", (Object)error.getCommand());
        Assertions.assertEquals((Object)"answer-me", (Object)error.getHeader("receipt-id"));
        StompV11Test.unsubscribe(this.conn, "sub1");
        this.conn.disconnect();
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Message message = consumer.receive(1000L);
        Assertions.assertNotNull((Object)message);
    }

    @TestTemplate
    public void testAckModeClient() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.subscribe(this.conn, "sub1", "client");
        int num = 50;
        for (int i = 0; i < num; ++i) {
            this.sendJmsMessage("client-ack" + i);
        }
        ClientStompFrame frame = null;
        for (int i = 0; i < num; ++i) {
            frame = this.conn.receiveFrame();
            Assertions.assertNotNull((Object)frame);
        }
        StompV11Test.ack(this.conn, "sub1", frame);
        StompV11Test.unsubscribe(this.conn, "sub1");
        this.conn.disconnect();
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Message message = consumer.receive(100L);
        Assertions.assertNull((Object)message);
    }

    @TestTemplate
    public void testAckModeClient2() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.subscribe(this.conn, "sub1", "client");
        Thread.sleep(1000L);
        int num = 50;
        for (int i = 0; i < num; ++i) {
            this.sendJmsMessage("client-ack" + i);
        }
        ClientStompFrame frame = null;
        for (int i = 0; i < num; ++i) {
            frame = this.conn.receiveFrame();
            Assertions.assertNotNull((Object)frame);
            if (i != num - 2) continue;
            StompV11Test.ack(this.conn, "sub1", frame);
        }
        StompV11Test.unsubscribe(this.conn, "sub1");
        this.conn.disconnect();
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Message message = consumer.receive(10000L);
        Assertions.assertNotNull((Object)message);
        message = consumer.receive(100L);
        Assertions.assertNull((Object)message);
    }

    @TestTemplate
    public void testAckModeAuto() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.subscribe(this.conn, "sub1", "auto");
        int num = 50;
        for (int i = 0; i < num; ++i) {
            this.sendJmsMessage("auto-ack" + i);
        }
        ClientStompFrame frame = null;
        for (int i = 0; i < num; ++i) {
            frame = this.conn.receiveFrame();
            Assertions.assertNotNull((Object)frame);
        }
        StompV11Test.unsubscribe(this.conn, "sub1");
        this.conn.disconnect();
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Message message = consumer.receive(100L);
        Assertions.assertNull((Object)message);
    }

    @TestTemplate
    public void testAckModeClientIndividual() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.subscribe(this.conn, "sub1", "client-individual");
        int num = 50;
        for (int i = 0; i < num; ++i) {
            this.sendJmsMessage("client-individual-ack" + i);
        }
        ClientStompFrame frame = null;
        for (int i = 0; i < num; ++i) {
            frame = this.conn.receiveFrame();
            Assertions.assertNotNull((Object)frame);
            logger.debug("{} == received: {}", (Object)i, (Object)frame);
            if (i % 2 != 0) continue;
            StompV11Test.ack(this.conn, "sub1", frame);
        }
        StompV11Test.unsubscribe(this.conn, "sub1");
        this.conn.disconnect();
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        TextMessage message = null;
        for (int i = 0; i < num / 2; ++i) {
            message = (TextMessage)consumer.receive(1000L);
            Assertions.assertNotNull((Object)message);
            logger.debug("Legal: {}", (Object)message.getText());
        }
        message = (TextMessage)consumer.receive(100L);
        Assertions.assertNull((Object)message);
    }

    @TestTemplate
    public void testTwoSubscribers() throws Exception {
        this.conn.connect(this.defUser, this.defPass, CLIENT_ID);
        StompV11Test.subscribeTopic(this.conn, "sub1", "auto", null);
        StompClientConnection newConn = StompClientConnectionFactory.createClientConnection(this.uri);
        newConn.connect(this.defUser, this.defPass, "myclientid2");
        StompV11Test.subscribeTopic(newConn, "sub2", "auto", null);
        StompV11Test.send(newConn, StompV11Test.getTopicPrefix() + StompV11Test.getTopicName(), null, "Hello World");
        ClientStompFrame frame = this.conn.receiveFrame(5000L);
        logger.debug("received frame : {}", (Object)frame);
        Assertions.assertEquals((Object)"Hello World", (Object)frame.getBody());
        Assertions.assertEquals((Object)"sub1", (Object)frame.getHeader("subscription"));
        frame = newConn.receiveFrame(5000L);
        logger.debug("received 2 frame : {}", (Object)frame);
        Assertions.assertEquals((Object)"Hello World", (Object)frame.getBody());
        Assertions.assertEquals((Object)"sub2", (Object)frame.getHeader("subscription"));
        StompV11Test.unsubscribe(this.conn, "sub1", true);
        StompV11Test.unsubscribe(newConn, "sub2", true);
        this.conn.disconnect();
        newConn.disconnect();
    }

    @TestTemplate
    public void testSendAndReceiveOnDifferentConnections() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.send(this.conn, StompV11Test.getQueuePrefix() + StompV11Test.getQueueName(), null, "Hello World");
        StompClientConnection connV11_2 = StompClientConnectionFactory.createClientConnection(this.uri);
        connV11_2.connect(this.defUser, this.defPass);
        StompV11Test.subscribe(connV11_2, "sub1", "auto");
        ClientStompFrame frame = connV11_2.receiveFrame(2000L);
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assertions.assertEquals((Object)"Hello World", (Object)frame.getBody());
        this.conn.disconnect();
        connV11_2.disconnect();
    }

    @TestTemplate
    public void testBeginSameTransactionTwice() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.beginTransaction(this.conn, "tx1");
        StompV11Test.beginTransaction(this.conn, "tx1");
        ClientStompFrame f = this.conn.receiveFrame();
        Assertions.assertTrue((boolean)f.getCommand().equals("ERROR"));
    }

    @TestTemplate
    public void testBodyWithUTF8() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.subscribe(this.conn, this.getName(), "auto");
        String text = "A\u00ea\u00f1\u00fcC";
        logger.debug(text);
        this.sendJmsMessage(text);
        ClientStompFrame frame = this.conn.receiveFrame();
        logger.debug("{}", (Object)frame);
        Assertions.assertTrue((boolean)frame.getCommand().equals("MESSAGE"));
        Assertions.assertNotNull((Object)frame.getHeader("destination"));
        Assertions.assertTrue((boolean)frame.getBody().equals(text));
        this.conn.disconnect();
    }

    @TestTemplate
    public void testClientAckNotPartOfTransaction() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.subscribe(this.conn, this.getName(), "client");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame();
        Assertions.assertTrue((boolean)frame.getCommand().equals("MESSAGE"));
        Assertions.assertNotNull((Object)frame.getHeader("destination"));
        Assertions.assertTrue((boolean)frame.getBody().equals(this.getName()));
        Assertions.assertNotNull((Object)frame.getHeader("message-id"));
        String messageID = frame.getHeader("message-id");
        StompV11Test.beginTransaction(this.conn, "tx1");
        StompV11Test.ack(this.conn, this.getName(), messageID, "tx1");
        StompV11Test.abortTransaction(this.conn, "tx1");
        frame = this.conn.receiveFrame(100L);
        Assertions.assertNull((Object)frame);
        StompV11Test.unsubscribe(this.conn, this.getName());
        this.conn.disconnect();
    }

    @TestTemplate
    public void testDisconnectAndError() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.subscribe(this.conn, this.getName(), "client");
        String uuid = UUID.randomUUID().toString();
        ClientStompFrame frame = this.conn.createFrame("DISCONNECT").addHeader("receipt", uuid);
        ClientStompFrame result = this.conn.sendFrame(frame);
        if (result == null || !"RECEIPT".equals(result.getCommand()) || !uuid.equals(result.getHeader("receipt-id"))) {
            Assertions.fail((String)("Disconnect failed! " + result));
        }
        CountDownLatch latch = new CountDownLatch(1);
        Thread thr = new Thread(() -> {
            while (latch.getCount() != 0L) {
                try {
                    StompV11Test.send(this.conn, StompV11Test.getQueuePrefix() + StompV11Test.getQueueName(), null, "Hello World");
                    Thread.sleep(500L);
                }
                catch (InterruptedException interruptedException) {
                }
                catch (ClosedChannelException e) {
                    latch.countDown();
                    return;
                }
                catch (IOException e) {
                    latch.countDown();
                    return;
                }
                finally {
                    this.conn.destroy();
                }
            }
        });
        thr.start();
        latch.await(10L, TimeUnit.SECONDS);
        long count = latch.getCount();
        if (count != 0L) {
            latch.countDown();
        }
        thr.join();
        Assertions.assertTrue((count == 0L ? (byte)1 : 0) != 0, (String)"Server failed to disconnect.");
    }

    @TestTemplate
    public void testDurableSubscriber() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.subscribe(this.conn, "sub1", "client", this.getName());
        StompV11Test.subscribe(this.conn, "sub1", "client", this.getName(), false);
        ClientStompFrame frame = this.conn.receiveFrame();
        Assertions.assertTrue((boolean)frame.getCommand().equals("ERROR"));
        this.conn.disconnect();
    }

    @TestTemplate
    public void testDurableSubscriberWithReconnection() throws Exception {
        this.conn.connect(this.defUser, this.defPass, CLIENT_ID);
        StompV11Test.subscribeTopic(this.conn, "sub1", "auto", this.getName());
        String uuid = UUID.randomUUID().toString();
        ClientStompFrame frame = this.conn.createFrame("DISCONNECT").addHeader("receipt", uuid);
        ClientStompFrame result = this.conn.sendFrame(frame);
        if (result == null || !"RECEIPT".equals(result.getCommand()) || !uuid.equals(result.getHeader("receipt-id"))) {
            Assertions.fail((String)("Disconnect failed! " + result));
        }
        this.sendJmsMessage(this.getName(), (Destination)this.topic);
        this.conn.destroy();
        this.conn = StompClientConnectionFactory.createClientConnection(this.uri);
        this.conn.connect(this.defUser, this.defPass, CLIENT_ID);
        StompV11Test.subscribeTopic(this.conn, "sub1", "auto", this.getName());
        frame = this.conn.receiveFrame();
        Assertions.assertTrue((boolean)frame.getCommand().equals("MESSAGE"));
        Assertions.assertNotNull((Object)frame.getHeader("destination"));
        Assertions.assertEquals((Object)this.getName(), (Object)frame.getBody());
        StompV11Test.unsubscribe(this.conn, "sub1");
        this.conn.disconnect();
    }

    @TestTemplate
    public void testDurableUnSubscribe() throws Exception {
        SimpleString queueName = SimpleString.of((String)("myclientid." + this.getName()));
        this.conn.connect(this.defUser, this.defPass, CLIENT_ID);
        StompV11Test.subscribeTopic(this.conn, null, "auto", this.getName());
        Wait.assertTrue(() -> this.server.locateQueue(queueName) != null);
        this.conn.disconnect();
        this.conn.destroy();
        Queue queue = this.server.locateQueue(queueName);
        Assertions.assertNotNull((Object)queue);
        Wait.assertEquals((int)0, () -> ((Queue)queue).getConsumerCount(), (long)10000L, (long)10L);
        this.conn = StompClientConnectionFactory.createClientConnection(this.uri);
        this.conn.connect(this.defUser, this.defPass, CLIENT_ID);
        StompV11Test.unsubscribe(this.conn, this.getName(), null, false, true);
        Wait.assertTrue(() -> this.server.locateQueue(queueName) == null);
        this.conn.disconnect();
    }

    @TestTemplate
    public void testJMSXGroupIdCanBeSet() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompV11Test.getQueuePrefix() + StompV11Test.getQueueName()).addHeader("JMSXGroupID", "TEST").setBody("Hello World");
        this.conn.sendFrame(frame);
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assertions.assertNotNull((Object)message);
        Assertions.assertEquals((Object)"Hello World", (Object)message.getText());
        Assertions.assertEquals((Object)"TEST", (Object)message.getStringProperty("JMSXGroupID"));
    }

    @TestTemplate
    public void testMessagesAreInOrder() throws Exception {
        int i;
        int ctr = 10;
        String[] data = new String[ctr];
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.subscribe(this.conn, "sub1", "auto");
        for (int i2 = 0; i2 < ctr; ++i2) {
            data[i2] = this.getName() + i2;
            this.sendJmsMessage(data[i2]);
        }
        ClientStompFrame frame = null;
        for (i = 0; i < ctr; ++i) {
            frame = this.conn.receiveFrame();
            Assertions.assertTrue((boolean)frame.getBody().equals(data[i]), (String)"Message not in order");
        }
        for (i = 0; i < ctr; ++i) {
            data[i] = this.getName() + ":second:" + i;
            this.sendJmsMessage(data[i]);
        }
        for (i = 0; i < ctr; ++i) {
            frame = this.conn.receiveFrame();
            Assertions.assertTrue((boolean)frame.getBody().equals(data[i]), (String)"Message not in order");
        }
        this.conn.disconnect();
    }

    @TestTemplate
    public void testSubscribeWithAutoAckAndSelector() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.subscribe(this.conn, "sub1", "auto", null, "foo = 'zzz'");
        this.sendJmsMessage("Ignored message", "foo", "1234");
        this.sendJmsMessage("Real message", "foo", "zzz");
        ClientStompFrame frame = this.conn.receiveFrame();
        Assertions.assertTrue((boolean)frame.getBody().equals("Real message"), (String)("Should have received the real message but got: " + frame));
        this.conn.disconnect();
    }

    @TestTemplate
    public void testRedeliveryWithClientAck() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.subscribe(this.conn, "subscriptionId", "client");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame();
        Assertions.assertTrue((boolean)frame.getCommand().equals("MESSAGE"));
        this.conn.disconnect();
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Message message = consumer.receive(1000L);
        Assertions.assertNotNull((Object)message);
        Assertions.assertTrue((boolean)message.getJMSRedelivered());
    }

    @TestTemplate
    public void testSendManyMessages() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        int count = 1000;
        CountDownLatch latch = new CountDownLatch(count);
        consumer.setMessageListener(arg0 -> latch.countDown());
        for (int i = 1; i <= count; ++i) {
            StompV11Test.send(this.conn, StompV11Test.getQueuePrefix() + StompV11Test.getQueueName(), null, "Hello World");
        }
        Assertions.assertTrue((boolean)latch.await(60L, TimeUnit.SECONDS));
        this.conn.disconnect();
    }

    @TestTemplate
    public void testSendMessage() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.send(this.conn, StompV11Test.getQueuePrefix() + StompV11Test.getQueueName(), null, "Hello World");
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assertions.assertNotNull((Object)message);
        Assertions.assertEquals((Object)"Hello World", (Object)message.getText());
        Assertions.assertEquals((int)4, (int)message.getJMSPriority(), (String)"getJMSPriority");
        long tnow = System.currentTimeMillis();
        long tmsg = message.getJMSTimestamp();
        Assertions.assertTrue((Math.abs(tnow - tmsg) < 1000L ? (byte)1 : 0) != 0);
    }

    @TestTemplate
    public void testSendMessageWithContentLength() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        byte[] data = new byte[]{1, 0, 0, 4};
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompV11Test.getQueuePrefix() + StompV11Test.getQueueName()).setBody(new String(data, StandardCharsets.UTF_8)).addHeader("content-length", String.valueOf(data.length));
        this.conn.sendFrame(frame);
        BytesMessage message = (BytesMessage)consumer.receive(10000L);
        Assertions.assertNotNull((Object)message);
        Assertions.assertEquals((long)data.length, (long)message.getBodyLength());
        Assertions.assertEquals((byte)data[0], (byte)message.readByte());
        Assertions.assertEquals((byte)data[1], (byte)message.readByte());
        Assertions.assertEquals((byte)data[2], (byte)message.readByte());
        Assertions.assertEquals((byte)data[3], (byte)message.readByte());
    }

    @TestTemplate
    public void testSendMessageWithCustomHeadersAndSelector() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue, "foo = 'abc'");
        this.conn.connect(this.defUser, this.defPass);
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("foo", "abc").addHeader("bar", "123").addHeader("destination", StompV11Test.getQueuePrefix() + StompV11Test.getQueueName()).setBody("Hello World");
        this.conn.sendFrame(frame);
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assertions.assertNotNull((Object)message);
        Assertions.assertEquals((Object)"Hello World", (Object)message.getText());
        Assertions.assertEquals((Object)"abc", (Object)message.getStringProperty("foo"), (String)"foo");
        Assertions.assertEquals((Object)"123", (Object)message.getStringProperty("bar"), (String)"bar");
    }

    @TestTemplate
    public void testSendMessageWithLeadingNewLine() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Thread.sleep(1000L);
        this.conn.connect(this.defUser, this.defPass);
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompV11Test.getQueuePrefix() + StompV11Test.getQueueName()).setBody("Hello World");
        this.conn.sendWickedFrame(frame);
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assertions.assertNotNull((Object)message);
        Assertions.assertEquals((Object)"Hello World", (Object)message.getText());
        long tnow = System.currentTimeMillis();
        long tmsg = message.getJMSTimestamp();
        Assertions.assertTrue((Math.abs(tnow - tmsg) < 1000L ? (byte)1 : 0) != 0);
        Assertions.assertNull((Object)consumer.receive(100L));
        this.conn.disconnect();
    }

    @TestTemplate
    public void testSendMessageWithReceipt() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.send(this.conn, StompV11Test.getQueuePrefix() + StompV11Test.getQueueName(), null, "Hello World", true);
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assertions.assertNotNull((Object)message);
        Assertions.assertEquals((Object)"Hello World", (Object)message.getText());
        long tnow = System.currentTimeMillis();
        long tmsg = message.getJMSTimestamp();
        Assertions.assertTrue((Math.abs(tnow - tmsg) < 1000L ? (byte)1 : 0) != 0);
        this.conn.disconnect();
    }

    @TestTemplate
    public void testSendMessageWithStandardHeaders() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompV11Test.getQueuePrefix() + StompV11Test.getQueueName()).addHeader("correlation-id", "c123").addHeader("persistent", "true").addHeader("priority", "3").addHeader("type", "t345").addHeader("JMSXGroupID", "abc").addHeader("foo", "abc").addHeader("bar", "123").setBody("Hello World");
        frame = this.conn.sendFrame(frame);
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assertions.assertNotNull((Object)message);
        Assertions.assertEquals((Object)"Hello World", (Object)message.getText());
        Assertions.assertEquals((Object)"c123", (Object)message.getJMSCorrelationID(), (String)"JMSCorrelationID");
        Assertions.assertEquals((Object)"t345", (Object)message.getJMSType(), (String)"getJMSType");
        Assertions.assertEquals((int)3, (int)message.getJMSPriority(), (String)"getJMSPriority");
        Assertions.assertEquals((int)2, (int)message.getJMSDeliveryMode());
        Assertions.assertEquals((Object)"abc", (Object)message.getStringProperty("foo"), (String)"foo");
        Assertions.assertEquals((Object)"123", (Object)message.getStringProperty("bar"), (String)"bar");
        Assertions.assertEquals((Object)"abc", (Object)message.getStringProperty("JMSXGroupID"), (String)"JMSXGroupID");
        this.conn.disconnect();
    }

    @TestTemplate
    public void testSendMessageWithLongHeaders() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        StringBuffer buffer = new StringBuffer();
        for (int i = 0; i < 2048; ++i) {
            buffer.append("a");
        }
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompV11Test.getQueuePrefix() + StompV11Test.getQueueName()).addHeader("correlation-id", "c123").addHeader("persistent", "true").addHeader("priority", "3").addHeader("type", "t345").addHeader("JMSXGroupID", "abc").addHeader("foo", "abc").addHeader("longHeader", buffer.toString()).setBody("Hello World");
        frame = this.conn.sendFrame(frame);
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assertions.assertNotNull((Object)message);
        Assertions.assertEquals((Object)"Hello World", (Object)message.getText());
        Assertions.assertEquals((Object)"c123", (Object)message.getJMSCorrelationID(), (String)"JMSCorrelationID");
        Assertions.assertEquals((Object)"t345", (Object)message.getJMSType(), (String)"getJMSType");
        Assertions.assertEquals((int)3, (int)message.getJMSPriority(), (String)"getJMSPriority");
        Assertions.assertEquals((int)2, (int)message.getJMSDeliveryMode());
        Assertions.assertEquals((Object)"abc", (Object)message.getStringProperty("foo"), (String)"foo");
        Assertions.assertEquals((int)2048, (int)message.getStringProperty("longHeader").length(), (String)"longHeader");
        Assertions.assertEquals((Object)"abc", (Object)message.getStringProperty("JMSXGroupID"), (String)"JMSXGroupID");
        this.conn.disconnect();
    }

    @TestTemplate
    public void testSubscribeToTopic() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.subscribeTopic(this.conn, "sub1", null, null, true);
        this.sendJmsMessage(this.getName(), (Destination)this.topic);
        ClientStompFrame frame = this.conn.receiveFrame();
        Assertions.assertTrue((boolean)frame.getCommand().equals("MESSAGE"));
        Assertions.assertTrue((boolean)frame.getHeader("destination").equals(StompV11Test.getTopicPrefix() + StompV11Test.getTopicName()));
        Assertions.assertTrue((boolean)frame.getBody().equals(this.getName()));
        StompV11Test.unsubscribe(this.conn, "sub1", true);
        this.sendJmsMessage(this.getName(), (Destination)this.topic);
        frame = this.conn.receiveFrame(100L);
        Assertions.assertNull((Object)frame);
        this.conn.disconnect();
    }

    @TestTemplate
    public void testSubscribeToTopicWithNoLocal() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.subscribeTopic(this.conn, "sub1", null, null, true, true);
        StompV11Test.send(this.conn, StompV11Test.getTopicPrefix() + StompV11Test.getTopicName(), null, "Hello World");
        ClientStompFrame frame = this.conn.receiveFrame(100L);
        Assertions.assertNull((Object)frame);
        this.sendJmsMessage(this.getName(), (Destination)this.topic);
        frame = this.conn.receiveFrame();
        Assertions.assertTrue((boolean)frame.getCommand().equals("MESSAGE"));
        Assertions.assertTrue((boolean)frame.getHeader("destination").equals(StompV11Test.getTopicPrefix() + StompV11Test.getTopicName()));
        Assertions.assertTrue((boolean)frame.getBody().equals(this.getName()));
        StompV11Test.unsubscribe(this.conn, "sub1");
        this.conn.disconnect();
    }

    @TestTemplate
    public void testSubscribeWithAutoAck() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.subscribe(this.conn, "sub1", "auto");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame();
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assertions.assertNotNull((Object)frame.getHeader("destination"));
        Assertions.assertEquals((Object)this.getName(), (Object)frame.getBody());
        this.conn.disconnect();
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Message message = consumer.receive(100L);
        Assertions.assertNull((Object)message);
    }

    @TestTemplate
    public void testSubscribeWithAutoAckAndBytesMessage() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.subscribe(this.conn, "sub1", "auto");
        byte[] payload = new byte[]{1, 2, 3, 4, 5};
        this.sendJmsMessage(payload, (Destination)this.queue);
        ClientStompFrame frame = this.conn.receiveFrame();
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        logger.debug("Message: {}", (Object)frame);
        Assertions.assertEquals((Object)"5", (Object)frame.getHeader("content-length"));
        Assertions.assertNull((Object)frame.getHeader("type"));
        Assertions.assertEquals((Object)frame.getBody(), (Object)new String(payload, StandardCharsets.UTF_8));
        this.conn.disconnect();
    }

    @TestTemplate
    public void testSubscribeWithClientAck() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.subscribe(this.conn, "sub1", "client");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame();
        Assertions.assertEquals((int)this.getName().length(), (int)Integer.parseInt(frame.getHeader("content-length")));
        StompV11Test.ack(this.conn, "sub1", frame);
        this.conn.disconnect();
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Message message = consumer.receive(100L);
        Assertions.assertNull((Object)message);
    }

    @TestTemplate
    public void testSubscribeWithClientAckThenConsumingAgainWithAutoAckWithExplicitDisconnect() throws Exception {
        this.assertSubscribeWithClientAckThenConsumeWithAutoAck(true);
    }

    @TestTemplate
    public void testSubscribeWithClientAckThenConsumingAgainWithAutoAckWithNoDisconnectFrame() throws Exception {
        this.assertSubscribeWithClientAckThenConsumeWithAutoAck(false);
    }

    @TestTemplate
    public void testSubscribeWithID() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.subscribe(this.conn, "mysubid", "auto");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame();
        Assertions.assertTrue((frame.getHeader("subscription") != null ? (byte)1 : 0) != 0);
        this.conn.disconnect();
    }

    @TestTemplate
    public void testSubscribeWithMessageSentWithProperties() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.subscribe(this.conn, "sub1", "auto");
        MessageProducer producer = this.session.createProducer((Destination)this.queue);
        BytesMessage message = this.session.createBytesMessage();
        message.setStringProperty("S", "value");
        message.setBooleanProperty("n", false);
        message.setByteProperty("byte", (byte)9);
        message.setDoubleProperty("d", 2.0);
        message.setFloatProperty("f", 6.0f);
        message.setIntProperty("i", 10);
        message.setLongProperty("l", 121L);
        message.setShortProperty("s", (short)12);
        message.writeBytes("Hello World".getBytes(StandardCharsets.UTF_8));
        producer.send((Message)message);
        ClientStompFrame frame = this.conn.receiveFrame();
        Assertions.assertNotNull((Object)frame);
        Assertions.assertTrue((frame.getHeader("S") != null ? (byte)1 : 0) != 0);
        Assertions.assertTrue((frame.getHeader("n") != null ? (byte)1 : 0) != 0);
        Assertions.assertTrue((frame.getHeader("byte") != null ? (byte)1 : 0) != 0);
        Assertions.assertTrue((frame.getHeader("d") != null ? (byte)1 : 0) != 0);
        Assertions.assertTrue((frame.getHeader("f") != null ? (byte)1 : 0) != 0);
        Assertions.assertTrue((frame.getHeader("i") != null ? (byte)1 : 0) != 0);
        Assertions.assertTrue((frame.getHeader("l") != null ? (byte)1 : 0) != 0);
        Assertions.assertTrue((frame.getHeader("s") != null ? (byte)1 : 0) != 0);
        Assertions.assertEquals((Object)"Hello World", (Object)frame.getBody());
        this.conn.disconnect();
    }

    @TestTemplate
    public void testSuccessiveTransactionsWithSameID() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.beginTransaction(this.conn, "tx1");
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompV11Test.getQueuePrefix() + StompV11Test.getQueueName()).addHeader("transaction", "tx1").setBody("Hello World");
        this.conn.sendFrame(frame);
        StompV11Test.commitTransaction(this.conn, "tx1");
        Message message = consumer.receive(1000L);
        Assertions.assertNotNull((Object)message, (String)"Should have received a message");
        StompV11Test.beginTransaction(this.conn, "tx1");
        frame = this.conn.createFrame("SEND").addHeader("destination", StompV11Test.getQueuePrefix() + StompV11Test.getQueueName()).addHeader("transaction", "tx1").setBody("Hello World");
        this.conn.sendFrame(frame);
        StompV11Test.commitTransaction(this.conn, "tx1");
        message = consumer.receive(1000L);
        Assertions.assertNotNull((Object)message, (String)"Should have received a message");
        this.conn.disconnect();
    }

    @TestTemplate
    public void testTransactionCommit() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.beginTransaction(this.conn, "tx1");
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompV11Test.getQueuePrefix() + StompV11Test.getQueueName()).addHeader("transaction", "tx1").addHeader("receipt", "123").setBody("Hello World");
        frame = this.conn.sendFrame(frame);
        Assertions.assertEquals((Object)"123", (Object)frame.getHeader("receipt-id"));
        Assertions.assertNull((Object)consumer.receive(100L));
        StompV11Test.commitTransaction(this.conn, "tx1", true);
        Message message = consumer.receive(1000L);
        Assertions.assertNotNull((Object)message, (String)"Should have received a message");
        this.conn.disconnect();
    }

    @TestTemplate
    public void testTransactionRollback() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.beginTransaction(this.conn, "tx1");
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompV11Test.getQueuePrefix() + StompV11Test.getQueueName()).addHeader("transaction", "tx1").setBody("first message");
        this.conn.sendFrame(frame);
        StompV11Test.abortTransaction(this.conn, "tx1");
        StompV11Test.beginTransaction(this.conn, "tx1");
        frame = this.conn.createFrame("SEND").addHeader("destination", StompV11Test.getQueuePrefix() + StompV11Test.getQueueName()).addHeader("transaction", "tx1").setBody("second message");
        this.conn.sendFrame(frame);
        StompV11Test.commitTransaction(this.conn, "tx1", true);
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assertions.assertNotNull((Object)message);
        Assertions.assertEquals((Object)"second message", (Object)message.getText());
        this.conn.disconnect();
    }

    @TestTemplate
    public void testUnsubscribe() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.subscribe(this.conn, "sub1", "auto");
        this.sendJmsMessage("first message");
        ClientStompFrame frame = this.conn.receiveFrame();
        Assertions.assertTrue((boolean)frame.getCommand().equals("MESSAGE"));
        StompV11Test.unsubscribe(this.conn, "sub1", true);
        this.sendJmsMessage("second message");
        frame = this.conn.receiveFrame(100L);
        Assertions.assertNull((Object)frame);
        this.conn.disconnect();
    }

    @TestTemplate
    public void testHeartBeat3() throws Exception {
        this.connection.close();
        Assertions.assertEquals((int)0, (int)this.server.getRemotingService().getConnections().size());
        ClientStompFrame frame = this.conn.createFrame("CONNECT");
        frame.addHeader("host", "127.0.0.1");
        frame.addHeader("login", this.defUser);
        frame.addHeader("passcode", this.defPass);
        frame.addHeader("heart-beat", "500,500");
        frame.addHeader("accept-version", "1.0,1.1");
        ClientStompFrame reply = this.conn.sendFrame(frame);
        Assertions.assertEquals((Object)"CONNECTED", (Object)reply.getCommand());
        Assertions.assertEquals((Object)"500,500", (Object)reply.getHeader("heart-beat"));
        logger.debug("========== start pinger!");
        this.conn.startPinger(100L);
        Assertions.assertEquals((int)1, (int)this.server.getRemotingService().getConnections().size());
        StompConnection stompConnection = (StompConnection)this.server.getRemotingService().getConnections().iterator().next();
        StompFrameHandlerV11 stompFrameHandler = (StompFrameHandlerV11)stompConnection.getStompVersionHandler();
        Thread.sleep(1000L);
        int size = this.conn.getServerPingNumber();
        this.conn.stopPinger();
        Wait.assertEquals((int)0, () -> this.server.getRemotingService().getConnections().size());
        Wait.assertFalse((String)"HeartBeater is still running!!", () -> stompFrameHandler.getHeartBeater().isStarted());
    }

    @TestTemplate
    public void testHeartBeat4() throws Exception {
        this.connection.close();
        ClientStompFrame frame = this.conn.createFrame("CONNECT");
        frame.addHeader("host", "127.0.0.1");
        frame.addHeader("login", this.defUser);
        frame.addHeader("passcode", this.defPass);
        frame.addHeader("heart-beat", "500,500");
        frame.addHeader("accept-version", "1.1,1.2");
        ClientStompFrame reply = this.conn.sendFrame(frame);
        logger.debug("Reply: {}", (Object)reply.toString());
        Assertions.assertEquals((Object)"CONNECTED", (Object)reply.getCommand());
        RemotingConnection remotingConnection2 = null;
        StompConnection stompConnection = null;
        for (RemotingConnection remotingConnection2 : this.server.getRemotingService().getConnections()) {
            if (!(remotingConnection2 instanceof StompConnection)) continue;
            stompConnection = (StompConnection)remotingConnection2;
        }
        StompFrameHandlerV11 stompFrameHandler = (StompFrameHandlerV11)stompConnection.getStompVersionHandler();
        logger.debug("========== start pinger!");
        this.conn.startPinger(100L);
        ClientStompFrame subFrame = this.conn.createFrame("SUBSCRIBE");
        subFrame.addHeader("destination", StompV11Test.getTopicPrefix() + StompV11Test.getTopicName());
        subFrame.addHeader("id", "0");
        ClientStompFrame f = this.conn.sendFrame(subFrame);
        f = this.conn.sendFrame(subFrame);
        f = this.conn.sendFrame(subFrame);
        f = this.conn.receiveFrame(1000L);
        logger.debug("Received {}", (Object)f.toString());
        Assertions.assertTrue((boolean)f.getCommand().equals("ERROR"));
        this.conn.stopPinger();
        Wait.assertEquals((int)0, () -> this.server.getRemotingService().getConnections().size());
        Wait.assertFalse((String)"HeartBeater is still running!!", () -> stompFrameHandler.getHeartBeater().isStarted());
    }

    protected void assertSubscribeWithClientAckThenConsumeWithAutoAck(boolean sendDisconnect) throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.subscribe(this.conn, "sub1", "client");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame();
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        logger.info("Reconnecting!");
        if (sendDisconnect) {
            this.conn.disconnect();
            this.conn = StompClientConnectionFactory.createClientConnection(this.uri);
        } else {
            this.conn.destroy();
            this.conn = StompClientConnectionFactory.createClientConnection(this.uri);
        }
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.subscribe(this.conn, "sub1", null);
        frame = this.conn.receiveFrame();
        Assertions.assertTrue((boolean)frame.getCommand().equals("MESSAGE"));
        this.conn.disconnect();
        this.conn.destroy();
        this.conn = StompClientConnectionFactory.createClientConnection(this.uri);
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.subscribe(this.conn, "sub1", null, null, true);
        this.sendJmsMessage("shouldBeNextMessage");
        frame = this.conn.receiveFrame();
        Assertions.assertTrue((boolean)frame.getCommand().equals("MESSAGE"));
        Assertions.assertEquals((Object)"shouldBeNextMessage", (Object)frame.getBody());
    }

    @TestTemplate
    public void testSendMessageToNonExistentQueueWithAutoCreation() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.send(this.conn, "NonExistentQueue" + UUID.randomUUID().toString(), null, "Hello World", true, RoutingType.ANYCAST);
        this.conn.disconnect();
    }

    @TestTemplate
    public void testSendAndReceiveWithEscapedCharactersInSenderId() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.send(this.conn, StompV11Test.getQueuePrefix() + StompV11Test.getQueueName(), null, "Hello World 1!");
        StompV11Test.subscribe(this.conn, "ID\\cMYMACHINE-50616-635482262727823605-1\\c1\\c1\\c1");
        ClientStompFrame frame = this.conn.receiveFrame();
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assertions.assertEquals((Object)"ID:MYMACHINE-50616-635482262727823605-1:1:1:1", (Object)frame.getHeader("subscription"));
        Assertions.assertNotNull((Object)frame.getHeader("message-id"));
        Assertions.assertEquals((Object)(StompV11Test.getQueuePrefix() + StompV11Test.getQueueName()), (Object)frame.getHeader("destination"));
        Assertions.assertEquals((Object)"Hello World 1!", (Object)frame.getBody());
        StompV11Test.unsubscribe(this.conn, "ID\\cMYMACHINE-50616-635482262727823605-1\\c1\\c1\\c1");
        this.conn.disconnect();
    }

    @TestTemplate
    public void testReceiveContentType() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        ClientStompFrame response = StompV11Test.send(this.conn, StompV11Test.getQueuePrefix() + StompV11Test.getQueueName(), "text/plain", "Hello World");
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assertions.assertNotNull((Object)message);
        Assertions.assertEquals((Object)"text/plain", (Object)message.getStringProperty(org.apache.activemq.artemis.api.core.Message.HDR_CONTENT_TYPE.toString()));
    }

    @TestTemplate
    public void testSendContentType() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.subscribe(this.conn, "sub1", "auto");
        MessageProducer producer = this.session.createProducer((Destination)this.queue);
        BytesMessage message = this.session.createBytesMessage();
        message.setStringProperty(org.apache.activemq.artemis.api.core.Message.HDR_CONTENT_TYPE.toString(), "text/plain");
        message.writeBytes("Hello World".getBytes(StandardCharsets.UTF_8));
        producer.send((Message)message);
        ClientStompFrame frame = this.conn.receiveFrame();
        Assertions.assertNotNull((Object)frame);
        Assertions.assertEquals((Object)"text/plain", (Object)frame.getHeader("content-type"));
        this.conn.disconnect();
    }

    @TestTemplate
    public void testSameMessageHasDifferentMessageIdPerConsumer() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompV11Test.subscribeTopic(this.conn, "sub1", "client-individual", null);
        StompV11Test.subscribeTopic(this.conn, "sub2", "client-individual", null);
        this.sendJmsMessage(this.getName(), (Destination)this.topic);
        ClientStompFrame frame1 = this.conn.receiveFrame();
        String firstMessageID = frame1.getHeader("message-id");
        Assertions.assertNotNull((Object)firstMessageID);
        ClientStompFrame frame2 = this.conn.receiveFrame();
        String secondMessageID = frame2.getHeader("message-id");
        Assertions.assertNotNull((Object)secondMessageID);
        Assertions.assertTrue((!firstMessageID.equals(secondMessageID) ? (byte)1 : 0) != 0, (String)(firstMessageID + " must not equal " + secondMessageID));
        StompV11Test.ack(this.conn, "sub1", frame1);
        StompV11Test.ack(this.conn, "sub2", frame2);
        StompV11Test.unsubscribe(this.conn, "sub1");
        StompV11Test.unsubscribe(this.conn, "sub2");
        this.conn.disconnect();
    }
}

