/*
 * Decompiled with CFR 0.152.
 */
package org.wildfly.security.pem;

import java.io.IOException;
import java.security.GeneralSecurityException;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.function.BiFunction;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.sshd.common.config.keys.FilePasswordProvider;
import org.apache.sshd.common.config.keys.loader.openssh.OpenSSHKeyPairResourceParser;
import org.wildfly.common.Assert;
import org.wildfly.common.bytes.ByteStringBuilder;
import org.wildfly.common.iteration.ByteIterator;
import org.wildfly.common.iteration.CodePointIterator;
import org.wildfly.security.asn1.DERDecoder;
import org.wildfly.security.pem.PemEntry;
import org.wildfly.security.x500.cert.PKCS10CertificateSigningRequest;
import org.wildfly.security.x500.cert._private.ElytronMessages;

public final class Pem {
    private static final Pattern VALID_LABEL = Pattern.compile("[^ -~&&[^-]]");
    private static final String PUBLIC_KEY_FORMAT = "PUBLIC KEY";
    private static final String CERTIFICATE_FORMAT = "CERTIFICATE";
    private static final String PRIVATE_KEY_FORMAT = "PRIVATE KEY";
    private static final String CERTIFICATE_REQUEST_FORMAT = "CERTIFICATE REQUEST";
    public static final String OPENSSH_PRIVATE_KEY_FORMAT = "OPENSSH PRIVATE KEY";

    public static <R> R parsePemContent(CodePointIterator pemContent, BiFunction<String, ByteIterator, R> contentFunction) throws IllegalArgumentException {
        Assert.checkNotNullParam((String)"pemContent", (Object)pemContent);
        Assert.checkNotNullParam((String)"contentFunction", contentFunction);
        long matchIdx = -1L;
        while (pemContent.hasNext()) {
            int cp = pemContent.next();
            if (cp != 45) continue;
            long idx = pemContent.getIndex();
            if (pemContent.limitedTo(10L).contentEquals("----BEGIN ")) {
                String type = pemContent.delimitedBy(new int[]{45}).drainToString();
                Matcher matcher = VALID_LABEL.matcher(type);
                if (!matcher.find() && pemContent.limitedTo(5L).contentEquals("-----")) {
                    matchIdx = idx;
                    break;
                }
                while (pemContent.getIndex() > idx) {
                    pemContent.previous();
                }
                continue;
            }
            while (pemContent.getIndex() > idx) {
                pemContent.previous();
            }
        }
        if (matchIdx == -1L) {
            return null;
        }
        while (pemContent.getIndex() > matchIdx) {
            pemContent.previous();
        }
        if (!pemContent.limitedTo(10L).contentEquals("----BEGIN ")) {
            throw ElytronMessages.log.malformedPemContent(pemContent.getIndex());
        }
        String type = pemContent.delimitedBy(new int[]{45}).drainToString();
        Matcher matcher = VALID_LABEL.matcher(type);
        if (matcher.find()) {
            throw ElytronMessages.log.malformedPemContent(matcher.start() + 11);
        }
        if (!pemContent.limitedTo(5L).contentEquals("-----")) {
            throw ElytronMessages.log.malformedPemContent(pemContent.getIndex());
        }
        CodePointIterator delimitedIterator = pemContent.delimitedBy(new int[]{45}).skip(Character::isWhitespace).skipCrLf();
        ByteIterator byteIterator = delimitedIterator.base64Decode();
        R result = contentFunction.apply(type, byteIterator);
        delimitedIterator.skipAll();
        if (!pemContent.limitedTo(9L).contentEquals("-----END ")) {
            throw ElytronMessages.log.malformedPemContent(pemContent.getIndex());
        }
        if (!pemContent.limitedTo((long)type.length()).contentEquals(type)) {
            throw ElytronMessages.log.malformedPemContent(pemContent.getIndex());
        }
        if (!pemContent.limitedTo(5L).contentEquals("-----")) {
            throw ElytronMessages.log.malformedPemContent(pemContent.getIndex());
        }
        return result;
    }

    public static Iterator<PemEntry<?>> parsePemContent(final CodePointIterator pemContent) {
        return new Iterator<PemEntry<?>>(){
            private PemEntry<?> next;

            @Override
            public boolean hasNext() {
                if (this.next == null) {
                    if (!pemContent.hasNext()) {
                        return false;
                    }
                    this.next = Pem.parsePemContent(pemContent, (type, byteIterator) -> {
                        switch (type) {
                            case "CERTIFICATE": {
                                X509Certificate x509Certificate = Pem.parsePemX509CertificateContent(type, byteIterator);
                                return new PemEntry<X509Certificate>(x509Certificate);
                            }
                            case "PUBLIC KEY": {
                                PublicKey publicKey = Pem.parsePemPublicKey(type, byteIterator);
                                return new PemEntry<PublicKey>(publicKey);
                            }
                            case "PRIVATE KEY": {
                                PrivateKey privateKey = Pem.parsePemPrivateKey(type, byteIterator);
                                return new PemEntry<PrivateKey>(privateKey);
                            }
                        }
                        throw ElytronMessages.log.malformedPemContent(pemContent.getIndex());
                    });
                    if (this.next == null) {
                        return false;
                    }
                }
                return true;
            }

            @Override
            public PemEntry<?> next() {
                if (!this.hasNext()) {
                    throw new NoSuchElementException();
                }
                try {
                    PemEntry<?> pemEntry = this.next;
                    return pemEntry;
                }
                finally {
                    this.next = null;
                }
            }
        };
    }

    public static Iterator<PemEntry<?>> parsePemOpenSSHContent(final CodePointIterator pemContent, final FilePasswordProvider passphraseProvider) throws IllegalArgumentException {
        return new Iterator<PemEntry<?>>(){
            private PemEntry<?> next;

            @Override
            public boolean hasNext() {
                if (this.next == null) {
                    if (!pemContent.hasNext()) {
                        return false;
                    }
                    this.next = Pem.parsePemContent(pemContent, (type, byteIterator) -> {
                        switch (type) {
                            case "OPENSSH PRIVATE KEY": {
                                KeyPair keyPair = Pem.parseOpenSSHKeys(byteIterator, passphraseProvider);
                                return new PemEntry<KeyPair>(keyPair);
                            }
                        }
                        throw ElytronMessages.log.malformedPemContent(pemContent.getIndex());
                    });
                    if (this.next == null) {
                        return false;
                    }
                }
                return true;
            }

            @Override
            public PemEntry<?> next() {
                if (!this.hasNext()) {
                    throw new NoSuchElementException();
                }
                try {
                    PemEntry<?> pemEntry = this.next;
                    return pemEntry;
                }
                finally {
                    this.next = null;
                }
            }
        };
    }

    public static void generatePemContent(ByteStringBuilder target, String type, ByteIterator content) throws IllegalArgumentException {
        Assert.checkNotNullParam((String)"target", (Object)target);
        Assert.checkNotNullParam((String)"type", (Object)type);
        Assert.checkNotNullParam((String)"content", (Object)content);
        Matcher matcher = VALID_LABEL.matcher(type);
        if (matcher.find()) {
            throw ElytronMessages.log.invalidPemType("<any valid PEM type>", type);
        }
        target.append("-----BEGIN ").append(type).append("-----");
        target.append(content.base64Encode().drainToString(System.lineSeparator(), 64));
        target.append(System.lineSeparator()).append("-----END ").append(type).append("-----").append(System.lineSeparator());
    }

    public static byte[] extractDerContent(CodePointIterator pemContent) {
        return Pem.parsePemContent(pemContent, (type, byteIterator) -> byteIterator.drain());
    }

    private static X509Certificate parsePemX509CertificateContent(String type, ByteIterator byteIterator) throws IllegalArgumentException {
        if (!type.equals(CERTIFICATE_FORMAT)) {
            throw ElytronMessages.log.invalidPemType(CERTIFICATE_FORMAT, type);
        }
        try {
            CertificateFactory instance = CertificateFactory.getInstance("X.509");
            return (X509Certificate)instance.generateCertificate(byteIterator.asInputStream());
        }
        catch (CertificateException e) {
            throw ElytronMessages.log.certificateParseError(e);
        }
    }

    private static PublicKey parsePemPublicKey(String type, ByteIterator byteIterator) throws IllegalArgumentException {
        if (!type.equals(PUBLIC_KEY_FORMAT)) {
            throw ElytronMessages.log.invalidPemType(PUBLIC_KEY_FORMAT, type);
        }
        try {
            byte[] der = byteIterator.drain();
            DERDecoder derDecoder = new DERDecoder(der);
            derDecoder.startSequence();
            switch (derDecoder.peekType()) {
                case 48: {
                    derDecoder.startSequence();
                    String algorithm = derDecoder.decodeObjectIdentifierAsKeyAlgorithm();
                    if (algorithm != null) {
                        return KeyFactory.getInstance(algorithm).generatePublic(new X509EncodedKeySpec(der));
                    }
                    throw ElytronMessages.log.asnUnrecognisedAlgorithm(algorithm);
                }
            }
            throw ElytronMessages.log.asnUnexpectedTag();
        }
        catch (Exception cause) {
            throw ElytronMessages.log.publicKeyParseError(cause);
        }
    }

    private static PrivateKey parsePemPrivateKey(String type, ByteIterator byteIterator) throws IllegalArgumentException {
        if (!type.equals(PRIVATE_KEY_FORMAT)) {
            throw ElytronMessages.log.invalidPemType(PRIVATE_KEY_FORMAT, type);
        }
        try {
            byte[] der = byteIterator.drain();
            DERDecoder derDecoder = new DERDecoder(der);
            derDecoder.startSequence();
            if (derDecoder.peekType() != 2) {
                throw ElytronMessages.log.asnUnexpectedTag();
            }
            derDecoder.skipElement();
            derDecoder.startSequence();
            String algorithm = derDecoder.decodeObjectIdentifierAsKeyAlgorithm();
            if (algorithm != null) {
                return KeyFactory.getInstance(algorithm).generatePrivate(new PKCS8EncodedKeySpec(der));
            }
            throw ElytronMessages.log.asnUnrecognisedAlgorithm(algorithm);
        }
        catch (Exception cause) {
            throw ElytronMessages.log.privateKeyParseError(cause);
        }
    }

    private static KeyPair parseOpenSSHKeys(ByteIterator byteIterator, FilePasswordProvider passphraseProvider) throws IllegalArgumentException {
        OpenSSHKeyPairResourceParser resourceParser = new OpenSSHKeyPairResourceParser();
        byte[] stream = byteIterator.drain();
        try {
            return (KeyPair)resourceParser.extractKeyPairs(null, null, "BEGIN OPENSSH PRIVATE KEY", "END OPENSSH PRIVATE KEY", passphraseProvider, stream, null).iterator().next();
        }
        catch (IOException e) {
            throw ElytronMessages.log.openSshParseError(e.getMessage());
        }
        catch (GeneralSecurityException e) {
            throw ElytronMessages.log.openSshGeneratingError(e.getMessage());
        }
    }

    public static X509Certificate parsePemX509Certificate(CodePointIterator pemContent) throws IllegalArgumentException {
        Assert.checkNotNullParam((String)"pemContent", (Object)pemContent);
        return Pem.parsePemContent(pemContent, Pem::parsePemX509CertificateContent);
    }

    public static PublicKey parsePemPublicKey(CodePointIterator pemContent) throws IllegalArgumentException {
        Assert.checkNotNullParam((String)"pemContent", (Object)pemContent);
        return Pem.parsePemContent(pemContent, Pem::parsePemPublicKey);
    }

    public static void generatePemX509Certificate(ByteStringBuilder target, X509Certificate certificate) {
        Assert.checkNotNullParam((String)"target", (Object)target);
        Assert.checkNotNullParam((String)"certificate", (Object)certificate);
        try {
            Pem.generatePemContent(target, CERTIFICATE_FORMAT, ByteIterator.ofBytes((byte[])certificate.getEncoded()));
        }
        catch (CertificateEncodingException e) {
            throw ElytronMessages.log.certificateParseError(e);
        }
    }

    public static void generatePemPublicKey(ByteStringBuilder target, PublicKey publicKey) {
        Assert.checkNotNullParam((String)"target", (Object)target);
        Assert.checkNotNullParam((String)"publicKey", (Object)publicKey);
        try {
            KeyFactory instance = KeyFactory.getInstance(publicKey.getAlgorithm());
            X509EncodedKeySpec keySpec = instance.getKeySpec(publicKey, X509EncodedKeySpec.class);
            Pem.generatePemContent(target, PUBLIC_KEY_FORMAT, ByteIterator.ofBytes((byte[])keySpec.getEncoded()));
        }
        catch (Exception e) {
            throw ElytronMessages.log.publicKeyParseError(e);
        }
    }

    public static void generatePemPKCS10CertificateSigningRequest(ByteStringBuilder target, PKCS10CertificateSigningRequest certificateSigningRequest) {
        Assert.checkNotNullParam((String)"target", (Object)target);
        Assert.checkNotNullParam((String)"certificateSigningRequest", (Object)certificateSigningRequest);
        Pem.generatePemContent(target, CERTIFICATE_REQUEST_FORMAT, ByteIterator.ofBytes((byte[])certificateSigningRequest.getEncoded()));
    }
}

