/*
 * Copyright (c) 2016 WisePersist.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wisepersist.apuava.gwt.client.dom;

import com.google.common.collect.Sets;
import com.google.gwt.dom.client.Element;
import com.google.gwt.user.client.Event;
import com.google.gwt.user.client.EventListener;

import java.util.Set;

/**
 * Implementation of {@link NativeDom} interface.
 *
 * @author delight.wjk@gmail.com
 */
public class NativeDomImpl implements NativeDom {

  private Element element;
  private int eventBits;

  private final Set<NativeClickHandler> clickHandlers = Sets.newHashSet();
  private final Set<NativeKeyUpHandler> keyUpHandlers = Sets.newHashSet();
  private final Set<NativeKeyDownHandler> keyDownHandlers = Sets.newHashSet();
  private final Set<NativeFocusHandler> focusHandlers = Sets.newHashSet();
  private final Set<NativeMouseOutHandler> mouseOutHandlers = Sets.newHashSet();
  private final Set<NativeMouseOverHandler> mouseOverHandlers = Sets.newHashSet();
  private final Set<NativeDblClickHandler> dblClickHandlers = Sets.newHashSet();
  private final Set<NativeChangeHandler> changeHandlers = Sets.newHashSet();

  @Override
  public final NativeDom wrap(final Element theElement) {
    this.element = theElement;
    return this;
  }

  @Override
  public final NativeDomImpl addClickHandler(final NativeClickHandler clickHandler) {
    this.eventBits |= Event.ONCLICK;
    this.clickHandlers.add(clickHandler);
    bindEventListeners();
    return this;
  }

  @Override
  public final NativeDomImpl addKeyUpHandler(final NativeKeyUpHandler keyUpHandler) {
    this.eventBits |= Event.ONKEYUP;
    this.keyUpHandlers.add(keyUpHandler);
    bindEventListeners();
    return this;
  }

  @Override
  public final NativeDomImpl addKeyDownHandler(final NativeKeyDownHandler keyDownHandler) {
    this.eventBits |= Event.ONKEYDOWN;
    this.keyDownHandlers.add(keyDownHandler);
    bindEventListeners();
    return this;
  }

  @Override
  public final NativeDomImpl addFocusHandler(final NativeFocusHandler focusHandler) {
    this.eventBits |= Event.ONKEYDOWN;
    this.focusHandlers.add(focusHandler);
    bindEventListeners();
    return this;
  }

  @Override
  public final NativeDomImpl addMouseOutHandler(final NativeMouseOutHandler mouseOutHandler) {
    this.eventBits |= Event.ONMOUSEOUT;
    this.mouseOutHandlers.add(mouseOutHandler);
    bindEventListeners();
    return this;
  }

  @Override
  public final NativeDomImpl addMouseOverHandler(final NativeMouseOverHandler mouseOverHandler) {
    this.eventBits |= Event.ONMOUSEOVER;
    this.mouseOverHandlers.add(mouseOverHandler);
    bindEventListeners();
    return this;
  }

  @Override
  public final NativeDomImpl addDblClickHandler(final NativeDblClickHandler dblClickHandler) {
    this.eventBits |= Event.ONDBLCLICK;
    this.dblClickHandlers.add(dblClickHandler);
    bindEventListeners();
    return this;
  }

  @Override
  public final NativeDomImpl addChangeHandler(final NativeChangeHandler changeHandler) {
    this.eventBits |= Event.ONCHANGE;
    this.changeHandlers.add(changeHandler);
    bindEventListeners();
    return this;
  }

  /**
   * Binds event listeners to the element.
   */
  private void bindEventListeners() {
    Event.sinkEvents(element, eventBits);
    Event.setEventListener(element, new EventListener() {
      @Override
      public void onBrowserEvent(final Event event) {
        handleBrowserEvent(event);
      }
    });
  }

  /**
   * Handles the browser event specified.
   *
   * @param event The browser event specified.
   */
  @SuppressWarnings("MethodLength")
  private void handleBrowserEvent(final Event event) {
    switch (event.getTypeInt()) {
      case Event.ONCLICK:
        checkAndExecuteClickHandlers(event);
        break;
      case Event.ONKEYUP:
        checkAndExecuteKeyUpHandlers(event);
        break;
      case Event.ONKEYDOWN:
        checkAndExecuteKeyDownHandlers(event);
        break;
      case Event.ONFOCUS:
        checkAndExecuteFocusHandlers(event);
        break;
      case Event.ONMOUSEOUT:
        checkAndExecuteMouseOutHandlers(event);
        break;
      case Event.ONMOUSEOVER:
        checkAndExecuteMouseOverHandlers(event);
        break;
      case Event.ONDBLCLICK:
        checkAndExecuteDbClickHandlers(event);
        break;
      default:
        // Do nothing by default
    }
  }

  /**
   * Checks and executes double click handlers.
   *
   * @param event The browser event specified.
   */
  private void checkAndExecuteDbClickHandlers(final Event event) {
    if ((eventBits & Event.ONDBLCLICK) == Event.ONDBLCLICK) {
      for (final NativeDblClickHandler dblClickHandler : dblClickHandlers) {
        dblClickHandler.onDblClick(event);
      }
    }
  }

  /**
   * Checks and executes mouse over handlers.
   *
   * @param event The browser event specified.
   */
  private void checkAndExecuteMouseOverHandlers(final Event event) {
    if ((eventBits & Event.ONMOUSEOVER) == Event.ONMOUSEOVER) {
      for (final NativeMouseOverHandler mouseOverHandler : mouseOverHandlers) {
        mouseOverHandler.onMouseOver(event);
      }
    }
  }

  /**
   * Checks and executes mouse out handlers.
   *
   * @param event The browser event specified.
   */
  private void checkAndExecuteMouseOutHandlers(final Event event) {
    if ((eventBits & Event.ONMOUSEOUT) == Event.ONMOUSEOUT) {
      for (final NativeMouseOutHandler mouseOutHandler : mouseOutHandlers) {
        mouseOutHandler.onMouseOut(event);
      }
    }
  }

  /**
   * Checks and executes focus handlers.
   *
   * @param event The browser event specified.
   */
  private void checkAndExecuteFocusHandlers(final Event event) {
    if ((eventBits & Event.ONFOCUS) == Event.ONFOCUS) {
      for (final NativeFocusHandler focusHandler : focusHandlers) {
        focusHandler.onFocus(event);
      }
    }
  }

  /**
   * Checks and executes key down handlers.
   *
   * @param event The browser event specified.
   */
  private void checkAndExecuteKeyDownHandlers(final Event event) {
    if ((eventBits & Event.ONKEYDOWN) == Event.ONKEYDOWN) {
      for (final NativeKeyDownHandler keyDownHandler : keyDownHandlers) {
        keyDownHandler.onKeyDown(event);
      }
    }
  }

  /**
   * Checks and executes key up handlers.
   *
   * @param event The browser event specified.
   */
  private void checkAndExecuteKeyUpHandlers(final Event event) {
    if ((eventBits & Event.ONKEYUP) == Event.ONKEYUP) {
      for (final NativeKeyUpHandler keyUpHandler : keyUpHandlers) {
        keyUpHandler.onKeyUp(event);
      }
    }
  }

  /**
   * Checks and executes click handlers.
   *
   * @param event The browser event specified.
   */
  private void checkAndExecuteClickHandlers(final Event event) {
    if ((eventBits & Event.ONCLICK) == Event.ONCLICK) {
      for (final NativeClickHandler clickHandler : clickHandlers) {
        clickHandler.onClick(event);
      }
    }
  }
}
