/*
 * Copyright (c) 2016 WisePersist.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wisepersist.apuava.gwt.client.ui;

import com.google.gwt.core.client.Scheduler;
import com.google.gwt.dom.client.StyleInjector;
import com.google.gwt.user.client.Event;
import com.google.gwt.user.client.Timer;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.HTML;
import com.google.gwt.user.client.ui.PopupPanel;
import com.google.inject.Singleton;

import org.wisepersist.apuava.gwt.client.dom.NativeClickHandler;
import org.wisepersist.apuava.gwt.client.dom.NativeDomWrapper;
import org.wisepersist.apuava.gwt.client.dom.NativeDomWrapperImpl;

/**
 * Implementation of {@link Notifier} interface.
 *
 * @author jake.wang@practiceinsight.io
 */
@Singleton
public class NotifierImpl extends PopupPanel implements Notifier {

  private static final int AUTO_CLOSE_DELAY = 5;

  private final NativeDomWrapper domWrapper;
  private final HTML html;
  private final Timer timer;

  /**
   * Constructs a new instance of {@link NotifierImpl} class.
   */
  public NotifierImpl() {
    super(false);
    this.domWrapper = new NativeDomWrapperImpl();
    this.html = new HTML();
    this.timer = new Timer() {
      @Override
      public void run() {
        NotifierImpl.this.hide();
      }
    };

    setWidget(html); //NOPMD

    StyleInjector.inject(NotifierStyle.INSTANCE.css().getText()); //NOPMD
    getElement().addClassName("notifier"); //NOPMD
  }

  @Override
  public final void info(final String htmlMessage) {
    getElement().getFirstChildElement().setClassName("popupContent info");
    message(htmlMessage, NotifierImpl.AUTO_CLOSE_DELAY, true);
  }

  @Override
  public final void success(final String htmlMessage) {
    getElement().getFirstChildElement().setClassName("popupContent success");
    message(htmlMessage, NotifierImpl.AUTO_CLOSE_DELAY, true);
  }

  @Override
  public final void progress(final String htmlMessage) {
    getElement().getFirstChildElement().setClassName("popupContent progress");
    message(htmlMessage, -1, false);
  }

  @Override
  public final void warning(final String htmlMessage) {
    getElement().getFirstChildElement().setClassName("popupContent warning");
    message(htmlMessage, -1, true);
  }

  @Override
  public final void error(final String htmlMessage) {
    getElement().getFirstChildElement().setClassName("popupContent error");
    message(htmlMessage, -1, true);
  }

  /**
   * Shows notification message in a popup.
   *
   * @param htmlMessage The HTML message to be shown in the message popup.
   * @param autoCloseDelay The delay seconds specified to auto close, zero or negative value
   *     disables auto-close.
   * @param closeOnClick Whether we should close the message when user click it.
   */
  private void message(final String htmlMessage, final int autoCloseDelay,
                       final boolean closeOnClick) {
    html.setHTML(htmlMessage);
    setPopupPositionAndShow(new PositionCallback() {
      @Override
      public void setPosition(final int offsetWidth, final int offsetHeight) {
        final int left = (Window.getClientWidth() - offsetWidth) / 2;
        setPopupPosition(left, 0);
      }
    });
    bindClickHandler(closeOnClick);
    scheduleAutoClose(autoCloseDelay);
  }

  @Override
  public final void close() {
    timer.cancel();
    hide();
  }

  /**
   * Schedules auto close with delay in seconds specified.
   *
   * @param autoCloseDelay The delay of auto close in seconds specified.
   */
  private void scheduleAutoClose(final int autoCloseDelay) {
    if (autoCloseDelay > 0) {
      final int millis = 1000;
      timer.schedule(autoCloseDelay * millis);
    }
  }

  /**
   * Binds click handler to notification popup.
   *
   * @param closeOnClick Whether we should enable the click handler for the notifier.
   */
  private void bindClickHandler(final boolean closeOnClick) {
    Scheduler.get().scheduleDeferred(new Scheduler.ScheduledCommand() {
      @Override
      public void execute() {
        domWrapper.wrap(getElement()).addClickHandler(new NativeClickHandler() {
          @Override
          public void onClick(final Event event) {
            if (closeOnClick) {
              NotifierImpl.this.hide();
            }
          }
        });
      }
    });
  }
}
