/*
 * Copyright (c) 2017 WisePersist.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wisepersist.apuava.cli;

import static org.fusesource.jansi.Ansi.Color.CYAN;
import static org.fusesource.jansi.Ansi.Color.GREEN;
import static org.fusesource.jansi.Ansi.Color.RED;
import static org.fusesource.jansi.Ansi.Color.YELLOW;
import static org.fusesource.jansi.Ansi.ansi;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.SystemUtils;
import org.fusesource.jansi.Ansi;
import org.fusesource.jansi.AnsiConsole;

/**
 * Logging in terminal console. This class should be used in the main thread only, because in
 * multi-threads on Windows, the previous line cannot be cleared correctly sometimes, and you
 * would see format errors. If we keep it all in the main thread, no such issues.
 *
 * @author delight.wjk@gmail.com
 */
@SuppressWarnings("PMD")
public final class Console {

  private static int currLineLength;

  static {
    AnsiConsole.systemInstall();
  }

  /**
   * Private constructor.
   */
  private Console() {
    // Do nothing
  }

  /**
   * Logs done message in console.
   *
   * @param message The message specified.
   */
  public static void done(final String message) {
    done(message, StringUtils.EMPTY);
  }

  /**
   * Logs done message in console.
   *
   * @param message The message specified.
   * @param highlight The highlight word specified.
   */
  public static void done(final String message, final String highlight) {
    clearCurrLine();
    System.out.println(createAnsi("Done", message, highlight));
  }

  /**
   * Logs progress message in console.
   *
   * @param label The label specified.
   * @param message The message specified.
   */
  public static void progress(final String label, final String message) {
    progress(label, message, StringUtils.EMPTY);
  }

  /**
   * Logs progress message in console.
   *
   * @param label The label specified.
   * @param message The message specified.
   * @param highlight The highlight word specified.
   */
  public static void progress(final String label, final String message, final String highlight) {
    clearCurrLine();
    System.out.print(createAnsi(label, message, highlight));
    System.out.flush();
  }

  /**
   * Logs error message in console.
   *
   * @param message The message specified.
   */
  public static void error(final String message) {
    clearCurrLine();
    System.out.println(createAnsi("Error", message, StringUtils.EMPTY));
  }

  /**
   * Creates a {@link Ansi} object for logging.
   *
   * @param label The label specified.
   * @param message The message specified.
   * @param highlight The highlight specified.
   * @return The {@link Ansi} object specified.
   */
  private static Ansi createAnsi(final String label, final String message, final String highlight) {
    Ansi.Color labelColor = CYAN;
    if (StringUtils.containsIgnoreCase(label, "Done")) {
      labelColor = GREEN;
    } else if (StringUtils.containsIgnoreCase(label, "Error")) {
      labelColor = RED;
    }
    Ansi ansi = ansi().fg(labelColor).a("[" + label + "] ").reset().a(message);
    if (StringUtils.isNotBlank(highlight)) {
      ansi = ansi.a(" ").fg(YELLOW).a(highlight).reset();
    }
    saveLineLength(label, message, highlight);
    return ansi;
  }

  /**
   * Saves the current line length for next time clearing the line on Windows.
   *
   * @param label The label specified.
   * @param message The message specified.
   * @param highlight The highlight specified.
   */
  private static void saveLineLength(
      final String label, final String message, final String highlight) {
    final String line = "[" + label + "] " + message + " " + StringUtils.trimToEmpty(highlight);
    currLineLength = line.length();
  }

  /**
   * Clears the current line.
   */
  private static void clearCurrLine() {
    if (SystemUtils.IS_OS_WINDOWS) {
      System.out.print("\r"); // Move cursor to the beginning of the line
      for (int i = 0; i < currLineLength; i += 1) {
        System.out.print(" ");
      }
    } else {
      System.out.print("\033[2K");
    }
    System.out.print("\r");
    System.out.flush();
  }
}
