/*
 * Copyright (c) 2017 WisePersist.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wisepersist.pinpayments;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.TypeAdapterFactory;
import com.google.inject.AbstractModule;
import java.util.ServiceLoader;
import org.wisepersist.pinpayments.annotation.PinBaseUrl;
import org.wisepersist.pinpayments.annotation.PinSecretKey;
import org.wisepersist.pinpayments.annotation.PinVersion;

/**
 * Main Guice module for Pin Payments API.
 *
 * @author delight.wjk@gmail.com
 */
public class PinPaymentsModule extends AbstractModule {

  private static final String PIN_BASEURL_TEST = "https://test-api.pin.net.au";
  private static final String PIN_BASEURL_PROD = "https://api.pin.net.au";
  private static final String PIN_VERSION = "1";

  private final Env env;
  private final String pinSecretKey;

  public PinPaymentsModule(final Env env, final String pinSecretKey) {
    super();
    this.env = env;
    this.pinSecretKey = pinSecretKey;
  }

  @Override
  protected final void configure() {
    bind(Gson.class).toInstance(createGson());
    if (env == Env.Prod) {
      bind(String.class).annotatedWith(PinBaseUrl.class).toInstance(PIN_BASEURL_PROD);
    } else {
      bind(String.class).annotatedWith(PinBaseUrl.class).toInstance(PIN_BASEURL_TEST);
    }
    bind(String.class).annotatedWith(PinVersion.class).toInstance(PIN_VERSION);
    bind(String.class).annotatedWith(PinSecretKey.class).toInstance(pinSecretKey);
  }

  private Gson createGson() {
    final GsonBuilder gsonBuilder = new GsonBuilder().serializeNulls();
    for (final TypeAdapterFactory factory : ServiceLoader.load(TypeAdapterFactory.class)) {
      gsonBuilder.registerTypeAdapterFactory(factory);
    }
    return gsonBuilder.create();
  }

  /**
   * @author delight.wjk@gmail.com
   */
  public enum Env {
    Test,
    Prod
  }
}
