/*
 * Copyright (c) 2017 WisePersist.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wisepersist.pinpayments.api.impl;

import com.google.common.base.Charsets;
import com.google.common.io.BaseEncoding;
import com.google.gson.Gson;
import com.google.inject.Inject;
import java.io.IOException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.wisepersist.apuava.http.HttpMethod;
import org.wisepersist.apuava.http.HttpResponse;
import org.wisepersist.apuava.http.HttpStatus;
import org.wisepersist.apuava.http.HttpTransport;
import org.wisepersist.pinpayments.ParamsHelper;
import org.wisepersist.pinpayments.annotation.PinBaseUrl;
import org.wisepersist.pinpayments.annotation.PinSecretKey;
import org.wisepersist.pinpayments.annotation.PinVersion;
import org.wisepersist.pinpayments.pojo.DeletionResult;
import org.wisepersist.pinpayments.pojo.ErrorResponse;
import org.wisepersist.pinpayments.pojo.ImmutableDeletionResult;
import org.wisepersist.pinpayments.pojo.ImmutableErrorResponse;
import org.wisepersist.pinpayments.pojo.SearchCriteria;

/**
 * @author delight.wjk@gmail.com
 */
public abstract class AbstractBaseApi {

  private static final Logger log = LoggerFactory.getLogger(AbstractBaseApi.class); // NOPMD

  @Inject
  @PinBaseUrl
  private String pinBaseUrl;
  @Inject
  @PinVersion
  private String pinVersion;
  @Inject
  @PinSecretKey
  private String pinSecretKey;
  @Inject
  private Gson gson;
  @Inject
  private ParamsHelper paramsHelper;

  protected final String getApiUrl() {
    return String.format("%s/%s", getApiBase(), getApiName());
  }

  protected final String getApiBase() {
    return String.format("%s/%s", pinBaseUrl, pinVersion);
  }

  protected abstract String getApiName();

  protected final HttpTransport httpTransportForPost() {
    return httpTransport()
        .addHeader("Content-Type", "application/x-www-form-urlencoded")
        .setHttpMethod(HttpMethod.POST);
  }

  protected final HttpTransport httpTransportForPut() {
    return httpTransport()
        .addHeader("Content-Type", "application/x-www-form-urlencoded")
        .setHttpMethod(HttpMethod.PUT);
  }

  protected final HttpTransport httpTransportForGet() {
    return httpTransport().setHttpMethod(HttpMethod.GET);
  }

  protected final HttpTransport httpTransportForDelete() {
    return httpTransport().setHttpMethod(HttpMethod.DELETE);
  }

  private HttpTransport httpTransport() {
    return new HttpTransport()
        .addHeader("Authorization", String.format("Basic %s:", getEncodedSecretKey()));
  }

  private String getEncodedSecretKey() {
    return BaseEncoding.base64().encode(pinSecretKey.getBytes(Charsets.UTF_8));
  }

  protected final DeletionResult getDeletionResult(final HttpResponse httpResponse) {
    final DeletionResult result;
    if (httpResponse.getStatusCode() == HttpStatus.NO_CONTENT.getCode()) {
      result = ImmutableDeletionResult.builder().success(true).build();
    } else {
      final ErrorResponse errorResponse = gson.fromJson(
          httpResponse.getContent(), ImmutableErrorResponse.class);
      result = ImmutableDeletionResult.builder().success(false)
          .errorResponse(errorResponse)
          .build();
    }
    return result;
  }

  protected final <T> T executeList(final String url, final Class<? extends T> clazz)
      throws IOException {
    final HttpResponse httpResponse = httpTransportForGet().setUrl(url).execute();
    log.debug("httpResponse: " + httpResponse);
    return gson.fromJson(httpResponse.getContent(), clazz);
  }

  protected final <T> T executeSearch(final String url, final SearchCriteria searchCriteria,
      final Class<? extends T> clazz)
      throws IOException {
    final HttpResponse httpResponse = httpTransportForGet().setUrl(url)
        .addParams(paramsHelper.toFlatParams(searchCriteria)).execute();
    log.debug("httpResponse: " + httpResponse);
    return gson.fromJson(httpResponse.getContent(), clazz);
  }
}
