/*
 * Decompiled with CFR 0.152.
 */
package org.wso2.carbon.uiserver.internal.io.http;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.BasicFileAttributes;
import java.time.Duration;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.commons.io.FilenameUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.wso2.carbon.uiserver.api.App;
import org.wso2.carbon.uiserver.api.Extension;
import org.wso2.carbon.uiserver.api.Theme;
import org.wso2.carbon.uiserver.api.http.HttpRequest;
import org.wso2.carbon.uiserver.api.http.HttpResponse;
import org.wso2.carbon.uiserver.internal.exception.BadRequestException;
import org.wso2.carbon.uiserver.internal.exception.FileOperationException;
import org.wso2.carbon.uiserver.internal.exception.ResourceNotFoundException;
import org.wso2.carbon.uiserver.internal.http.ResponseBuilder;
import org.wso2.carbon.uiserver.internal.io.util.MimeMapper;
import org.wso2.carbon.uiserver.internal.io.util.PathUtils;

public class StaticRequestDispatcher {
    private static final DateTimeFormatter HTTP_DATE_FORMATTER;
    private static final ZoneId GMT_TIME_ZONE;
    private static final Logger LOGGER;
    private final App app;
    private final Map<Path, ZonedDateTime> resourcesLastModifiedDates;

    public StaticRequestDispatcher(App app) {
        this.app = app;
        this.resourcesLastModifiedDates = new ConcurrentHashMap<Path, ZonedDateTime>();
    }

    public HttpResponse serveDefaultFavicon(HttpRequest request) {
        InputStream inputStream = this.getClass().getClassLoader().getResourceAsStream("default-favicon.png");
        if (inputStream == null) {
            LOGGER.error("Cannot find default favicon 'default-favicon.png' in classpath.");
            return ResponseBuilder.notFound("Cannot find default favicon").build();
        }
        return ResponseBuilder.ok(inputStream, "image/png").build();
    }

    public HttpResponse serve(HttpRequest request) {
        try {
            Path resourcePath = this.resolveResource(request);
            ZonedDateTime lastModifiedDate = this.getLastModifiedDate(resourcePath);
            if (lastModifiedDate == null) {
                return ResponseBuilder.ok(resourcePath, StaticRequestDispatcher.getContentType(request, resourcePath)).headers(this.app.getConfiguration().getResponseHeaders().forStaticResources()).build();
            }
            ZonedDateTime ifModifiedSinceDate = StaticRequestDispatcher.getIfModifiedSinceDate(request);
            if (ifModifiedSinceDate != null && Duration.between(ifModifiedSinceDate, lastModifiedDate).isZero()) {
                return ResponseBuilder.status(304).build();
            }
            return ResponseBuilder.ok(resourcePath, StaticRequestDispatcher.getContentType(request, resourcePath)).header("Last-Modified", HTTP_DATE_FORMATTER.format(lastModifiedDate)).header("Cache-Control", "public,max-age=2592000").headers(this.app.getConfiguration().getResponseHeaders().forStaticResources()).build();
        }
        catch (BadRequestException e) {
            return ResponseBuilder.badRequest("Static resource URI '" + request.getUri() + "' is invalid.").build();
        }
        catch (ResourceNotFoundException e) {
            return ResponseBuilder.notFound("Requested resource '" + request.getUri() + "' does not exists.").build();
        }
        catch (FileOperationException e) {
            LOGGER.error("An error occurred when manipulating paths for static resource request '{}'.", (Object)request, (Object)e);
            return ResponseBuilder.serverError("A server error occurred while serving for static resource request.").build();
        }
    }

    private ZonedDateTime getLastModifiedDate(Path resourcePath) {
        return this.resourcesLastModifiedDates.computeIfAbsent(resourcePath, StaticRequestDispatcher::readLastModifiedDate);
    }

    private Path resolveResource(HttpRequest request) throws BadRequestException {
        if (request.isAppStaticResourceRequest()) {
            return StaticRequestDispatcher.resolveResourceInApp(this.app, request.getUriWithoutContextPath());
        }
        if (request.isExtensionStaticResourceRequest()) {
            return StaticRequestDispatcher.resolveResourceInExtension(this.app, request.getUriWithoutContextPath());
        }
        if (request.isThemeStaticResourceRequest()) {
            return StaticRequestDispatcher.resolveResourceInTheme(this.app, request.getUriWithoutContextPath());
        }
        throw new BadRequestException("Invalid static resource URI '" + request.getUri() + "'.");
    }

    private static Path resolveResourceInApp(App app, String uriWithoutContextPath) throws BadRequestException {
        int slashesCount = 0;
        int thirdSlashIndex = -1;
        for (int i = 0; i < uriWithoutContextPath.length(); ++i) {
            if (uriWithoutContextPath.charAt(i) != '/') continue;
            if (++slashesCount == 3) {
                thirdSlashIndex = i;
                continue;
            }
            if (slashesCount == 4) break;
        }
        if (slashesCount != 4) {
            throw new BadRequestException("Invalid static resource URI '" + uriWithoutContextPath + "'.");
        }
        String relativeFilePath = uriWithoutContextPath.substring(thirdSlashIndex + 1, uriWithoutContextPath.length());
        return StaticRequestDispatcher.selectPath(app.getPaths(), "public", relativeFilePath);
    }

    private static Path resolveResourceInExtension(App app, String uriWithoutContextPath) throws BadRequestException, ResourceNotFoundException {
        int slashesCount = 0;
        int thirdSlashIndex = -1;
        int fourthSlashIndex = -1;
        int fifthSlashIndex = -1;
        for (int i = 0; i < uriWithoutContextPath.length(); ++i) {
            if (uriWithoutContextPath.charAt(i) != '/') continue;
            if (++slashesCount == 3) {
                thirdSlashIndex = i;
                continue;
            }
            if (slashesCount == 4) {
                fourthSlashIndex = i;
                continue;
            }
            if (slashesCount != 5) continue;
            fifthSlashIndex = i;
            break;
        }
        if (slashesCount != 5) {
            throw new BadRequestException("Invalid static resource URI '" + uriWithoutContextPath + "'.");
        }
        String extensionType = uriWithoutContextPath.substring(thirdSlashIndex + 1, fourthSlashIndex);
        String extensionName = uriWithoutContextPath.substring(fourthSlashIndex + 1, fifthSlashIndex);
        Extension extension = app.getExtension(extensionType, extensionName).orElseThrow(() -> new ResourceNotFoundException("Extension '" + extensionType + ":" + extensionName + "' found in URI '" + uriWithoutContextPath + "' does not exists."));
        String relativeFilePath = uriWithoutContextPath.substring(fifthSlashIndex + 1, uriWithoutContextPath.length());
        return StaticRequestDispatcher.selectPath(extension.getPaths(), relativeFilePath);
    }

    private static Path resolveResourceInTheme(App app, String uriWithoutContextPath) throws BadRequestException, ResourceNotFoundException {
        int slashesCount = 0;
        int thirdSlashIndex = -1;
        int fourthSlashIndex = -1;
        for (int i = 0; i < uriWithoutContextPath.length(); ++i) {
            if (uriWithoutContextPath.charAt(i) != '/') continue;
            if (++slashesCount == 3) {
                thirdSlashIndex = i;
                continue;
            }
            if (slashesCount == 4) {
                fourthSlashIndex = i;
                continue;
            }
            if (slashesCount == 5) break;
        }
        if (slashesCount != 5) {
            throw new BadRequestException("Invalid static resource URI '" + uriWithoutContextPath + "'.");
        }
        String themeName = uriWithoutContextPath.substring(thirdSlashIndex + 1, fourthSlashIndex);
        Theme theme = app.getTheme(themeName).orElseThrow(() -> new ResourceNotFoundException("Theme '" + themeName + "' found in URI '" + uriWithoutContextPath + "' does not exists."));
        String relativeFilePath = uriWithoutContextPath.substring(fourthSlashIndex + 1, uriWithoutContextPath.length());
        return StaticRequestDispatcher.selectPath(theme.getPaths(), relativeFilePath);
    }

    private static ZonedDateTime readLastModifiedDate(Path resourcePath) throws ResourceNotFoundException, FileOperationException {
        BasicFileAttributes fileAttributes;
        if (!Files.isReadable(resourcePath)) {
            throw new ResourceNotFoundException("Static resource file '" + resourcePath + "' is not readable.");
        }
        try {
            fileAttributes = Files.readAttributes(resourcePath, BasicFileAttributes.class, new LinkOption[0]);
        }
        catch (FileNotFoundException | NoSuchFileException e) {
            throw new ResourceNotFoundException("Static resource file '" + resourcePath + "' does not exists.", e);
        }
        catch (IOException | UnsupportedOperationException e) {
            throw new FileOperationException("Cannot read file attributes from static resource file '" + resourcePath + "'.", e);
        }
        if (fileAttributes.isRegularFile()) {
            return ZonedDateTime.ofInstant(fileAttributes.lastModifiedTime().toInstant(), GMT_TIME_ZONE);
        }
        throw new ResourceNotFoundException("Static resource file '" + resourcePath + "' is not a regular file.");
    }

    private static ZonedDateTime getIfModifiedSinceDate(HttpRequest request) {
        String ifModifiedSinceHeader = request.getHeaders().get("If-Modified-Since");
        if (ifModifiedSinceHeader == null) {
            return null;
        }
        try {
            return ZonedDateTime.parse(ifModifiedSinceHeader, HTTP_DATE_FORMATTER);
        }
        catch (DateTimeParseException e) {
            LOGGER.debug("Cannot parse 'If-Modified-Since' HTTP header value '{}'.", (Object)ifModifiedSinceHeader, (Object)e);
            return null;
        }
    }

    private static String getContentType(HttpRequest request, Path resource) {
        String extensionFromUri = FilenameUtils.getExtension((String)request.getUriWithoutContextPath());
        Optional<String> contentType = MimeMapper.getMimeType(extensionFromUri);
        if (contentType.isPresent()) {
            return contentType.get();
        }
        String extensionFromPath = PathUtils.getExtension(resource);
        return MimeMapper.getMimeType(extensionFromPath).orElse("*/*");
    }

    private static Path selectPath(List<String> parentDirectories, String ... relativeFilePath) {
        Path path = null;
        for (String parentDirectory : parentDirectories) {
            path = Paths.get(parentDirectory, relativeFilePath);
            if (!Files.exists(path, new LinkOption[0])) continue;
            return path;
        }
        return path;
    }

    static {
        LOGGER = LoggerFactory.getLogger(StaticRequestDispatcher.class);
        HTTP_DATE_FORMATTER = DateTimeFormatter.ofPattern("EEE, dd MMM yyyy HH:mm:ss zzz");
        GMT_TIME_ZONE = ZoneId.of("GMT");
    }
}

