/*
 * Decompiled with CFR 0.152.
 */
package org.wso2.carbon.launcher.extensions;

import java.io.IOException;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.jar.JarFile;
import java.util.jar.Manifest;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.wso2.carbon.launcher.extensions.model.BundleInfo;
import org.wso2.carbon.launcher.extensions.model.BundleInstallStatus;
import org.wso2.carbon.launcher.extensions.model.BundleLocation;

public class OSGiLibBundleDeployerUtils {
    private static final Logger logger = Logger.getLogger(OSGiLibBundleDeployerUtils.class.getName());

    public static synchronized void updateOSGiLib(String carbonHome, String carbonProfile, List<BundleInfo> newBundlesInfo) throws IOException {
        if (carbonHome == null || carbonHome.isEmpty()) {
            throw new IllegalArgumentException("Carbon home specified is invalid");
        }
        if (carbonProfile == null || carbonProfile.isEmpty()) {
            throw new IllegalArgumentException("Carbon Runtime specified is invalid");
        }
        if (newBundlesInfo == null) {
            throw new IllegalArgumentException("No new OSGi bundle information specified, for updating the Carbon Runtime: " + carbonProfile);
        }
        Path bundlesInfoFile = Paths.get(carbonHome, "wso2", carbonProfile, "configuration", "org.eclipse.equinox.simpleconfigurator", "bundles.info");
        Map<BundleLocation, List<BundleInfo>> existingBundlesInfo = Files.readAllLines(bundlesInfoFile).stream().filter(line -> !line.startsWith("#")).map(BundleInfo::getInstance).collect(Collectors.groupingBy(BundleInfo::isFromOSGiLib));
        Map<BundleInstallStatus, List<BundleInfo>> updatableBundles = OSGiLibBundleDeployerUtils.getUpdatableBundles(newBundlesInfo, existingBundlesInfo.get((Object)BundleLocation.OSGI_LIB_BUNDLE));
        if (updatableBundles.get((Object)BundleInstallStatus.TO_BE_INSTALLED).size() > 0 || updatableBundles.get((Object)BundleInstallStatus.TO_BE_REMOVED).size() > 0) {
            logger.log(Level.FINE, OSGiLibBundleDeployerUtils.getBundleInstallationSummary(updatableBundles));
            List<BundleInfo> effectiveNewBundleInfo = OSGiLibBundleDeployerUtils.mergeOSGiLibWithExistingBundlesInfo(newBundlesInfo, existingBundlesInfo);
            OSGiLibBundleDeployerUtils.updateBundlesInfoFile(effectiveNewBundleInfo, bundlesInfoFile);
            logger.log(Level.INFO, "Successfully updated the OSGi bundle information of Carbon Runtime: " + carbonProfile);
        } else {
            logger.log(Level.FINE, String.format("No changes detected in the %s directory in comparison with the profile, skipped the OSGi bundle information update for Carbon Runtime: %s", "lib", carbonProfile));
        }
    }

    public static List<BundleInfo> getBundlesInfo(Path sourceDirectory) throws IOException {
        if (sourceDirectory == null || !Files.exists(sourceDirectory, new LinkOption[0])) {
            throw new IOException("Invalid OSGi bundle source directory. The specified path may not exist or user may not have required file permissions for the specified path: " + sourceDirectory);
        }
        return ((Stream)Files.list(sourceDirectory).parallel()).map(child -> {
            BundleInfo bundleInfo = null;
            try {
                bundleInfo = OSGiLibBundleDeployerUtils.getBundleInfo(child).orElse(null);
            }
            catch (IOException e) {
                logger.log(Level.WARNING, "Error when loading the OSGi bundle information from " + child, e);
            }
            return bundleInfo;
        }).distinct().filter(Objects::nonNull).collect(Collectors.toList());
    }

    private static Optional<BundleInfo> getBundleInfo(Path bundlePath) throws IOException {
        if (bundlePath == null || !Files.exists(bundlePath, new LinkOption[0])) {
            throw new IOException("Invalid OSGi bundle path. The specified path may not exist or user may not have required file permissions for the specified path");
        }
        Path bundleFileName = bundlePath.getFileName();
        if (bundleFileName == null) {
            throw new IOException("Specified OSGi bundle file name is null: " + bundlePath);
        }
        String fileName = bundleFileName.toString();
        if (!fileName.endsWith(".jar")) {
            return Optional.empty();
        }
        try (JarFile jarFile = new JarFile(bundlePath.toString());){
            Manifest manifest = jarFile.getManifest();
            if (manifest == null || manifest.getMainAttributes() == null) {
                throw new IOException("Invalid OSGi bundle found in the lib folder");
            }
            String bundleSymbolicName = manifest.getMainAttributes().getValue("Bundle-SymbolicName");
            String bundleVersion = manifest.getMainAttributes().getValue("Bundle-Version");
            if (bundleSymbolicName == null || bundleVersion == null) {
                throw new IOException("Required bundle manifest headers do not exist");
            }
            logger.log(Level.FINE, "Loading information from OSGi bundle: " + bundleSymbolicName + ":" + bundleVersion + "...");
            if (bundleSymbolicName.contains(";")) {
                bundleSymbolicName = bundleSymbolicName.split(";")[0];
            }
            boolean isFragment = manifest.getMainAttributes().getValue("Fragment-Host") != null;
            int defaultBundleStartLevel = 4;
            BundleInfo generated = new BundleInfo(bundleSymbolicName, bundleVersion, "../../lib/" + fileName, defaultBundleStartLevel, isFragment);
            logger.log(Level.FINE, "Successfully loaded information from OSGi bundle: " + bundleSymbolicName + ":" + bundleVersion);
            Optional<BundleInfo> optional = Optional.of(generated);
            return optional;
        }
    }

    private static Map<BundleInstallStatus, List<BundleInfo>> getUpdatableBundles(List<BundleInfo> newBundlesInfo, List<BundleInfo> existingBundleInfo) {
        HashMap<BundleInstallStatus, List<BundleInfo>> updatableBundles = new HashMap<BundleInstallStatus, List<BundleInfo>>();
        List bundlesToBeInstalled = ((List)Optional.ofNullable(newBundlesInfo).orElse(new ArrayList())).stream().filter(bundleInfo -> !((List)Optional.ofNullable(existingBundleInfo).orElse(new ArrayList())).contains(bundleInfo)).collect(Collectors.toList());
        List bundlesToBeRemoved = ((List)Optional.ofNullable(existingBundleInfo).orElse(new ArrayList())).stream().filter(bundleInfo -> !((List)Optional.ofNullable(newBundlesInfo).orElse(new ArrayList())).contains(bundleInfo)).collect(Collectors.toList());
        updatableBundles.put(BundleInstallStatus.TO_BE_INSTALLED, bundlesToBeInstalled);
        updatableBundles.put(BundleInstallStatus.TO_BE_REMOVED, bundlesToBeRemoved);
        return updatableBundles;
    }

    private static List<BundleInfo> mergeOSGiLibWithExistingBundlesInfo(List<BundleInfo> newBundlesInfo, Map<BundleLocation, List<BundleInfo>> existingBundleInfo) {
        List<BundleInfo> effectiveBundlesInfo = existingBundleInfo.get((Object)BundleLocation.NON_OSGI_LIB_BUNDLE);
        if (effectiveBundlesInfo != null) {
            effectiveBundlesInfo.addAll(newBundlesInfo);
        } else {
            effectiveBundlesInfo = newBundlesInfo;
        }
        return effectiveBundlesInfo.stream().distinct().collect(Collectors.toList());
    }

    private static void updateBundlesInfoFile(List<BundleInfo> info, Path bundlesInfoFilePath) throws IOException {
        if (bundlesInfoFilePath != null && Files.exists(bundlesInfoFilePath, new LinkOption[0])) {
            if (info != null) {
                ArrayList bundleInfoLines = new ArrayList();
                info.stream().forEach(information -> bundleInfoLines.add(information.toString()));
                Path parent = bundlesInfoFilePath.getParent();
                if (parent != null) {
                    Path newBundlesInfoFile = Paths.get(parent.toString(), "new.info");
                    Files.write(newBundlesInfoFile, bundleInfoLines, new OpenOption[0]);
                    Files.deleteIfExists(bundlesInfoFilePath);
                    Files.move(newBundlesInfoFile, newBundlesInfoFile.resolveSibling("bundles.info"), new CopyOption[0]);
                }
            }
        } else {
            throw new IOException("Invalid file path. The specified path may not exist or user may not have required file permissions for the specified path: " + bundlesInfoFilePath);
        }
    }

    public static List<String> getCarbonProfiles(String carbonHome) throws IOException {
        Path carbonProfilesHome = Paths.get(carbonHome, "wso2");
        Path osgiRepoPath = Paths.get(carbonHome, "wso2", "lib");
        Stream<Path> profiles = Files.list(carbonProfilesHome);
        ArrayList<String> profileNames = new ArrayList<String>();
        ((Stream)profiles.parallel()).filter(profile -> !osgiRepoPath.equals(profile)).forEach(profile -> Optional.ofNullable(profile.getFileName()).ifPresent(name -> profileNames.add(name.toString())));
        if (profileNames.size() == 0) {
            throw new IOException("No profiles found in " + carbonHome + "/" + "wso2");
        }
        return profileNames;
    }

    private static String getBundleInstallationSummary(Map<BundleInstallStatus, List<BundleInfo>> updatableBundleInfo) {
        StringBuilder message = new StringBuilder("\nInstallation/Removal Summary of OSGi bundles from OSGi-lib\n");
        Optional.ofNullable(updatableBundleInfo.get((Object)BundleInstallStatus.TO_BE_INSTALLED)).ifPresent(list -> {
            if (list.size() > 0) {
                message.append("\nOSGi bundles to be newly installed:\n");
                list.stream().forEach(bundleInfo -> message.append("Symbolic-name: ").append(bundleInfo.getBundleSymbolicName()).append(" --> ").append("Version: ").append(bundleInfo.getBundleVersion()).append("\n"));
            }
        });
        Optional.ofNullable(updatableBundleInfo.get((Object)BundleInstallStatus.TO_BE_REMOVED)).ifPresent(list -> {
            if (list.size() > 0) {
                message.append("\nOSGi bundles to be removed:\n");
                list.stream().forEach(bundleInfo -> message.append("Symbolic-name: ").append(bundleInfo.getBundleSymbolicName()).append(" --> ").append("Version: ").append(bundleInfo.getBundleVersion()).append("\n"));
            }
        });
        return message.toString();
    }
}

