/*
 * Copyright (c) 2024, WSO2 LLC. (http://www.wso2.org) All Rights Reserved.
 *
 * WSO2 Inc. licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.wso2.maven;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;

/**
 * Generates the connector.xml file for the connector project.
 */
public class ConnectorXmlGenerator {

    /**
     * Generates the connector.xml file.
     *
     * @param connectorName    The name of the connector
     * @param connectorPackage The package of the connector
     */
    public static void generateConnectorXml(String connectorName, String connectorPackage,
                                            ConnectorMojo connectorMojo) {

        // Specify the output file name
        String outputFileName = Constants.DEFAULT_TARGET_FOLDER + File.separator + Constants.CLASSES +
                File.separator + Constants.CONNECTOR_XML;

        // if file already exists, skip the generation
        File outputFile = new File(outputFileName);
        if (outputFile.exists()) {
            connectorMojo.getLog().info("connector.xml file already exists. Skipping generation.");
            return;
        }

        try (FileWriter writer = new FileWriter(outputFileName)) {
            // Write the license information and autogenerated comment
            writer.write("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n");

            writer.write("<!-- This file is autogenerated. Do not modify manually. -->\n\n");

            // Start the <connector> element
            writer.write("<connector>\n");
            writer.write("    <component name=\"" + connectorName + "\" package=\"" + connectorPackage + "\">\n");

            // Get the folder structure
            File rootFolder = new File("src/main/resources");
            File[] files = rootFolder.listFiles();

            if (files != null) {
                for (File file : files) {
                    // check if file is a directory and directory contains xml files
                    if (file.isDirectory() && file.listFiles() != null) {
                        boolean containsXml = false;
                        for (File innerFile : file.listFiles()) {
                            if (innerFile.isFile() && innerFile.getName().endsWith(".xml")) {
                                // Add a <resource> element for each xml file
                                containsXml = true;
                                break;
                            }
                        }
                        if (containsXml) {
                            writer.write("        <dependency component=\"" + file.getName() + "\"/>\n");
                        }
                    }
                }
            }
            // Add a description element
            writer.write("        <description>wso2 " + connectorName + " connector library</description>\n");

            // Close the <component> and <connector> elements
            writer.write("    </component>\n");
            String iconPath = deriveIconPath();
            if (!iconPath.isEmpty()) {
                writer.write("    <icon>" + iconPath + "</icon>\n");
            }
            writer.write("</connector>\n");
            connectorMojo.getLog().info("connector.xml file generated successfully.");
        } catch (IOException e) {
            connectorMojo.getLog().error("Error occurred while generating connector.xml file.", e);
        }
    }

    /**
     * Derives the path of the icon file.
     *
     * @return The path of the icon file
     */
    private static String deriveIconPath() {

        String iconFolder = "src" + File.separator + "main" + File.separator + "resources" + File.separator + "icon";
        if (new File(iconFolder + File.separator + Constants.ICON_SMALL_GIF).exists()) {
            return Constants.ICON + File.separator + Constants.ICON_SMALL_GIF;
        } else if (new File(iconFolder + File.separator + Constants.ICON_SMALL_PNG).exists()) {
            return Constants.ICON + File.separator + Constants.ICON_SMALL_PNG;
        } else {
            return "";
        }
    }
}
