/*
 * Decompiled with CFR 0.152.
 */
package org.wso2.broker.coordination.rdbms;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.wso2.broker.coordination.CoordinationException;
import org.wso2.broker.coordination.CoordinationStrategy;
import org.wso2.broker.coordination.node.NodeDetail;
import org.wso2.broker.coordination.node.NodeHeartbeatData;
import org.wso2.broker.coordination.rdbms.CoordinationConfiguration;
import org.wso2.broker.coordination.rdbms.MembershipEventType;
import org.wso2.broker.coordination.rdbms.RdbmsCoordinationDaoImpl;
import org.wso2.broker.coordination.rdbms.RdbmsMembershipEventingEngine;
import org.wso2.broker.coordination.rdbms.RdbmsMembershipListener;

public class RdbmsCoordinationStrategy
implements CoordinationStrategy,
RdbmsMembershipListener {
    private Logger logger = LoggerFactory.getLogger(RdbmsCoordinationStrategy.class);
    private RdbmsMembershipEventingEngine membershipEventingEngine;
    private int coordinatorEntryCreationWaitTime;
    private final int heartBeatInterval;
    private final int heartbeatMaxAge;
    private CoordinatorElectionTask coordinatorElectionTask;
    private NodeState currentNodeState;
    private RdbmsCoordinationDaoImpl coordinationDao;
    private String localNodeId;
    private final ExecutorService threadExecutor;
    private final ScheduledExecutorService scheduledExecutorService;
    private int eventPollingInterval;

    public RdbmsCoordinationStrategy(RdbmsCoordinationDaoImpl rdbmsCoordinationDaoImpl, CoordinationConfiguration.RdbmsCoordinationConfiguration rdbmsCoordinationConfiguration) {
        ThreadFactory namedThreadFactory = new ThreadFactoryBuilder().setNameFormat("RdbmsCoordinationStrategy-%d").build();
        this.threadExecutor = Executors.newSingleThreadExecutor(namedThreadFactory);
        this.scheduledExecutorService = Executors.newSingleThreadScheduledExecutor();
        this.heartBeatInterval = rdbmsCoordinationConfiguration.getHeartbeatInterval();
        this.eventPollingInterval = rdbmsCoordinationConfiguration.getEventPollingInterval();
        this.coordinatorEntryCreationWaitTime = rdbmsCoordinationConfiguration.getCoordinatorEntryCreationWaitTime();
        this.heartbeatMaxAge = this.heartBeatInterval * 2;
        if (this.heartBeatInterval <= this.coordinatorEntryCreationWaitTime) {
            throw new RuntimeException("Configuration error. " + this.heartBeatInterval + " * 2 should be greater than " + this.coordinatorEntryCreationWaitTime);
        }
        this.localNodeId = rdbmsCoordinationConfiguration.getNodeId();
        this.coordinationDao = rdbmsCoordinationDaoImpl;
    }

    @Override
    public void memberAdded(String nodeID) {
    }

    @Override
    public void memberRemoved(String nodeID) {
    }

    @Override
    public void coordinatorChanged(String coordinator) {
    }

    @Override
    public void start() {
        this.membershipEventingEngine = new RdbmsMembershipEventingEngine(this.coordinationDao);
        this.membershipEventingEngine.start(this.localNodeId, this.eventPollingInterval);
        this.membershipEventingEngine.addEventListener(this);
        this.currentNodeState = NodeState.ELECTION;
        try {
            this.coordinationDao.clearMembershipEvents(this.localNodeId);
            this.coordinationDao.removeNodeHeartbeat(this.localNodeId);
        }
        catch (CoordinationException e) {
            this.logger.warn("Error while clearing old membership events for local node (" + this.localNodeId + ")", (Throwable)e);
        }
        this.coordinatorElectionTask = new CoordinatorElectionTask();
        this.threadExecutor.execute(this.coordinatorElectionTask);
        int timeout = 500;
        int waitTime = 0;
        int maxWaitTime = this.heartbeatMaxAge * 5;
        while (this.currentNodeState == NodeState.ELECTION) {
            try {
                TimeUnit.MILLISECONDS.sleep(timeout);
                if ((waitTime += timeout) != maxWaitTime) continue;
                throw new RuntimeException("Node is stuck in the ELECTION state for " + waitTime + " milliseconds.");
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new RuntimeException("An error occurred while waiting to get current node state.", e);
            }
        }
    }

    @Override
    public boolean isCoordinator() {
        return this.currentNodeState == NodeState.COORDINATOR;
    }

    @Override
    public String getNodeIdentifierOfCoordinator() throws CoordinationException {
        return this.coordinationDao.getCoordinatorNodeId();
    }

    @Override
    public List<String> getAllNodeIdentifiers() throws CoordinationException {
        List<NodeHeartbeatData> allNodeInformation = this.coordinationDao.getAllHeartBeatData();
        return this.getNodeIds(allNodeInformation);
    }

    @Override
    public List<NodeDetail> getAllNodeDetails() throws CoordinationException {
        ArrayList<NodeDetail> nodeDetails = new ArrayList<NodeDetail>();
        List<NodeHeartbeatData> allHeartBeatData = this.coordinationDao.getAllHeartBeatData();
        String coordinatorNodeId = this.coordinationDao.getCoordinatorNodeId();
        for (NodeHeartbeatData nodeHeartBeatData : allHeartBeatData) {
            boolean isCoordinatorNode = coordinatorNodeId.equals(nodeHeartBeatData.getNodeId());
            nodeDetails.add(new NodeDetail(nodeHeartBeatData.getNodeId(), isCoordinatorNode));
        }
        return nodeDetails;
    }

    @Override
    public void stop() {
        if (this.isCoordinator()) {
            try {
                this.coordinationDao.removeCoordinator();
            }
            catch (CoordinationException e) {
                this.logger.error("Error occurred while removing coordinator when shutting down", (Throwable)e);
            }
        }
        this.membershipEventingEngine.stop();
        this.coordinatorElectionTask.stop();
        this.threadExecutor.shutdown();
        this.scheduledExecutorService.shutdown();
    }

    private List<String> getNodeIds(List<NodeHeartbeatData> allHeartbeatData) {
        ArrayList<String> allNodeIds = new ArrayList<String>(allHeartbeatData.size());
        for (NodeHeartbeatData nodeHeartBeatData : allHeartbeatData) {
            allNodeIds.add(nodeHeartBeatData.getNodeId());
        }
        return allNodeIds;
    }

    private class CoordinatorElectionTask
    implements Runnable {
        private boolean running = true;
        private ScheduledFuture<?> scheduledFuture;

        private CoordinatorElectionTask() {
        }

        @Override
        public void run() {
            while (this.running) {
                try {
                    if (RdbmsCoordinationStrategy.this.logger.isDebugEnabled()) {
                        RdbmsCoordinationStrategy.this.logger.debug("Current node state: " + (Object)((Object)RdbmsCoordinationStrategy.this.currentNodeState));
                    }
                    switch (RdbmsCoordinationStrategy.this.currentNodeState) {
                        case CANDIDATE: {
                            RdbmsCoordinationStrategy.this.currentNodeState = this.performStandByTask();
                            break;
                        }
                        case COORDINATOR: {
                            RdbmsCoordinationStrategy.this.currentNodeState = this.performCoordinatorTask();
                            break;
                        }
                        case ELECTION: {
                            RdbmsCoordinationStrategy.this.currentNodeState = this.performElectionTask();
                        }
                    }
                }
                catch (Throwable e) {
                    RdbmsCoordinationStrategy.this.logger.error("Error detected while running coordination algorithm. Node became a " + (Object)((Object)NodeState.CANDIDATE) + " node", e);
                    this.cancelStateExpirationTask();
                    RdbmsCoordinationStrategy.this.currentNodeState = NodeState.CANDIDATE;
                }
            }
        }

        private NodeState performStandByTask() throws CoordinationException, InterruptedException {
            NodeState nextState;
            this.updateNodeHeartBeat();
            boolean coordinatorValid = RdbmsCoordinationStrategy.this.coordinationDao.checkIfCoordinatorValid(RdbmsCoordinationStrategy.this.heartbeatMaxAge);
            TimeUnit.MILLISECONDS.sleep(RdbmsCoordinationStrategy.this.heartBeatInterval);
            if (coordinatorValid) {
                nextState = NodeState.CANDIDATE;
            } else {
                coordinatorValid = RdbmsCoordinationStrategy.this.coordinationDao.checkIfCoordinatorValid(RdbmsCoordinationStrategy.this.heartbeatMaxAge);
                if (coordinatorValid) {
                    nextState = NodeState.CANDIDATE;
                } else {
                    RdbmsCoordinationStrategy.this.logger.info("Going for election since the Coordinator is invalid");
                    RdbmsCoordinationStrategy.this.coordinationDao.removeCoordinator();
                    nextState = NodeState.ELECTION;
                }
            }
            return nextState;
        }

        private void updateNodeHeartBeat() throws CoordinationException {
            boolean heartbeatEntryExists = RdbmsCoordinationStrategy.this.coordinationDao.updateNodeHeartbeat(RdbmsCoordinationStrategy.this.localNodeId);
            if (!heartbeatEntryExists) {
                RdbmsCoordinationStrategy.this.coordinationDao.createNodeHeartbeatEntry(RdbmsCoordinationStrategy.this.localNodeId);
            }
        }

        private NodeState performCoordinatorTask() throws CoordinationException, InterruptedException {
            boolean stillCoordinator = RdbmsCoordinationStrategy.this.coordinationDao.updateCoordinatorHeartbeat(RdbmsCoordinationStrategy.this.localNodeId);
            if (stillCoordinator) {
                this.resetScheduleStateExpirationTask();
                long startTime = System.currentTimeMillis();
                this.updateNodeHeartBeat();
                long currentTimeMillis = System.currentTimeMillis();
                List<NodeHeartbeatData> allNodeInformation = RdbmsCoordinationStrategy.this.coordinationDao.getAllHeartBeatData();
                List allActiveNodeIds = RdbmsCoordinationStrategy.this.getNodeIds(allNodeInformation);
                ArrayList<String> newNodes = new ArrayList<String>();
                ArrayList<String> removedNodes = new ArrayList<String>();
                for (NodeHeartbeatData nodeHeartBeatData : allNodeInformation) {
                    long heartbeatAge = currentTimeMillis - nodeHeartBeatData.getLastHeartbeat();
                    String nodeId = nodeHeartBeatData.getNodeId();
                    if (nodeHeartBeatData.isNewNode()) {
                        newNodes.add(nodeId);
                        RdbmsCoordinationStrategy.this.coordinationDao.markNodeAsNotNew(nodeId);
                        continue;
                    }
                    if (heartbeatAge < (long)RdbmsCoordinationStrategy.this.heartbeatMaxAge) continue;
                    removedNodes.add(nodeId);
                    allActiveNodeIds.remove(nodeId);
                    RdbmsCoordinationStrategy.this.coordinationDao.removeNodeHeartbeat(nodeId);
                }
                for (String newNode : newNodes) {
                    RdbmsCoordinationStrategy.this.logger.info("Member added " + newNode);
                    RdbmsCoordinationStrategy.this.membershipEventingEngine.notifyMembershipEvent(allActiveNodeIds, MembershipEventType.MEMBER_ADDED, newNode);
                }
                for (String removedNode : removedNodes) {
                    RdbmsCoordinationStrategy.this.logger.info("Member removed " + removedNode);
                    RdbmsCoordinationStrategy.this.membershipEventingEngine.notifyMembershipEvent(allActiveNodeIds, MembershipEventType.MEMBER_REMOVED, removedNode);
                }
                long endTime = System.currentTimeMillis();
                long timeToWait = (long)RdbmsCoordinationStrategy.this.heartBeatInterval - (endTime - startTime);
                if (timeToWait > 0L) {
                    TimeUnit.MILLISECONDS.sleep(timeToWait);
                } else {
                    RdbmsCoordinationStrategy.this.logger.warn("Sending membership events took more than the heart beat interval");
                }
                return NodeState.COORDINATOR;
            }
            RdbmsCoordinationStrategy.this.logger.info("Going for election since Coordinator state is lost");
            this.cancelStateExpirationTask();
            return NodeState.ELECTION;
        }

        private NodeState performElectionTask() throws InterruptedException {
            NodeState nextState;
            try {
                nextState = this.tryToElectSelfAsCoordinator();
            }
            catch (CoordinationException e) {
                RdbmsCoordinationStrategy.this.logger.info("Current node became a " + (Object)((Object)NodeState.CANDIDATE) + " node", (Throwable)e);
                nextState = NodeState.CANDIDATE;
            }
            return nextState;
        }

        private NodeState tryToElectSelfAsCoordinator() throws CoordinationException, InterruptedException {
            NodeState nextState;
            boolean electedAsCoordinator = RdbmsCoordinationStrategy.this.coordinationDao.createCoordinatorEntry(RdbmsCoordinationStrategy.this.localNodeId);
            if (electedAsCoordinator) {
                TimeUnit.MILLISECONDS.sleep(RdbmsCoordinationStrategy.this.coordinatorEntryCreationWaitTime);
                boolean isCoordinator = RdbmsCoordinationStrategy.this.coordinationDao.checkIsCoordinator(RdbmsCoordinationStrategy.this.localNodeId);
                if (isCoordinator) {
                    RdbmsCoordinationStrategy.this.coordinationDao.updateCoordinatorHeartbeat(RdbmsCoordinationStrategy.this.localNodeId);
                    this.resetScheduleStateExpirationTask();
                    RdbmsCoordinationStrategy.this.logger.info("Elected current node as the coordinator");
                    nextState = NodeState.COORDINATOR;
                    RdbmsCoordinationStrategy.this.membershipEventingEngine.notifyMembershipEvent(RdbmsCoordinationStrategy.this.getAllNodeIdentifiers(), MembershipEventType.COORDINATOR_CHANGED, RdbmsCoordinationStrategy.this.localNodeId);
                } else {
                    RdbmsCoordinationStrategy.this.logger.info("Election resulted in current node becoming a " + (Object)((Object)NodeState.CANDIDATE) + " node");
                    nextState = NodeState.CANDIDATE;
                }
            } else {
                RdbmsCoordinationStrategy.this.logger.info("Election resulted in current node becoming a " + (Object)((Object)NodeState.CANDIDATE) + " node");
                nextState = NodeState.CANDIDATE;
            }
            return nextState;
        }

        public void stop() {
            this.running = false;
        }

        private void cancelStateExpirationTask() {
            if (this.scheduledFuture != null) {
                this.scheduledFuture.cancel(true);
                this.scheduledFuture = null;
            }
        }

        private void resetScheduleStateExpirationTask() {
            this.cancelStateExpirationTask();
            this.scheduledFuture = RdbmsCoordinationStrategy.this.scheduledExecutorService.schedule(new Runnable(){

                @Override
                public void run() {
                    RdbmsCoordinationStrategy.this.currentNodeState = NodeState.ELECTION;
                }
            }, (long)RdbmsCoordinationStrategy.this.heartbeatMaxAge, TimeUnit.MILLISECONDS);
        }
    }

    private static enum NodeState {
        COORDINATOR,
        CANDIDATE,
        ELECTION;

    }
}

