/*
 * Decompiled with CFR 0.152.
 */
package org.wso2.broker.coordination.rdbms;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.wso2.broker.coordination.CoordinationException;
import org.wso2.broker.coordination.CoordinationStrategy;
import org.wso2.broker.coordination.node.NodeDetail;
import org.wso2.broker.coordination.node.NodeHeartbeatData;
import org.wso2.broker.coordination.rdbms.RdbmsCoordinationDaoImpl;
import org.wso2.broker.coordination.rdbms.RdbmsCoordinationListener;

public class RdbmsCoordinationStrategy
implements CoordinationStrategy {
    private Logger logger = LoggerFactory.getLogger(RdbmsCoordinationStrategy.class);
    private final int coordinatorEntryCreationWaitTime;
    private final int heartBeatInterval;
    private final int heartbeatMaxAge;
    private CoordinatorElectionTask coordinatorElectionTask;
    private NodeState currentNodeState;
    private RdbmsCoordinationDaoImpl coordinationDao;
    private String localNodeId;
    private final ExecutorService threadExecutor;
    private final ScheduledExecutorService scheduledExecutorService;
    private List<RdbmsCoordinationListener> coordinationListeners = new ArrayList<RdbmsCoordinationListener>();

    public RdbmsCoordinationStrategy(RdbmsCoordinationDaoImpl rdbmsCoordinationDaoImpl, Map<String, String> rdbmsCoordinationOptions) {
        ThreadFactory namedThreadFactory = new ThreadFactoryBuilder().setNameFormat("RdbmsCoordinationStrategy-%d").build();
        this.threadExecutor = Executors.newSingleThreadExecutor(namedThreadFactory);
        this.scheduledExecutorService = Executors.newSingleThreadScheduledExecutor();
        this.heartBeatInterval = rdbmsCoordinationOptions.get("heartbeatInterval") != null ? Integer.parseInt(rdbmsCoordinationOptions.get("heartbeatInterval")) : 5000;
        this.coordinatorEntryCreationWaitTime = rdbmsCoordinationOptions.get("coordinatorEntryCreationWaitTime") != null ? Integer.parseInt(rdbmsCoordinationOptions.get("coordinatorEntryCreationWaitTime")) : 3000;
        this.localNodeId = rdbmsCoordinationOptions.get("nodeId");
        if (this.localNodeId == null) {
            this.localNodeId = UUID.randomUUID().toString();
        }
        this.heartbeatMaxAge = this.heartBeatInterval * 2;
        if (this.heartBeatInterval <= this.coordinatorEntryCreationWaitTime) {
            throw new RuntimeException("Configuration error. " + this.heartBeatInterval + " * 2 should be greater than " + this.coordinatorEntryCreationWaitTime);
        }
        this.coordinationDao = rdbmsCoordinationDaoImpl;
    }

    private void becameCoordinatorNode() {
        for (RdbmsCoordinationListener rdbmsCoordinationListener : this.coordinationListeners) {
            this.scheduledExecutorService.submit(rdbmsCoordinationListener::becameCoordinatorNode);
        }
    }

    private void lostCoordinatorState() {
        for (RdbmsCoordinationListener rdbmsCoordinationListener : this.coordinationListeners) {
            this.scheduledExecutorService.submit(rdbmsCoordinationListener::lostCoordinatorState);
        }
    }

    @Override
    public void start() {
        this.setCurrentNodeState(NodeState.ELECTION);
        this.coordinatorElectionTask = new CoordinatorElectionTask();
        this.threadExecutor.execute(this.coordinatorElectionTask);
        int timeout = 500;
        int waitTime = 0;
        int maxWaitTime = this.heartbeatMaxAge * 5;
        while (this.currentNodeState == NodeState.ELECTION) {
            try {
                TimeUnit.MILLISECONDS.sleep(timeout);
                if ((waitTime += timeout) != maxWaitTime) continue;
                throw new RuntimeException("Node is stuck in the ELECTION state for " + waitTime + " milliseconds.");
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new RuntimeException("An error occurred while waiting to get current node state.", e);
            }
        }
    }

    @Override
    public boolean isCoordinator() {
        return this.currentNodeState == NodeState.COORDINATOR;
    }

    @Override
    public String getNodeIdentifierOfCoordinator() throws CoordinationException {
        return this.coordinationDao.getCoordinatorNodeId();
    }

    @Override
    public List<String> getAllNodeIdentifiers() throws CoordinationException {
        List<NodeHeartbeatData> allNodeInformation = this.coordinationDao.getAllHeartBeatData();
        return this.getNodeIds(allNodeInformation);
    }

    @Override
    public List<NodeDetail> getAllNodeDetails() throws CoordinationException {
        ArrayList<NodeDetail> nodeDetails = new ArrayList<NodeDetail>();
        List<NodeHeartbeatData> allHeartBeatData = this.coordinationDao.getAllHeartBeatData();
        String coordinatorNodeId = this.coordinationDao.getCoordinatorNodeId();
        for (NodeHeartbeatData nodeHeartBeatData : allHeartBeatData) {
            boolean isCoordinatorNode = coordinatorNodeId.equals(nodeHeartBeatData.getNodeId());
            nodeDetails.add(new NodeDetail(nodeHeartBeatData.getNodeId(), isCoordinatorNode));
        }
        return nodeDetails;
    }

    @Override
    public void stop() {
        if (this.isCoordinator()) {
            try {
                this.coordinationDao.removeCoordinator();
            }
            catch (CoordinationException e) {
                this.logger.error("Error occurred while removing coordinator when shutting down", (Throwable)e);
            }
        }
        this.coordinatorElectionTask.stop();
        this.threadExecutor.shutdown();
        this.scheduledExecutorService.shutdown();
    }

    private List<String> getNodeIds(List<NodeHeartbeatData> allHeartbeatData) {
        ArrayList<String> allNodeIds = new ArrayList<String>(allHeartbeatData.size());
        for (NodeHeartbeatData nodeHeartBeatData : allHeartbeatData) {
            allNodeIds.add(nodeHeartBeatData.getNodeId());
        }
        return allNodeIds;
    }

    public void addCoordinationListener(RdbmsCoordinationListener rdbmsCoordinationListener) {
        this.coordinationListeners.add(rdbmsCoordinationListener);
    }

    public void removeCoordinationListener(RdbmsCoordinationListener rdbmsCoordinationListener) {
        this.coordinationListeners.remove(rdbmsCoordinationListener);
    }

    private void setCurrentNodeState(NodeState newNodeState) {
        if (NodeState.COORDINATOR.equals((Object)this.currentNodeState)) {
            if (NodeState.ELECTION.equals((Object)newNodeState)) {
                this.lostCoordinatorState();
            }
        } else if (NodeState.ELECTION.equals((Object)this.currentNodeState) && NodeState.COORDINATOR.equals((Object)newNodeState)) {
            this.becameCoordinatorNode();
        }
        this.currentNodeState = newNodeState;
    }

    private class CoordinatorElectionTask
    implements Runnable {
        private boolean running = true;
        private ScheduledFuture<?> scheduledFuture;

        private CoordinatorElectionTask() {
        }

        @Override
        public void run() {
            while (this.running) {
                try {
                    if (RdbmsCoordinationStrategy.this.logger.isDebugEnabled()) {
                        RdbmsCoordinationStrategy.this.logger.debug("Current node state: " + (Object)((Object)RdbmsCoordinationStrategy.this.currentNodeState));
                    }
                    switch (RdbmsCoordinationStrategy.this.currentNodeState) {
                        case CANDIDATE: {
                            RdbmsCoordinationStrategy.this.setCurrentNodeState(this.performStandByTask());
                            break;
                        }
                        case COORDINATOR: {
                            RdbmsCoordinationStrategy.this.setCurrentNodeState(this.performCoordinatorTask());
                            break;
                        }
                        case ELECTION: {
                            RdbmsCoordinationStrategy.this.setCurrentNodeState(this.performElectionTask());
                        }
                    }
                }
                catch (Throwable e) {
                    RdbmsCoordinationStrategy.this.logger.error("Error detected while running coordination algorithm. Node became a " + (Object)((Object)NodeState.ELECTION) + " node", e);
                    this.cancelStateExpirationTask();
                    RdbmsCoordinationStrategy.this.setCurrentNodeState(NodeState.ELECTION);
                }
            }
        }

        private NodeState performStandByTask() throws CoordinationException, InterruptedException {
            NodeState nextState;
            this.updateNodeHeartBeat();
            boolean coordinatorValid = RdbmsCoordinationStrategy.this.coordinationDao.checkIfCoordinatorValid(RdbmsCoordinationStrategy.this.heartbeatMaxAge);
            TimeUnit.MILLISECONDS.sleep(RdbmsCoordinationStrategy.this.heartBeatInterval);
            if (coordinatorValid) {
                nextState = NodeState.CANDIDATE;
            } else {
                coordinatorValid = RdbmsCoordinationStrategy.this.coordinationDao.checkIfCoordinatorValid(RdbmsCoordinationStrategy.this.heartbeatMaxAge);
                if (coordinatorValid) {
                    nextState = NodeState.CANDIDATE;
                } else {
                    RdbmsCoordinationStrategy.this.logger.info("Going for election since the Coordinator is invalid");
                    RdbmsCoordinationStrategy.this.coordinationDao.removeCoordinator();
                    nextState = NodeState.ELECTION;
                }
            }
            return nextState;
        }

        private void updateNodeHeartBeat() throws CoordinationException {
            boolean heartbeatEntryExists = RdbmsCoordinationStrategy.this.coordinationDao.updateNodeHeartbeat(RdbmsCoordinationStrategy.this.localNodeId);
            if (!heartbeatEntryExists) {
                RdbmsCoordinationStrategy.this.coordinationDao.createNodeHeartbeatEntry(RdbmsCoordinationStrategy.this.localNodeId);
            }
        }

        private NodeState performCoordinatorTask() throws CoordinationException, InterruptedException {
            boolean stillCoordinator = RdbmsCoordinationStrategy.this.coordinationDao.updateCoordinatorHeartbeat(RdbmsCoordinationStrategy.this.localNodeId);
            if (stillCoordinator) {
                this.resetScheduleStateExpirationTask();
                long startTime = System.currentTimeMillis();
                this.updateNodeHeartBeat();
                long currentTimeMillis = System.currentTimeMillis();
                List<NodeHeartbeatData> allNodeInformation = RdbmsCoordinationStrategy.this.coordinationDao.getAllHeartBeatData();
                List allActiveNodeIds = RdbmsCoordinationStrategy.this.getNodeIds(allNodeInformation);
                ArrayList<String> newNodes = new ArrayList<String>();
                ArrayList<String> removedNodes = new ArrayList<String>();
                for (NodeHeartbeatData nodeHeartBeatData : allNodeInformation) {
                    long heartbeatAge = currentTimeMillis - nodeHeartBeatData.getLastHeartbeat();
                    String nodeId = nodeHeartBeatData.getNodeId();
                    if (nodeHeartBeatData.isNewNode()) {
                        newNodes.add(nodeId);
                        RdbmsCoordinationStrategy.this.coordinationDao.markNodeAsNotNew(nodeId);
                        continue;
                    }
                    if (heartbeatAge < (long)RdbmsCoordinationStrategy.this.heartbeatMaxAge) continue;
                    removedNodes.add(nodeId);
                    allActiveNodeIds.remove(nodeId);
                    RdbmsCoordinationStrategy.this.coordinationDao.removeNodeHeartbeat(nodeId);
                }
                for (String newNode : newNodes) {
                    RdbmsCoordinationStrategy.this.logger.info("Member added " + newNode);
                }
                for (String removedNode : removedNodes) {
                    RdbmsCoordinationStrategy.this.logger.info("Member removed " + removedNode);
                }
                long endTime = System.currentTimeMillis();
                long timeToWait = (long)RdbmsCoordinationStrategy.this.heartBeatInterval - (endTime - startTime);
                if (timeToWait > 0L) {
                    TimeUnit.MILLISECONDS.sleep(timeToWait);
                } else {
                    RdbmsCoordinationStrategy.this.logger.warn("Sending membership events took more than the heart beat interval");
                }
                return NodeState.COORDINATOR;
            }
            RdbmsCoordinationStrategy.this.logger.info("Going for election since Coordinator state is lost");
            this.cancelStateExpirationTask();
            return NodeState.ELECTION;
        }

        private NodeState performElectionTask() throws InterruptedException {
            NodeState nextState;
            try {
                nextState = this.tryToElectSelfAsCoordinator();
            }
            catch (CoordinationException e) {
                RdbmsCoordinationStrategy.this.logger.info("Current node became a " + (Object)((Object)NodeState.CANDIDATE) + " node", (Throwable)e);
                nextState = NodeState.CANDIDATE;
            }
            return nextState;
        }

        private NodeState tryToElectSelfAsCoordinator() throws CoordinationException, InterruptedException {
            NodeState nextState;
            boolean electedAsCoordinator = RdbmsCoordinationStrategy.this.coordinationDao.createCoordinatorEntry(RdbmsCoordinationStrategy.this.localNodeId);
            if (electedAsCoordinator) {
                TimeUnit.MILLISECONDS.sleep(RdbmsCoordinationStrategy.this.coordinatorEntryCreationWaitTime);
                boolean isCoordinator = RdbmsCoordinationStrategy.this.coordinationDao.checkIsCoordinator(RdbmsCoordinationStrategy.this.localNodeId);
                if (isCoordinator) {
                    RdbmsCoordinationStrategy.this.coordinationDao.updateCoordinatorHeartbeat(RdbmsCoordinationStrategy.this.localNodeId);
                    this.resetScheduleStateExpirationTask();
                    RdbmsCoordinationStrategy.this.logger.info("Elected current node as the coordinator");
                    nextState = NodeState.COORDINATOR;
                } else {
                    RdbmsCoordinationStrategy.this.logger.info("Election resulted in current node becoming a " + (Object)((Object)NodeState.CANDIDATE) + " node");
                    nextState = NodeState.CANDIDATE;
                }
            } else {
                RdbmsCoordinationStrategy.this.logger.info("Election resulted in current node becoming a " + (Object)((Object)NodeState.CANDIDATE) + " node");
                nextState = NodeState.CANDIDATE;
            }
            return nextState;
        }

        public void stop() {
            this.running = false;
        }

        private void cancelStateExpirationTask() {
            if (this.scheduledFuture != null) {
                this.scheduledFuture.cancel(true);
                this.scheduledFuture = null;
            }
        }

        private void resetScheduleStateExpirationTask() {
            this.cancelStateExpirationTask();
            this.scheduledFuture = RdbmsCoordinationStrategy.this.scheduledExecutorService.schedule(new Runnable(){

                @Override
                public void run() {
                    RdbmsCoordinationStrategy.this.setCurrentNodeState(NodeState.ELECTION);
                }
            }, (long)RdbmsCoordinationStrategy.this.heartbeatMaxAge, TimeUnit.MILLISECONDS);
        }
    }

    private static enum NodeState {
        COORDINATOR,
        CANDIDATE,
        ELECTION;

    }
}

