/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package wvlet.surface

import scala.language.experimental.macros
import scala.reflect.macros.{blackbox => sm}

/**
  *
  */
private[surface] object SurfaceMacros {

  class SurfaceGenerator[C <: sm.Context](val c: C) {

    import c.universe._

    private val seen = scala.collection.mutable.Set[Type]()
    private val memo = scala.collection.mutable.Map[Type, c.Tree]()

    type SurfaceFactory = PartialFunction[c.Type, c.Tree]

    private def isEnum(t: c.Type): Boolean = {
      t.baseClasses.exists(x =>
        x.isJava && x.isType && x.asType.name.decodedName.toString.startsWith("java.lang.Enum")
      )
    }

    private def typeArgsOf(t: c.Type): List[c.Type] = t match {
      case TypeRef(prefix, symbol, args) =>
        args
      case ExistentialType(quantified, underlying) =>
        typeArgsOf(underlying)
      case other =>
        List.empty
    }

    private def typeNameOf(t: c.Type): String = {
      t.dealias.typeSymbol.fullName
    }

    private def elementTypeOf(t: c.Type): c.Tree = {
      typeArgsOf(t).map(toSurface(_)).head
    }

    private val toAlias: SurfaceFactory = {
      case alias@TypeRef(prefix, symbol, args)
        if symbol.isType &&
          symbol.asType.isAliasType &&
          !belongsToScalaDefault(alias)
      =>
        val inner = toSurface(alias.dealias)
        val name = symbol.asType.name.decodedName.toString
        val fullName = s"${prefix.typeSymbol.fullName}.${name}"
        q"wvlet.surface.Alias(${name}, ${fullName}, $inner)"
    }

    private val toPrimitive: SurfaceFactory = {
      case t if t == typeOf[Short] => q"wvlet.surface.Primitive.Short"
      case t if t == typeOf[Boolean] => q"wvlet.surface.Primitive.Boolean"
      case t if t == typeOf[Byte] => q"wvlet.surface.Primitive.Byte"
      case t if t == typeOf[Char] => q"wvlet.surface.Primitive.Char"
      case t if t == typeOf[Int] => q"wvlet.surface.Primitive.Int"
      case t if t == typeOf[Float] => q"wvlet.surface.Primitive.Float"
      case t if t == typeOf[Long] => q"wvlet.surface.Primitive.Long"
      case t if t == typeOf[Double] => q"wvlet.surface.Primitive.Double"
      case t if t == typeOf[String] => q"wvlet.surface.Primitive.String"
      case t if t == typeOf[Unit] => q"wvlet.surface.Primitive.Unit"
    }

    private val toArray: SurfaceFactory = {
      case t if typeNameOf(t) == "scala.Array" =>
        q"wvlet.surface.ArraySurface(classOf[$t], ${elementTypeOf(t)})"
    }

    private val toOption: SurfaceFactory = {
      case t if typeNameOf(t) == "scala.Option" =>
        q"wvlet.surface.OptionSurface(classOf[$t], ${elementTypeOf(t)})"
    }

    private val toTuple: SurfaceFactory = {
      case t if t <:< typeOf[Product] && t.typeSymbol.fullName.startsWith("scala.Tuple") =>
        val paramType = typeArgsOf(t).map(x => toSurface(x))
        q"new wvlet.surface.TupleSurface(classOf[$t], IndexedSeq(..$paramType))"
    }

    private val toJavaUtil: SurfaceFactory = {
      case t if
      t =:= typeOf[java.io.File] ||
        t =:= typeOf[java.util.Date] ||
        t =:= typeOf[java.time.temporal.Temporal]
      =>
        q"new wvlet.surface.GenericSurface(classOf[$t])"
    }

    private val toEnum: SurfaceFactory = {
      case t if isEnum(t) =>
        q"wvlet.surface.EnumSurface(classOf[$t])"
    }

    private val scalaDefaultPackages = Seq("scala.", "scala.Predef.", "scala.util.")
    private def belongsToScalaDefault(t: c.Type) = {
      t match {
        case TypeRef(prefix, _, _) =>
          scalaDefaultPackages.exists(p => prefix.dealias.typeSymbol.fullName.startsWith(p))
        case _ => false
      }
    }

    private def isPhantomConstructor(constructor: Symbol): Boolean =
      constructor.asMethod.fullName.endsWith("$init$")

    def publicConstructorsOf(t: c.Type): Iterable[MethodSymbol] = {
      t.members
      .filter(m => m.isMethod && m.asMethod.isConstructor && m.isPublic)
      .filterNot(isPhantomConstructor)
      .map(_.asMethod)
    }

    def findPrimaryConstructorOf(t: c.Type): Option[MethodSymbol] = {
      publicConstructorsOf(t).find(x => x.isPrimaryConstructor)
    }

    def hasAbstractMethods(t: c.Type): Boolean = t.members.exists(x =>
      x.isMethod && x.isAbstract && !x.isAbstractOverride
    )

    private def isAbstract(t: c.Type): Boolean = {
      t.typeSymbol.isAbstract && hasAbstractMethods(t)
    }

    case class MethodArg(paramName:Symbol, tpe:c.Type) {
      def name : Literal = Literal(Constant(paramName.name.decodedName.toString))
      def typeSurface : c.Tree = toSurface(tpe)
    }

    def methodArgsOf(targetType:c.Type, constructor:MethodSymbol) : List[MethodArg] = {
      val classTypeParams = targetType.typeSymbol.asClass.typeParams
      val params = constructor.paramLists.flatten
      val concreteArgTypes = params.map(_.typeSignature.substituteTypes(classTypeParams, targetType.typeArgs))
      for ((p, t) <- params.zip(concreteArgTypes)) yield {
        MethodArg(p, t)
      }
    }

    def methodParmetersOf(targetType:c.Type, method: MethodSymbol) : c.Tree = {
      val surfaceParams = methodArgsOf(targetType, method).map { arg =>
        val t = arg.name

        //accessor = { x : Any => x.asInstanceOf[${arg.tpe}].${arg.paramName} }
        q"""wvlet.surface.Parameter(
            name=${arg.name},
            surface=${arg.typeSurface}
          )
          """
      }
      // Using IndexedSeq is necessary for Serialization
      q"IndexedSeq(..${surfaceParams})"
    }

    def createObjectFactoryOf(targetType: c.Type) : Option[c.Tree] = {
      if(isAbstract(targetType) || hasAbstractMethods(targetType)) {
        None
      }
      else {
        findPrimaryConstructorOf(targetType).map {primaryConstructor =>
          val args = methodArgsOf(targetType, primaryConstructor)
          val argExtractor: List[c.Tree] = for ((a, i) <- args.zipWithIndex) yield {
            val param = Apply(Ident(TermName("args")), List(Literal(Constant(i))))
            // TODO natural type conversion (e.g., Int -> Long, etc.)
            q"${param}.asInstanceOf[${a.tpe}]"
          }
          val constructor: c.Tree = Apply(Select(New(Ident(targetType.dealias.typeSymbol)), termNames.CONSTRUCTOR), argExtractor)
          val expr =
          q"""new wvlet.surface.ObjectFactory {
            def newInstance(args:Seq[Any]) : ${targetType} = { $constructor }
          }
          """
          expr
        }
      }
    }

    private val toSurfaceWithConstructorParams: SurfaceFactory = new SurfaceFactory {
      override def isDefinedAt(t: c.Type): Boolean = {
        !isAbstract(t) && findPrimaryConstructorOf(t).exists(!_.paramLists.isEmpty)
      }
      override def apply(t: c.Type): c.Tree = {
        val primaryConstructor = findPrimaryConstructorOf(t).get
        val typeArgs = typeArgsOf(t).map(toSurface(_))
        val factory = createObjectFactoryOf(t) match {
          case Some(x) => q"Some($x)"
          case None => q"None"
        }
        q"""
          new wvlet.surface.GenericSurface(
            classOf[$t],
            IndexedSeq(..$typeArgs),
            params = ${methodParmetersOf(t, primaryConstructor)},
            objectFactory=${factory}
        )"""
      }
    }

    private val toExistentialType: SurfaceFactory = {
      case t@ExistentialType(quantified, underlying) =>
        toSurface(underlying)
    }

    private val toGenericSurface: SurfaceFactory = {
      case t@TypeRef(prefix, symbol, args) if !args.isEmpty =>
        val typeArgs = typeArgsOf(t).map(toSurface(_))
        q"new wvlet.surface.GenericSurface(classOf[$t], typeArgs = IndexedSeq(..$typeArgs))"
      case t@TypeRef(NoPrefix, symbol, args) if !t.typeSymbol.isClass =>
        q"wvlet.surface.ExistentialType"
      case t =>
        val expr = q"new wvlet.surface.GenericSurface(classOf[$t])"
        expr
    }

    private val surfaceFactorySet: SurfaceFactory =
      toAlias orElse
        toArray orElse
        toOption orElse
        toTuple orElse
        toJavaUtil orElse
        toEnum orElse
        toSurfaceWithConstructorParams orElse
        toExistentialType orElse
        toGenericSurface

    def toSurface(t: c.Type): c.Tree = {
      if (seen.contains(t)) {
        if (memo.contains(t)) {
          memo(t)
        }
        else {
          c.abort(c.enclosingPosition, s"recursive type: ${t.typeSymbol.fullName}")
        }
      }
      else {
        seen += t
        // We don't need to cache primitive types
        val surfaceGenerator =
          toPrimitive orElse {
            surfaceFactorySet andThen {tree =>
              // cache the generated Surface instance
              q"wvlet.surface.Surface.surfaceCache.getOrElseUpdate(${fullTypeNameOf(t)}, ${tree})"
            }
          }
        val surface = surfaceGenerator(t)
        memo += (t -> surface)
        surface
      }
    }

    private def fullTypeNameOf(typeEv: c.Type): String = {
      typeEv match {
        case TypeRef(prefix, typeSymbol, args) =>
          if (args.isEmpty) {
            typeSymbol.fullName
          }
          else {
            val typeArgs = args.map(fullTypeNameOf(_)).mkString(",")
            s"${typeSymbol.fullName}[${typeArgs}]"
          }
        case other =>
          typeEv.typeSymbol.fullName
      }
    }

    private def isOwnedByTargetClass(m: MethodSymbol, t: c.Type): Boolean = {
      m.owner == t.typeSymbol
    }

    private def isTargetMethod(m: MethodSymbol, target: c.Type): Boolean = {
      // synthetic is used for functions returning default values of method arguments (e.g., ping$default$1)
      val methodName = m.name.decodedName.toString
      m.isMethod &&
        !m.isImplicit &&
        !m.isSynthetic &&
        !m.isAccessor &&
        !methodName.startsWith("$") &&
        methodName != "<init>" &&
        isOwnedByTargetClass(m, target)
    }

    def modifierBitMaskOf(m: MethodSymbol): Int = {
      var mod = 0
      if (m.isPublic) {
        mod |= 0x1
      }
      if (m.isPrivate) {
        mod |= 0x2
      }
      if (m.isProtected) {
        mod |= 0x4
      }
      if (m.isStatic) {
        mod |= 0x8
      }
      if (m.isFinal) {
        mod |= 0x10
      }
      if (m.isAbstract) {
        mod |= 0x400
      }
      mod
    }

    def createSurfaceOf(targetType: c.Type): c.Tree = {
      toSurface(targetType)
    }

    def localMethodsOf(t:c.Type) : Iterable[MethodSymbol] = {
      t.members
      .filter(x => x.isMethod && !x.isConstructor)
      .map(_.asMethod)
      .filter(isTargetMethod(_, t))
    }

    def createMethodSurfaceOf(targetType: c.Type): c.Tree = {
      val result = targetType match {
        case t@TypeRef(prefix, typeSymbol, typeArgs) =>
          val list = for(m <- localMethodsOf(targetType)) yield {
            val mod = modifierBitMaskOf(m)
            val owner = toSurface(t)
            val name = m.name.decodedName.toString
            val ret = toSurface(m.returnType)
            val args = methodParmetersOf(m.owner.typeSignature, m)
            q"wvlet.surface.ClassMethod(${mod}, ${owner}, ${name}, ${ret}, ${args}.toIndexedSeq)"
          }
          q"IndexedSeq(..$list)"
        case _ =>
          q"Seq.empty"
      }
      val fullName = fullTypeNameOf(targetType)
      q"wvlet.surface.Surface.methodSurfaceCache.getOrElseUpdate(${fullName}, ${result})"
    }

  }

  def of[A: c.WeakTypeTag](c: sm.Context): c.Tree = {
    val targetType = implicitly[c.WeakTypeTag[A]].tpe
    new SurfaceGenerator[c.type](c).createSurfaceOf(targetType)
  }

  def methodsOf[A: c.WeakTypeTag](c: sm.Context): c.Tree = {
    val targetType = implicitly[c.WeakTypeTag[A]].tpe
    new SurfaceGenerator[c.type](c).createMethodSurfaceOf(targetType)
  }
}
