/*
 * Copyright © 2023 Jacob Wysko
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wysko.kmidi

import org.wysko.kmidi.event.MidiEvent
import org.wysko.kmidi.event.MidiMetaEvent
import org.wysko.kmidi.event.MidiMetaEvent.EndOfTrack

private const val MAX_TPQ = 0b0111_1111_1111_1111

/**
 * Represents the data encoded in a Standard MIDI file.
 *
 * @param header The header of the file.
 * @param tracks The tracks of the file.
 */
data class StandardMidiFile(
    val header: Header,
    val tracks: List<Track>
) {
    init {
        check(tracks.size == header.trackCount.toInt()) {
            "Invalid track count: header specifies ${header.trackCount} tracks, but there are ${tracks.size} tracks"
        }
    }

    /**
     * The header of a Standard MIDI file.
     *
     * @param chunkType The chunk type, which is always "MThd".
     * @param format The format the file is encoded in.
     * @param trackCount The number of tracks in the file.
     * @param division The time division of the file.
     */
    data class Header(
        val chunkType: String,
        val format: Format,
        val trackCount: Short,
        val division: Division
    ) {
        init {
            require(chunkType == "MThd") {
                "Invalid header chunk type: $chunkType"
            }
        }

        /**
         * Specifies the overall organization of a [StandardMidiFile].
         */
        enum class Format {
            /**
             * The file contains a single multi-channel track.
             */
            Format0,

            /**
             * The file contains one or more simultaneous tracks (or MIDI outputs) of a sequence.
             */
            Format1,

            /**
             * The file contains one or more sequentially independent single-track patterns.
             */
            Format2;

            companion object {
                /**
                 * Converts a short value into a [Format] enum value.
                 *
                 * @param value The short value representing the format.
                 * @return The corresponding [Format] enum value.
                 * @throws IllegalArgumentException If the format value is invalid.
                 */
                fun fromValue(value: Short): Format = when (value.toInt()) {
                    0 -> Format0
                    1 -> Format1
                    2 -> Format2
                    else -> throw IllegalArgumentException("Invalid format value: $value")
                }
            }
        }

        /**
         * The time division of a [StandardMidiFile].
         */
        sealed class Division {
            /**
             * The time division is specified in terms of metrical time: ticks per quarter note.
             *
             * @param ticksPerQuarterNote The number of ticks (i.e., MIDI clocks) that elapse per quarter note.
             */
            data class MetricalTime(
                val ticksPerQuarterNote: Short
            ) : Division() {
                init {
                    require(ticksPerQuarterNote <= MAX_TPQ) {
                        "Invalid ticks per quarter note: $ticksPerQuarterNote"
                    }
                }
            }

            /**
             * The time division is specified in terms of an SMPTE timecode.
             *
             * @param framesPerSecond The number of frames per second.
             * @param ticksPerFrame The number of ticks (i.e., MIDI clocks) per frame.
             */
            data class TimecodeBasedTime(
                val framesPerSecond: Short,
                val ticksPerFrame: Short
            ) : Division() {
                @Suppress("MagicNumber")
                private val validFramesPerSecond = setOf(-24, -25, -29, -30).map { it.toShort() }.toSet()

                init {
                    require(framesPerSecond in validFramesPerSecond) {
                        "Invalid frames per second: $framesPerSecond"
                    }
                }
            }
        }
    }

    /**
     * A track of a [StandardMidiFile]. Tracks contain a sequence of MIDI events. The last event in a track is always
     * an [EndOfTrack] event.
     *
     * @param events The events in the track.
     */
    data class Track(val events: List<MidiEvent>) {
        /**
         * The name of the track, if it has one.
         */
        val name: String? = events.filterIsInstance<MidiMetaEvent.SequenceTrackName>().firstOrNull()?.text
    }
}