/*
 * Copyright © 2023 Jacob Wysko
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wysko.kmidi.event

import org.wysko.kmidi.SmpteTimecode

/**
 * Any meta MIDI event.
 */
sealed class MidiMetaEvent(override val time: Int) : MidiEvent(time) {

    /**
     * Specifies the number of a sequence.
     */
    data class SequenceNumber(val number: Short) : MidiMetaEvent(0)

    /**
     * Any amount of text describing anything.
     */
    data class Text(override val time: Int, val text: String) : MidiMetaEvent(time)

    /**
     * A copyright notice.
     */
    data class CopyrightNotice(val text: String) : MidiMetaEvent(0)

    /**
     * Sequence or track name.
     */
    data class SequenceTrackName(val text: String) : MidiMetaEvent(0)

    /**
     * Name of the instrument or voice.
     */
    data class InstrumentName(val text: String) : MidiMetaEvent(0)

    /**
     * A lyric to be sung.
     */
    data class Lyric(override val time: Int, val text: String) : MidiMetaEvent(time)

    /**
     * A marker.
     */
    data class Marker(override val time: Int, val text: String) : MidiMetaEvent(time)

    /**
     * A cue point.
     */
    data class CuePoint(override val time: Int, val text: String) : MidiMetaEvent(time)

    /**
     * A MIDI channel prefix assignment.
     */
    data class MidiChannelPrefix(override val time: Int, val channel: Byte) : MidiMetaEvent(time)

    /**
     * End of track.
     */
    data class EndOfTrack(override val time: Int) : MidiMetaEvent(time)

    /**
     * Set tempo.
     */
    @Suppress("MagicNumber")
    data class SetTempo(override val time: Int, val tempo: Int) : MidiMetaEvent(time) {
        /** Returns this tempo's value as expressed in beats per minute. */
        val beatsPerMinute: Double = 60_000_000.0 / tempo

        /** Returns this tempo's value as expressed in seconds per beat. */
        val secondsPerBeat: Double = 60 / beatsPerMinute
    }

    /**
     * SMPTE offset.
     */
    data class SmpteOffset(override val time: Int, val timecode: SmpteTimecode) : MidiMetaEvent(time)

    /**
     * Time signature.
     */
    data class TimeSignature(
        override val time: Int,
        val numerator: Byte,
        val denominator: Byte,
        val clocksInMetronomeClick: Byte,
        val thirtySecondNotesInMidiQuarterNote: Byte
    ) : MidiMetaEvent(time)

    /**
     * Key signature.
     */
    data class KeySignature(override val time: Int, val key: Key, val scale: Scale) : MidiMetaEvent(time) {

        @Suppress("MagicNumber")
        sealed class Key {
            data object CFlat : Key()
            data object GFlat : Key()
            data object DFlat : Key()
            data object AFlat : Key()
            data object EFlat : Key()
            data object BFlat : Key()
            data object F : Key()
            data object C : Key()
            data object G : Key()
            data object D : Key()
            data object A : Key()
            data object E : Key()
            data object B : Key()
            data object FSharp : Key()
            data object CSharp : Key()

            companion object {
                @Suppress("CyclomaticComplexMethod")
                fun fromValue(value: Byte): Key = when (value.toInt()) {
                    -7 -> CFlat
                    -6 -> GFlat
                    -5 -> DFlat
                    -4 -> AFlat
                    -3 -> EFlat
                    -2 -> BFlat
                    -1 -> F
                    0 -> C
                    1 -> G
                    2 -> D
                    3 -> A
                    4 -> E
                    5 -> B
                    6 -> FSharp
                    7 -> CSharp
                    else -> throw IllegalArgumentException("Invalid key value: $value")
                }
            }
        }

        sealed class Scale(val value: Int) {
            data object Major : Scale(0)
            data object Minor : Scale(1)
        }
    }

    /**
     * Sequencer specific event.
     */
    data class SequencerSpecific(override val time: Int, val data: ByteArray) : MidiMetaEvent(time) {
        override fun equals(other: Any?): Boolean {
            if (this === other) return true
            if (other == null || this::class != other::class) return false

            other as SequencerSpecific

            if (time != other.time) return false
            if (!data.contentEquals(other.data)) return false

            return true
        }

        override fun hashCode(): Int {
            var result = time
            result = 31 * result + data.contentHashCode()
            return result
        }
    }

    /**
     * Unknown meta event.
     */
    class Unknown(override val time: Int, val metaType: Byte, val data: ByteArray) : MidiEvent(time)
}