/*******************************************************************************
* Copyright 2007-2013 See AUTHORS file.
 * 
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
* 
*   http://www.apache.org/licenses/LICENSE-2.0
* 
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
******************************************************************************/
package org.xmeta.ui.session;

import org.xmeta.util.UtilResource;

import java.io.Serializable;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Locale;

/**
 * 会话，代表的是用户。
 */
public interface Session {
	/**
	 * 从会话中获取一个属性。
	 *
	 * @param name 属性名
	 * @return 属性值，可能会为null
	 */
	Object getAttribute(Object name);

	/**
	 * 设置一个属性缓存到会话中。
	 *
	 * @param name 属性名
	 * @param value 属性值
	 */
	void setAttribute(Object name, Object value);

	Object removeAttribute(Object key);

	/**
	 * 获取Session的Locale。
	 *
	 * @return 会话Locale
	 */
	Locale getLocale();

	/**
	 * 设置会话的Locale。
	 *
	 * @param locale Locale
	 */
	void setLocale(Locale locale);

	/**
	 * 是否含有某角色。
	 *
	 * @param role 角色名
	 *
	 * @return 是否有该角色
	 */
	boolean hasRole(String role);

	/**
	 * 如果含有所有给定的角色返回true，否则返回false。
	 *
	 * @param roleIdentifiers 角色集合
	 * @return 是否包含所有角色
	 */
	boolean hasAllRoles(Collection<String> roleIdentifiers);

	/**
	 * 检查角色列表，并返回是否包含对应角色的信息数组。
	 *
	 * @param roleIdentifiers 角色列表
	 * @return 是否包含角色的数组
	 */
	boolean[] hasRoles(List<String> roleIdentifiers);

	/**
	 * 返回一个布尔数组，包含是否包含对应权限的值。
	 *
	 * @param permissions 权限列表
	 * @return 对应是否包含权限的数组
	 */
	boolean[] isPermitted(String... permissions);

	/**
	 * 返回一个布尔数组，包含是否包含对应权限的值。
	 *
	 * @param permissions 权限列表
	 * @return 对应是否包含权限的数组
	 */
	boolean[] isPermitted(List<String> permissions);

	/**
	 * 返回是否所有的权限都授权了。如果有一个没有授权就返回false。
	 *
	 * @param permissions 权限列表
	 * @return 是否都授权了
	 */
	boolean isPermittedAll(String... permissions);

	/**
	 * 返回是否所有的权限都授权了。如果有一个没有授权就返回false。
	 *
	 * @param permissions 权限列表
	 * @return 是否都授权了
	 */
	boolean isPermittedAll(Collection<String> permissions);

	/**
	 * 是否含有指定的权限。
	 *
	 * @param permission 权限
	 * @return 是否有权限
	 */
	boolean isPermitted(String permission);

	/**
	 * 获取多语言的资源。
	 *
	 * @return 多语言资源
	 */
	UtilResource getI18nResource();

	/**
	 * 设置多语言的资源。
	 *
	 * @param utilResource 多语言资源
	 */
	void setI18nResource(UtilResource utilResource);

	/**
	 * 返回首要的会话信息，比如登录后的用户信息。有可能为null。
	 *
	 * @param <T> 会话信息类型
	 * @return 会话信息
	 */
	<T> T getPrincipal();

	Serializable getId();

	Date getStartTimestamp();

	Date getLastAccessTime();

	long getTimeout();

	void setTimeout(long maxIdleTimeInMillis);

	String getHost();

	void touch();

	void stop();

	Collection<Object> getAttributeKeys();
}