/*******************************************************************************
* Copyright 2007-2013 See AUTHORS file.
 * 
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
* 
*   http://www.apache.org/licenses/LICENSE-2.0
* 
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
******************************************************************************/
package org.xmeta.util;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.text.DateFormat;
import java.text.DecimalFormat;
import java.text.FieldPosition;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.xmeta.Action;
import org.xmeta.ActionContext;
import org.xmeta.ActionException;
import org.xmeta.Thing;
import org.xmeta.World;

/**
 * 数据工具类。
 * 
 * @author <a href="mailto:zhangyuxiang@tom.com">zyx</a>
 *
 */
public class UtilData {
	public static final String VALUE_BLANK = "";
	public static final String VALUE_ONE = "1";
	public static final String VALUE_ZERO = "0";
	public static final String VALUE_FALSE = "false";
	public static final String VALUE_TRUE = "true";
	public static final String VALUE_CURRENT_DATE = "current_date";
	public static final String VALUE_NOW = "now";
	
	public static final String TYPE_STRING = "string";
	public static final String TYPE_INT = "int";
	public static final String TYPE_INTEGER = "integer";
	public static final String TYPE_LONG = "long";
	public static final String TYPE_FLOAT = "float";
	public static final String TYPE_DOUBLE = "double";
	public static final String TYPE_SHORT = "short";
	public static final String TYPE_BYTE = "byte";
	public static final String TYPE_BIGDECIMAL = "bigdecimal";
	public static final String TYPE_BIGINTEGER = "biginteger";
	public static final String TYPE_BOOLEAN = "boolean";
	public static final String TYPE_DATE = "date";
	public static final String TYPE_TIME = "time";
	public static final String TYPE_DATETIME = "datetime";
	public static final String TYPE_BYTES = "bytes";
	public static final String TYPE_OBJECT = "object";
	
	/**
	 * 如果str和matchs中任何一个匹配返回true，如果str=null或者matchs=null，返回false。
	 * 
	 * @param str 字符串
	 * @param matchs 匹配正则表达式
	 * @return 是否匹配
	 */
	public static boolean equalsOne(String str, String[] matchs){
		if(str == null || matchs == null){
			return false;
		}
		
		for(String match : matchs){
			if(match.equals(str)){
				return true;
			}
		}
		
		return false;
	}
	
	/**
	 * 把数据对象转换为字符串类型。
	 * 
	 * @param value 值
	 * @param pattern 格式
	 * @return 结果
	 */
	public static String format(Object value, String pattern){
		if(value == null) return VALUE_BLANK;
		
		if(value instanceof Number){
			DecimalFormat df = null;
			if(pattern == null || VALUE_BLANK.equals(pattern)){
				df = new DecimalFormat("#.####################");
			}else{
				df = new DecimalFormat(pattern);
			}
			
			return df.format(value, new StringBuffer(), new FieldPosition(0)).toString();
		}else if(value instanceof Date){
			SimpleDateFormat sf = null;
			if(pattern == null || VALUE_BLANK.equals(pattern)){
				sf = new SimpleDateFormat("yyyy-MM-dd");
			}else{
				sf = new SimpleDateFormat(pattern);
			}
			
			return sf.format(value);			
		}else{
			return value.toString();
		}
	}
	
	public static Object parse(String value, String valueType, String pattern) throws ParseException{
		if(value == null) return null;
		
		String type = valueType;
		if(type != null){
			type = type.toLowerCase();
		}else{
			type = TYPE_STRING;
		}
		
		if(VALUE_BLANK.equals(type) || TYPE_STRING.equals(type)){
			return value;
		}
		
		if(VALUE_BLANK.equals(value)) return null;
		
		if(TYPE_INTEGER.equals(type) || TYPE_LONG.equals(type) || TYPE_FLOAT.equals(type) 
				|| TYPE_DOUBLE.equals(type) || TYPE_SHORT.equals(type)
	            || TYPE_BYTE.equals(type)){
			DecimalFormat df;
			if(pattern == null || VALUE_BLANK.equals(pattern)){
				df = new DecimalFormat("#.####################");
			}else{
				df = new DecimalFormat(pattern);
			}
			
			Number n = df.parse(value);
			
			if(TYPE_INTEGER.equals(type)){
				return n.intValue();
			}
			
			if(TYPE_LONG.equals(type)){
				return n.longValue();
			}
			
			if(TYPE_FLOAT.equals(type)){
				return n.floatValue();
			}
			
			if(TYPE_DOUBLE.equals(type)){
				return n.doubleValue();
			}
			
			if(TYPE_SHORT.equals(type)){
				return n.shortValue();
			}
			
			if(TYPE_BYTE.equals(type)){
				return n.byteValue();
			}
			
			return null;
		}
		
		if(TYPE_BIGDECIMAL.equals(type)){
			return new BigDecimal(value);
		}
		
		if(TYPE_BIGINTEGER.equals(type)){
			return new BigInteger(value);
		}
		
		if(TYPE_BOOLEAN.equals(type)){
			return !VALUE_FALSE.equalsIgnoreCase(value) && !VALUE_ZERO.equals(value);
		}
		
		if(TYPE_DATE.equals(type) || TYPE_TIME.equals(type) || TYPE_DATETIME.equals(type)){
			if(VALUE_CURRENT_DATE.equals(value) || VALUE_NOW.equals(value)) return new Date();
			
			DateFormat sf = null;
			if(pattern == null || VALUE_BLANK.equals(pattern)){
				if(value.length() == 10){
					sf = new SimpleDateFormat("yyyy-MM-dd");
				}else if(TYPE_DATE.equals(type)){
					sf = SimpleDateFormat.getDateInstance();
				}else if(TYPE_TIME.equals(type)){
					sf = SimpleDateFormat.getTimeInstance();
				}else{
					sf = SimpleDateFormat.getDateTimeInstance();
				}
			}else{				
				sf = new SimpleDateFormat(pattern);
			}
			
			return sf.parse(value);
		}
		
		return value;
	}
	
	/**
	 * 添加指定的模型列表到已有的模型列表中，添加时如果存在第一个的描述者相同且名字相同的则不添加。
	 * 
	 * @param source 源模型列表
	 * @param forAdd 要添加的模型列表
	 * @param strict 是否严格判断，如果是严格判断那么校验两个模型必须相等，否则校验模型的描述者和名
	 * 
	 * @return 结果
	 */
	public static List<Thing> addToSource(List<Thing> source, List<Thing> forAdd, boolean strict){
		if(forAdd == null){
			return source;
		}
		
		if(source == null){
			source = new ArrayList<>();
		}
		
		for(Thing fadd : forAdd){
			addToSource(source, fadd, strict);
		}
		
		return source;
	}
	
	/**
	 * 添加指定的模型到源模型列表中。
	 * 
	 * @param source 源模型列表
	 * @param forAdd 需要添加的模型
	 * @param strict 是否严格判断，如果是严格判断那么校验两个模型必须相等，否则校验模型的描述者和名
	 * @return 模型列表
	 */
	public static List<Thing> addToSource(List<Thing> source, Thing forAdd, boolean strict){
		if(forAdd == null){
			return source;
		}
		
		if(source == null){
			source = new ArrayList<Thing>();
		}
		
		//Thing forAddDescriptor = forAdd.getDescriptors().get(0);
		boolean have = false;
		for(Thing src : source){
			if(src == forAdd){
				have = true;
				break;
			}//else if(//forAddDescriptor == src.getDescriptors().get(0) && 
			//		forAdd.getMetadata().getName().equals(src.getMetadata().getName())){
			//	have = true;
			//	break;				
			//}
		}
		
		if(!have){
			source.add(forAdd);
		}
		
		return source;
	}
	
	public static BigDecimal getBigDecimal(Object v, BigDecimal defaultValue){		
    	if(v == null || VALUE_BLANK.equals(v)){
    		return null;
    	}else if(v instanceof BigDecimal){
    		return (BigDecimal) v;
    	}else if(v instanceof String){
    		return new BigDecimal((String) v);
    	}else if(v instanceof Integer){
    		return new BigDecimal((Integer) v);
    	}else if(v instanceof Boolean){
    		return new BigDecimal(((Boolean) v) ? 1 : 0);
    	}else if(v instanceof Byte){
    		return new BigDecimal((Byte) v);
    	}else if(v instanceof BigInteger){
    		return new BigDecimal((BigInteger) v);
    	}else if(v instanceof Double){
    		return BigDecimal.valueOf((Double) v);
    	}else if(v instanceof Float){
    		return BigDecimal.valueOf((Float) v);
    	}else if(v instanceof Short){
    		return new BigDecimal((Short) v);
    	}
    	
    	return defaultValue;
	}
	
	public static BigInteger getBigInteger(Object v, BigInteger defaultValue){    	
    	if(v == null || VALUE_BLANK.equals(v)){
    		return null;
    	}else if(v instanceof BigInteger){
    		return (BigInteger) v;
    	}else if(v instanceof String){
    		return new BigInteger((String) v);
    	}else if(v instanceof Integer){
    		return new BigInteger(v.toString());
    	}else if(v instanceof Boolean){
    		return new BigInteger(((Boolean) v) ? VALUE_ONE : VALUE_ZERO);
    	}else if(v instanceof Byte){
    		return new BigInteger(v.toString());
    	}else if(v instanceof BigDecimal){
    		return new BigInteger(v.toString());
    	}else if(v instanceof Double){
    		return new BigInteger(v.toString());
    	}else if(v instanceof Float){
    		return new BigInteger(v.toString());
    	}else if(v instanceof Short){
    		return new BigInteger(v.toString());
    	}
    	
    	return defaultValue;
    }
	
	public static byte getByte(Object v, byte defaultValue){    	
    	if(v == null || VALUE_BLANK.equals(v)){
    		return defaultValue;
    	}else if(v instanceof Number){
    		return ((Number) v).byteValue();
    	}else if(v instanceof String){
    		try{
    			return Byte.parseByte((String) v);
    		}catch(Exception e){
    			if("true".equals(v) || "TRUE".equals(v)){
    				return 1;
    			}else{
    				return 0;
    			}
    		}
    	}else if(v instanceof Boolean){
    		return (byte) (((Boolean) v) ? 1 : 0);
    	}else if(v instanceof BigInteger){
    		return ((BigInteger) v).byteValue();
    	}else if(v instanceof BigDecimal){
    		return ((BigDecimal) v).byteValue();
    	}
    	
    	return defaultValue;
    }
	
	public static byte[] getBytes(Object v, byte[] defaultValue){
    	if(v == null || VALUE_BLANK.equals(v)){
    		return null;
    	}else if (v instanceof byte[]){
    		return (byte[]) v;
    	}else if(v instanceof String) {
    		return ((String) v).getBytes();
    	}else if(v instanceof java.io.Serializable){
    		ByteArrayOutputStream bout = new ByteArrayOutputStream();
    		ObjectOutputStream oout;
			try {
				oout = new ObjectOutputStream(bout);
				oout.writeObject(v);
			} catch (IOException ignored) {
			}
			
    		return bout.toByteArray();
    	}
    	
    	return defaultValue;
    }
	
	public static char getChar(Object v, char defaultValue){
    	if(v == null || VALUE_BLANK.equals(v)){
    		return defaultValue;
    	}else if(v instanceof Number){
    		return (char) ((Number) v).intValue();
    	}else if(v instanceof BigInteger){
    		return (char) ((BigInteger) v).intValue();
    	}else if(v instanceof String){
    		return (char) new Integer((String) v).intValue();
    	}else if(v instanceof Boolean){
    		return (char) (((Boolean) v) ? 1 : 0);
    	}else if(v instanceof BigDecimal){
    		return (char) ((BigDecimal) v).intValue();
    	}
    	
    	return defaultValue;
    }
	
	public static Date getDate(Object v, Date defaultValue){    	
    	if(v == null || VALUE_BLANK.equals(v)){
    		return null;
    	}else if(v instanceof Date){
    		return (Date) v;
    	}else if(v instanceof Long){
    		return new Date((long) v);
		}else if(v instanceof String){
    		String dateStr = (String) v;    		
    		SimpleDateFormat sf = getSimpleDateFormat(dateStr);    		


    		try {
    			if(sf != null) {
					return sf.parse((String) v);
				}
			} catch (ParseException ignored) {
			}
			Date date = getDateDefault((String) v);
			if(date != null){
				return date;
			}
    	}
    	
    	return defaultValue;
    }
	
	public static Date getDateDefault(String defaultValue){
		  //日期的默认值
        defaultValue = defaultValue.toLowerCase();
        String dateStr = defaultValue;
        String numberStr = "";
        int index = defaultValue.indexOf("+");
        if(index != -1){
            dateStr = defaultValue.substring(0, index).trim();
            numberStr = defaultValue.substring(index).trim();
        }else{
            index = defaultValue.indexOf("-");
            if(index != -1){
                dateStr = defaultValue.substring(0, index).trim();
                numberStr = defaultValue.substring(index).trim();
            }
        }
 
        Date date;
		switch (dateStr) {
			case "now":
			case "sysdate":
				date = new Date();
				break;
			case "tomorrow":
				date = UtilDate.getTomorrow();
				break;
			case "yesterday":
				date = UtilDate.getYesterday();
				break;
			case "weekstart":
				date = UtilDate.getWeekStart();
				break;
			case "weekend":
				date = UtilDate.getWeekEnd();
				break;
			case "monthstart":
				date = UtilDate.getMonthStart();
				break;
			case "monthend":
				date = UtilDate.getMonthEnd();
				break;
			case "yearstart":
				date = UtilDate.getYearStart();
				break;
			case "yearend":
				date = UtilDate.getYearEnd();
				break;
			default:
				date = new Date();
				try {
					double d = Double.parseDouble(dateStr);
					date = UtilDate.getDate(date, d);
				} catch (Exception ignored) {
				}
				break;
		}
        if(!numberStr.equals("")){
            try{
                double d = (Double) OgnlUtil.getValue(numberStr, null);
                //log.info("d=" + d);
                date = UtilDate.getDate(date, d);
            }catch(Exception e){
                //log.info("error", e);
            }
        }
        
        return date;
	}
	
	public static Date getDate(Object v, Date defaultValue, String pattern){    	
    	if(v == null || VALUE_BLANK.equals(v)){
    		return null;
    	}else if(v instanceof Date){
    		return (Date) v;
    	}else if(v instanceof String){
    		SimpleDateFormat sf;
    		String value = v.toString().replace("T"," ");
    		if(pattern == null || pattern.equals(VALUE_BLANK)){
    			sf = getSimpleDateFormat(value);   
    		}else{
    			sf = new SimpleDateFormat(pattern);
    		}
    		    	  	
    		try {
    			if(sf != null) {
					return sf.parse(value);
				}
			} catch (Exception ignored) {
			}
    		
    		try {
    			//可能是long类型
    			long time = Long.parseLong(value);
    			return new Date(time);
    		}catch(Exception ignored) {
    		}
    	}else if(v instanceof Long) {
    		return new Date((Long) v);
    	}
    	
    	return defaultValue;
    }
	
	public static SimpleDateFormat getSimpleDateFormat(String dateStr){
		if(dateStr.length() == 10){
			return new SimpleDateFormat("yyyy-MM-dd");
		}else if(dateStr.length() == 19){
			return new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
		}else if(dateStr.length() == 23){
			return new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS");
		}else if(dateStr.length() == 12){
			return new SimpleDateFormat("HH:mm:ss.SSS");
		}else if(dateStr.length() == 8){
			return new SimpleDateFormat("HH:mm:ss");
		}else if(dateStr.length() == 5){
			return new SimpleDateFormat("HH:mm");
		}if(dateStr.length() == 7){
			return new SimpleDateFormat("yyyy-MM");
		}else{
			return null;
		}
	}
	
	public static double getDouble(Object v, double defaultValue){    	
    	if(v == null || VALUE_BLANK.equals(v)){
    		return defaultValue;
    	}else if(v instanceof Number){
    		return ((Number) v).doubleValue();
    	}else if(v instanceof BigInteger){
    		return ((BigInteger) v).doubleValue();
    	}else if(v instanceof String){
    		return Double.parseDouble((String) v);
    	}else if(v instanceof Boolean){
    		return (byte) (((Boolean) v) ? 1 : 0);
    	}else if(v instanceof BigDecimal){
    		return ((BigDecimal) v).doubleValue();
    	}
    	
    	return defaultValue;
    }
	
	public static float getFloat(Object v, float defaultValue){    	
    	if(v == null || VALUE_BLANK.equals(v)){
    		return defaultValue;
    	}else if(v instanceof Number){
    		return ((Number) v).floatValue();
    	}else if(v instanceof BigInteger){
    		return ((BigInteger) v).floatValue();
    	}else if(v instanceof String){
    		return Float.parseFloat((String) v);
    	}else if(v instanceof Boolean){
    		return (byte) (((Boolean) v) ? 1 : 0);
    	}else if(v instanceof BigDecimal){
    		return ((BigDecimal) v).floatValue();
    	}
    	
    	return defaultValue;
    }
	
	public static long getLong(Object v, long defaultValue){    	
    	if(v == null || VALUE_BLANK.equals(v)){
    		return defaultValue;
    	}else if(v instanceof Long){
    		return (Long) v;
    	}else if(v instanceof Number){
    		return  ((Number) v).longValue();
    	}else if(v instanceof BigInteger){
    		return ((BigInteger) v).longValue();
    	}else if(v instanceof String){ 
    		return Long.parseLong((String) v);
    	}else if(v instanceof Integer){
    		return ((Integer) v).longValue();
    	}else if(v instanceof Boolean){
    		return (byte) (((Boolean) v) ? 1 : 0);
    	}else if(v instanceof BigDecimal){
    		return ((BigDecimal) v).longValue();
    	}
    	
    	return defaultValue;
    }
	
	public static int getInt(Object v, int defaultValue){    	
    	if(v == null || VALUE_BLANK.equals(v)){
    		return defaultValue;
    	}else if(v instanceof Integer){
    		return (Integer) v;
    	}else if(v instanceof Number){
    		return  ((Number) v).intValue();
    	}else if(v instanceof BigInteger){
    		return ((BigInteger) v).intValue();
    	}else if(v instanceof String){
    		try{
    			return Integer.parseInt((String) v);
    		}catch(Exception e){
    			String vs = ((String) v).toLowerCase();
    			if("false".equals(vs)){
    				return 0;
    			}else if("true".equals(vs)){
    				return 1;
    			}else{
    				throw new ActionException("Get int error", e);
    			}
    		}
    	}else if(v instanceof Boolean){
    		return (int) (((Boolean) v) ? 1 : 0);
    	}else if(v instanceof BigDecimal){
    		return ((BigDecimal) v).intValue();
    	}
    	
    	return defaultValue;
    }
	
	public static short getShort(Object v, short defaultValue){    	
    	if(v == null || VALUE_BLANK.equals(v)){
    		return defaultValue;
    	}else if(v instanceof Short){
    		return (Short) v;
    	}else if(v instanceof Number){
    		return ((Number) v).shortValue();
    	}else if(v instanceof BigInteger){
    		return (short) ((BigInteger) v).intValue();
    	}else if(v instanceof String){
    		return Short.parseShort((String) v);
    	}else if(v instanceof Boolean){
    		return (short) (((Boolean) v) ? 1 : 0);
    	}else if(v instanceof BigDecimal){
    		return (short) ((BigDecimal) v).shortValue();
    	}    	
    	return defaultValue;
    }
	
	public static boolean getBoolean(Object v, boolean defaultValue){
    	if(v instanceof Boolean){
    		return ((Boolean) v).booleanValue();
    	}else if(v instanceof String){
    		String value = (String) v;
	    	if(VALUE_TRUE.equals(value)){
	    		return true;
	    	}else if(VALUE_ONE.equals(value)){
	    		return true;
	    	}else{
	    		return false;
	    	}
    	}else if(v instanceof Number){
    		if(((Number) v).doubleValue() == 0){
    			return false;
    		}else{
    			return true;
    		}
    	}else if(v != null){
    		return true;
    	}else{
    		return defaultValue;
    	}
    }
	
	public static String getString(Object v, String defaultValue){    	
    	if(v != null){
    		if(v instanceof String){
    			return (String) v;
    		}else{
    			return v.toString();
    		}
    	}
    	
    	return defaultValue;
    }
	
	/** 
	 * 按照字节、千字节和兆返回大小的值。
	 * 
	 * @param size 大小
	 * @return 字符串
	 */
	public static String getSizeInfo(double size) {
		DecimalFormat sf = new DecimalFormat("#.##");
		if (size < 1024) {
			return String.valueOf(size) + "B";
		} else if (size < 1024 * 1024) {
			return sf.format(size / 1024) + "KB";
		} else {
			return sf.format(size / 1024 / 1024) + "MB";
		}
	}
	
	/**
	 * 将"00 01 02"形式的字符串转成byte[]
	 * 
	 * @param hex 16进制字符串
	 * @return 字节数组
	 */
	public static byte[] hexStringToByteArray(String hex) {
		if (hex == null) {
			return null;
		}

		int stringLength = hex.length();
		if (stringLength % 2 != 0) {
			throw new IllegalArgumentException("Hex String must have even number of characters!");
		}

		byte[] result = new byte[stringLength / 2];

		int j = 0;
		for (int i = 0; i < result.length; i++) {
			char hi = Character.toLowerCase(hex.charAt(j++));
			char lo = Character.toLowerCase(hex.charAt(j++));
			result[i] = (byte) ((Character.digit(hi, 16) << 4) | Character.digit(lo, 16));
		}

		return result;
	}
	
	/**
	 * 二进制转字符串。
	 * 
	 * @param bytes 字节数组
	 * @return 16进制字符串
	 */
	 public static String bytesToHexString(byte[] bytes) {
		char[] buf = new char[bytes.length * 2];
		int radix = 1 << 4;
		int mask = radix - 1;
		for (int i = 0; i < bytes.length; i++) {
			buf[2 * i] = hexDigit[(bytes[i] >>> 4) & mask];
			buf[2 * i + 1] = hexDigit[bytes[i] & mask];
		}

		return new String(buf);
	}
	 
	 /**
		 * 转换数据类型。
		 * 
		 * @param sourceValue 原值
		 * @param targetType 目标值
		 * @param pattern 格式
		 * @param patternType 格式类型
		 * @param patternAction 动作
		 * @return 转换后的结果
		 * @throws ParseException 异常
		 */
	public static Object transfer(Object sourceValue, String targetType, String pattern, String patternType, String patternAction) throws ParseException{
		if (pattern != null  && !"".equals(pattern) && sourceValue != null && !"".equals(sourceValue)) {
			if ("parse".equals(patternAction)) {
				return UtilData.parse(sourceValue.toString(), targetType, pattern);
			} else {
				sourceValue = UtilData.format(sourceValue, pattern);
			}
		}

		// 数据转换
		Object targetValue = sourceValue;
		if (targetType != null && !"".equals(targetType)) {
			switch (targetType) {
				case "byte":
					targetValue = UtilData.getByte(sourceValue, (byte) 0);
					break;
				case "short":
					targetValue = UtilData.getShort(sourceValue, (short) 0);
					break;
				case "int":
					targetValue = (int) UtilData.getLong(sourceValue, 0);
					break;
				case "long":
					targetValue = UtilData.getLong(sourceValue, 0);
					break;
				case "float":
					targetValue = UtilData.getFloat(sourceValue, 0);
					break;
				case "double":
					targetValue = UtilData.getDouble(sourceValue, 0);
					break;
				case "boolean":
					targetValue = UtilData.getBoolean(sourceValue, false);
					break;
				case "byte[]":
					if (sourceValue instanceof String) {
						targetValue = UtilString
								.hexStringToByteArray((String) sourceValue);
					} else if (sourceValue instanceof byte[]) {
						targetValue = sourceValue;
					} else {
						targetValue = null;
					}
					break;
				case "hex_byte[]":
					if (sourceValue instanceof byte[]) {
						targetValue = UtilString.toHexString((byte[]) sourceValue);
					} else if (sourceValue instanceof String) {
						targetValue = sourceValue;
					} else {
						targetValue = null;
					}
					break;
				default:
					targetValue = sourceValue;
					break;
			}
		}
		
		return targetValue;
	}
	
	/**
	 * 返回通过属性定义的对象，首先使用UtilData获取，如果没有从actionContext中获取。
	 * 
	 * @param thing 模型
	 * @param attributeName 属性名
	 * @param actionContext 变量上下文
	 * @return 对象
	 */
	public static Object getObject(Thing thing, String attributeName, ActionContext actionContext) {
		Object obj = UtilData.getData(thing, attributeName, actionContext);
		if(obj instanceof String){
			obj = actionContext.get((String) obj);
		}
		
		return obj;
	}
	
	@SuppressWarnings("unchecked")
	public static <T> T get(Thing thing, String attributeName, ActionContext actionContext){
		Object obj = UtilData.getData(thing, attributeName, actionContext);
		if(obj instanceof String){
			obj = actionContext.get((String) obj);
		}
		
		return (T) obj;
	}
	
	@SuppressWarnings("unchecked")
	public static <T> T getObjectByType(Thing thing, String attributeName, Class<T> t, ActionContext actionContext){
		Object obj = getObject(thing, attributeName, actionContext);
		if(t.isInstance(obj)){
			return (T) obj;
		}else{
			return null;
		}
	}
		
	/**
	 * 根据模型的属性返回指定的模型。
	 * 
	 * @param thing 模型
	 * @param attributeName 属性名
	 * @param actionContext 变量上下文
	 * @return 模型，如果不存在返回null
	 */
	public static Thing getThing(Thing thing, String attributeName, ActionContext actionContext) {
		Object obj = UtilData.getData(thing, attributeName, actionContext);
		if(obj instanceof Thing){
			return (Thing) obj;
		}else if(obj instanceof String){
			return World.getInstance().getThing((String) obj);
		}else if(obj != null){
			return World.getInstance().getThing(obj.toString());
		}else{
			return null;
		}
	}
	
	/**
	 * 先从模型指定的属性上获取模型，如果不存在从子节点的路径上获取。
	 * 
	 * @param thing 模型
	 * @param attributeName 属性名
	 * @param childThingPath 子模型节点
	 * @param actionContext 变量上下文
	 * @return 要返回的额模型，如果不存在返回null
	 */
	public static Thing getThing(Thing thing, String attributeName, String childThingPath, ActionContext actionContext) {
		Thing th = getThing(thing, attributeName, actionContext);
		if(th == null && childThingPath != null){
			return thing.getThing(childThingPath);
		}else{
			return th;
		}
	}
	
	/**
	 * 通过模型的属性获取数据。
	 * 
	 * @param thing 模型
	 * @param attributeName 属性名
	 * @param actionContext 变量上下文
	 * 
	 * @return 结果
	 */
	public static Object getData(Thing thing, String attributeName, ActionContext actionContext) {
		Object value = thing.get(attributeName);
		if(value instanceof String){
			String str = (String) value;
			if(str.startsWith("var:")){
				return actionContext.get(str.substring(4));
			}else if(str.startsWith("ognl:")){
				return OgnlUtil.getValue(thing, attributeName, actionContext);
			}else if(str.startsWith("thing:")){
				String thingPath = str.substring(6);
				return World.getInstance().getThing(thingPath);
			}else if(str.startsWith("action:")){
				String thingPath = str.substring(7);
				Action action =  World.getInstance().getAction(thingPath);
				if(action != null){
					return action.run(actionContext);
				}else{
					return null;
				}
			}else{				
				if("".equals(str)){
					return null;
				}else{
					return UtilString.getString(str, actionContext);
				}
				
				//return str;
			}
		}
		
		return value;
	}
	
	public static Object getData( String value, ActionContext actionContext) {
		if(value.startsWith("var:")){
			return actionContext.get(value.substring(4));
		}else if(value.startsWith("ognl:")){
			return OgnlUtil.getValue(value.substring(5), actionContext);
		}else if(value.startsWith("thing:")){
			String thingPath = value.substring(6);
			return World.getInstance().getThing(thingPath);
		}else if(value.startsWith("int:")) {
			return Integer.parseInt(value.substring(4));
		}else if(value.startsWith("byte:")) {
			return Byte.parseByte(value.substring(5));
		}else if(value.startsWith("float:")) {
			return Float.parseFloat(value.substring(6));
		}else if(value.startsWith("double:")) {
			return Double.parseDouble(value.substring(7));
		}else if(value.startsWith("string:")) {
			return value.substring(7);
		}else if(value.startsWith("long:")) {
			return Long.parseLong(value.substring(5));
		}else if(value.startsWith("bool:")) {
			return Boolean.parseBoolean(value.substring(5));
		}else if(value.startsWith("char:")) {
			return (char) Short.parseShort(value.substring(5));
		}else if(value.startsWith("short:")) {
			return Short.parseShort(value.substring(6));
		}else{
			if("".equals(value)){
				return null;
			}
			
			return UtilString.getString(value, actionContext);
			//return str;
		}
	}
	
	public static boolean isTrue(Object condition){
		return isTrue(condition, false);
	}
	
	/**
	 * 判断condition是否为true， 当condition==null时返回传入的默认值。
	 * 
	 * @param condition 条件
	 * @param defaultValue 默认值
	 * @return 是否为真
	 */
	public static boolean isTrue(Object condition, boolean defaultValue){
		boolean ok = false;
		if(condition != null){
			if(condition instanceof Boolean){
				ok = (Boolean) condition;
			}else if(condition instanceof String){
				String str = ((String) condition).toLowerCase().trim();
				ok = "true".equals(str) | "1".equals(str);
			}else if(condition instanceof Number){
				Number n = (Number) condition;
				ok = n.intValue() == 1;
			}else{
				ok = true;
			}
		}else{
			ok = defaultValue;
		}
		
		return ok;
	}
	
	public static String getString(Thing thing, String attributeName, ActionContext actionContext) {
		return UtilString.getString(thing, attributeName, actionContext);
	}
	 
	public static void resetAttributeByType(Thing thing, String name, String type){
		if ("int".equals(type)) {
			try {
				thing.getAttributes().put(name, thing.getInt(name));
			} catch (Exception e) {
				e.printStackTrace();
			}
		} else if ("long".equals(type)) {
			try {
				thing.getAttributes().put(name, thing.getLong(name));
			} catch (Exception e) {
				e.printStackTrace();
			}
		} else if ("double".equals(type)) {
			try {
				thing.getAttributes().put(name, thing.getDouble(name));
			} catch (Exception e) {
				e.printStackTrace();
			}
		} else if ("float".equals(type)) {
			try {
				thing.getAttributes().put(name, thing.getFloat(name));
			} catch (Exception e) {
				e.printStackTrace();
			}
		} else if ("bigDecimal".equals(type)) {
			try {
				thing.getAttributes().put(name, thing.getBigDecimal(name));
			} catch (Exception e) {
				e.printStackTrace();
			}
		} else if ("bigInteger".equals(type)) {
			try {
				thing.getAttributes().put(name, thing.getBigInteger(name));
			} catch (Exception e) {
				e.printStackTrace();
			}
		} else if ("boolean".equals(type)) {
			try {
				thing.getAttributes().put(name, thing.getBoolean(name));
			} catch (Exception e) {
				e.printStackTrace();
			}
		} else if ("byte".equals(type)) {
			try {
				thing.getAttributes().put(name, thing.getByte(name));
			} catch (Exception e) {
				e.printStackTrace();
			}
		} else if ("bytes".equals(type)) {
			try {
				thing.getAttributes().put(name, thing.getBytes(name));
			} catch (Exception e) {
				e.printStackTrace();
			}
		} else if ("char".equals(type)) {
			try {
				thing.getAttributes().put(name, thing.getChar(name));
			} catch (Exception e) {
				e.printStackTrace();
			}
		} else if ("short".equals(type)) {
			try {
				thing.getAttributes().put(name, thing.getShort(name));
			} catch (Exception e) {
				e.printStackTrace();
			}
		} else if ("date".equals(type) || "datetime".equals(type) || "time".equals(type)) {
			try {
				thing.getAttributes().put(name, thing.getDate(name));
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
	}
	
    /** A table of hex digits */
    private static final char[] hexDigit = {
	'0','1','2','3','4','5','6','7','8','9','A','B','C','D','E','F'
    };
}
