/*******************************************************************************
* Copyright 2007-2013 See AUTHORS file.
 * 
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
* 
*   http://www.apache.org/licenses/LICENSE-2.0
* 
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
******************************************************************************/
package org.xmeta;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

/**
 * 动作上下文中栈点保存的就是Bindings，Bindings是一个Map，用于保存变量和一些函数调用相关的数据。
 * 
 * @author <a href="mailto:zhangyuxiang@tom.com">zyx</a>
 *
 */
public class Bindings extends HashMap<String, Object>{
	private static final long serialVersionUID = 1L;
	
	/** 调用动作的对象，一般是actionContext压入栈时的对象 */
	private Object caller = null;
	/**
	 * 调用者执行的方法名称。
	 */
	private String callerMethod = null;

	/** 模型动作上下文的动作上下文 */
	private Map<Thing, ActionContext> contexts = null;//new HashMap<Thing, ActionContext>();
	
	/** Bindings可以设置动作的上下文，2016-10-26 */
	private ActionListenerProvider actionListenerProvider = null;
		
	public World world = World.getInstance();
	
	/** 是否关闭全局模型动作监听器，如果关闭子函数也都关闭 */
	public boolean disableGlobalContext = false;
	
	/** 额外的用于存储多个key的数据 */
	private static Map<Object, Object> dataMap = null; 
	
	/** 
	 * 是否是变量范围的标志，通常在函数调用（动作执行）、条件、循环等处设置为true，以便划分一个
	 * 变量范围，一般新的变量都保存到此变量范围中。
	 * 
	 * 为了实现类似其他语言如java的变量范围的设置。
	 */
	private boolean isVarScopeFlag = false;
	
	/**
	 * 是否是参数Bindings。
	 */
	private boolean parameterScope = false;
	
	/**
	 * 构造一个空的StackMap。
	 *
	 */
	public Bindings(){
		super();	
	}

	public boolean isVarScopeFlag(){
		return isVarScopeFlag;
	}
	
	/**
	 *　标记为本地变量栈层。
	 */
	public void setVarScopeFlag(){
		isVarScopeFlag = true;
	}
	
	public Object getCaller() {
		return caller;
	}

	public void setCaller(Object caller, String method) {
		this.caller = caller;
		this.callerMethod = method;
	}	
	
	public Map<Thing, ActionContext> getContexts(){
		if(contexts == null){
			contexts = new HashMap<Thing, ActionContext>();
		}
		return contexts;
	}
	
	public String getCallerMethod(){
		return callerMethod;
	}

	@Override
	public int hashCode() {
		int h = 0;
        Iterator<Entry<String,Object>> i = entrySet().iterator();
        while (i.hasNext()){
        	Entry<String,Object> obj = i.next();
        	if(obj.getValue() instanceof Bindings || obj.getValue() instanceof ActionContext){
        		//因为可能会引起递归，所以屏蔽这些类型了
        		continue;
        	}
        	
        	if(obj.getValue() != this){
        		h += obj.hashCode();
        	}
        }
        return h;
	}
	
	/**
	 * 设置动作监听器模型。子动作的执行都会触发监听器。。
	 * 
	 * @param actionListener 动作监听器模型
	 */
	public void setActionListener(Thing actionListener){
		this.actionListenerProvider = new ThingActionListenerProvider(actionListener);
	}

	/**
	 * 设置动作监听器提供者。如果要清除，设置为null。
	 *
	 * @param actionListenerProvider 动作监听器提供者
	 */
	public void setActionListenerProvider(ActionListenerProvider actionListenerProvider) {
		this.actionListenerProvider = actionListenerProvider;
	}

	public ActionListenerProvider getActionListenerProvider() {
		return actionListenerProvider;
	}

	public String toString(){
		//return super.toString();
		String str = "";
		if(caller != null){
			str = str + caller.getClass().getSimpleName() + ": " + (callerMethod != null ? callerMethod : "") + ", ";
		}
		
		str = str + "var count：" + this.size();
		return str;
	}

	public boolean isParameterScope() {
		return parameterScope;
	}

	public void setParameterScope(boolean parameterScope) {
		this.parameterScope = parameterScope;
	}
	
	public synchronized Map<Object, Object> getDataMap(){
		if(dataMap == null) {
			dataMap = new HashMap<Object, Object>();
		}
		
		return dataMap;
	}
	
	@SuppressWarnings("unchecked")
	public void setData(Object value, Object ... keys) {
		Map<Object, Object> m = getDataMap();
		for(int i=0; i<keys.length - 1; i++) {
			Map<Object, Object> m1 = (Map<Object, Object>) m.get(keys[i]);
			if(m1 == null) {
				m1 = new HashMap<Object, Object>();
				m.put(keys[i], m1);
			}
			
			m = m1;
		}
		
		m.put(keys[keys.length - 1], value);
	}
	
	@SuppressWarnings("unchecked")
	public Object getData(Object ... keys) {
		Map<Object, Object> m = getDataMap();
		for(int i=0; i<keys.length - 1; i++) {
			Map<Object, Object> m1 = (Map<Object, Object>) m.get(keys[i]);
			if(m1 == null) {
				return null;
			}else {			
				m = m1;
			}
		}
		
		return m.get(keys[keys.length -1]);
	}
	
	@SuppressWarnings("unchecked")
	public Object removeData(Object ... keys) {
		Map<Object, Object> m = getDataMap();
		for(int i=0; i<keys.length - 1; i++) {
			Map<Object, Object> m1 = (Map<Object, Object>) m.get(keys[i]);
			if(m1 == null) {
				return null;
			}else {			
				m = m1;
			}
		}
		
		return m.remove(keys[keys.length -1]);
	}
	
	@SuppressWarnings("unchecked")
	public void clearData(Object ... keys) {
		Map<Object, Object> m = getDataMap();
		for(int i=0; i<keys.length - 1; i++) {
			Map<Object, Object> m1 = (Map<Object, Object>) m.get(keys[i]);
			if(m1 == null) {
				return;
			}else {			
				m = m1;
			}
		}
		
		m.clear();
	}
}