/*******************************************************************************
* Copyright 2007-2013 See AUTHORS file.
 * 
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
* 
*   http://www.apache.org/licenses/LICENSE-2.0
* 
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
******************************************************************************/
package org.xmeta.ui.session;

import java.util.ArrayList;
import java.util.List;
import java.util.Stack;
import java.util.concurrent.Callable;

import org.xmeta.ActionContext;

/**
 * 会话管理者，表示的是UI交互中的会话，还需进一步设计。
 * 
 * 一个系统里可能会存在多种会话环境，比如HTTP，NETTY，RAP等等，判断当前属于哪一种会话的方法如下：
 * 1.是否设置了ThreadLocal&lt;SessionManager&gt;，如HTTP和NETTY的情况。
 * 2.遍历所有已注册的SessionManager，当有SessionManager.accept(actionContext)时使用。
 * 3.使用默认的defaultSessionManager。
 * 
 * @author zyx
 *
 */
public abstract class SessionManager {
	/**
	 * 默认的会话管理器。
	 */
	private static SessionManager defaultSessionManager;
	
	/**
	 * 会话管理器的列表。
	 */
	private static final List<SessionManager> sessionManagers = new ArrayList<>();
	
	/**
	 * 会话管理器Local。
	 */
	private static final ThreadLocal<Stack<SessionManager>> sessionManagerLocal = new ThreadLocal<>();

	/**
	 * 获取默认环境的会话，是本地会话，公用一个Session。
	 * 
	 * @param actionContext 上下文
	 * @return 会话
	 */
	public static Session getSession(ActionContext actionContext) {
		Session session = getSessionManager(actionContext).get(actionContext);
		if(session != null){
			return session;
		}else{
			return defaultSessionManager.get(actionContext);
		}
	}
	
	public static Session remove(ActionContext actionContext){
		return getSessionManager(actionContext).delete(actionContext);
	}
	
	public static void registSessionManager(SessionManager sessionManager) {		
		if(!sessionManagers.contains(sessionManager)) {
			sessionManagers.add(sessionManager);
		}
		//sessionManagers.put(env, sessionManager);
		//SessionManager.defaultSessionManager = sessionManager;
	}
	
	public static List<SessionManager> getSessionManagers(){
		return sessionManagers;
	}
	
	/**
	 * 根据环境返回会话管理器，如果不存在返回默认的会话管理器。
	 * 1.先从ThreadLocal中获取会话。
	 * 2.其次注册的会话管理器列表中判断，如果会话管理器接受上下文那么返回。
	 * 3.返回默认的会话管理器。
	 * 	
	 * @param actionContext 变量上下文
	 * @return 会话管理器
	 */
	public static SessionManager getSessionManager(ActionContext actionContext) {
		Stack<SessionManager> stackSessionManagers = sessionManagerLocal.get();
		if(stackSessionManagers != null && stackSessionManagers.size() > 0){
			return stackSessionManagers.peek();
		}
		
		for(SessionManager sm : sessionManagers) {
			if(sm.accept(actionContext)) {
				return sm;
			}
		}
		
		return getDefaultSessionManager();
	}

	/**
	 * 设置当前线程的会话管理器。
	 *
	 * 注意，必须在finally里中
	 *
	 * @param sessionManager 会话管理器
	 */
	public static void pushLocalSessionManager(SessionManager sessionManager) {
		Stack<SessionManager> sessionManagers = getLocalSessionManagerStack();

		sessionManagers.push(sessionManager);
	}

	private static Stack<SessionManager> getLocalSessionManagerStack(){
		Stack<SessionManager> sessionManagers = sessionManagerLocal.get();
		if(sessionManagers == null){
			sessionManagers = new Stack<>();
			sessionManagerLocal.set(sessionManagers);
		}

		return sessionManagers;
	}

	/**
	 * 包装一个Runnable，当它执行时使用当前线程的最后的一个SessionManager。
	 *
	 * @param runnable Runnable
	 * @return 包装后的Runnable
	 */
	public static Runnable associateWith(Runnable runnable){
		Stack<SessionManager> sessionManagers = getLocalSessionManagerStack();
		final SessionManager sessionManager = sessionManagers.size() > 0 ? sessionManagers.peek() : null;
		if(sessionManager == null){
			return runnable;
		}else {
			return () -> {
				try {
					SessionManager.pushLocalSessionManager(sessionManager);
					runnable.run();
				} finally {
					SessionManager.popLocalSessionManager();
				}
			};
		}
	}

	/**
	 * 包装一个Callable，当它运行时使用当前线程的最后一个SessionManager。
	 *
	 * @param callable Callable
	 * @return 包装后Callable
	 */
	public static Callable<?> associateWith(Callable<?> callable){
		Stack<SessionManager> sessionManagers = getLocalSessionManagerStack();
		final SessionManager sessionManager = sessionManagers.size() > 0 ? sessionManagers.peek() : null;
		if(sessionManager == null){
			return callable;
		}else {
			return () -> {
				try {
					SessionManager.pushLocalSessionManager(sessionManager);
					return callable.call();
				} finally {
					SessionManager.popLocalSessionManager();
				}
			};
		}
	}

	/**
	 * 弹出当前线程的会话管理器。
	 */
	public static void popLocalSessionManager(){
		Stack<SessionManager> sessionManagers = sessionManagerLocal.get();
		if(sessionManagers != null){
			sessionManagers.pop();
		}
	}
	
	/**
	 * 返回默认的会话管理器。
	 * 
	 * @return 默认会话管理器
	 */
	public static SessionManager getDefaultSessionManager() {
		if(defaultSessionManager == null){
			defaultSessionManager = new DefaultSessionManager();
		}
		return defaultSessionManager;
	}
	
	/**
	 * 获取一个指定的会话，其中name可以为null，如果Session不存在那么创建一个。。
	 * 
	 * @param actionContext 变量上下文
	 * @return 会话
	 */
	public abstract Session get(ActionContext actionContext);
	
	/**
	 * 删除当前会话。
	 * 
	 * @param actionContext 变量上下文
	 * @return 会话
	 */
	public abstract Session delete(ActionContext actionContext);
	
	/**
	 * 是否接受当前环境，即会话管理器是否是当前环境的会话管理器。
	 * 
	 * @param actionContext 变量上下文
	 * @return 返回是否能够管理Session
	 */
	public abstract boolean accept(ActionContext actionContext);
	
}