package org.xbib.helianthus.client;

import org.xbib.helianthus.client.http.HttpClientFactory;
import org.xbib.helianthus.common.Scheme;
import org.xbib.helianthus.common.util.ImmutableMap;

import java.net.URI;
import java.text.MessageFormat;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;

/**
 *
 */
public class AllInOneClientFactory extends NonDecoratingClientFactory {

    private static final Logger logger = Logger.getLogger(AllInOneClientFactory.class.getName());

    static {
        if (AllInOneClientFactory.class.getClassLoader() == ClassLoader.getSystemClassLoader()) {
            Runtime.getRuntime().addShutdownHook(new Thread(ClientFactory::closeDefault));
        }
    }

    private final Map<Scheme, ClientFactory> clientFactories;

    public AllInOneClientFactory() {
        this(SessionOptions.DEFAULT, false);
    }

    public AllInOneClientFactory(SessionOptions options, boolean useDaemonThreads) {
        super(options, useDaemonThreads);
        // We have only one session protocol at the moment, so this is OK so far.
        final HttpClientFactory httpClientFactory = new HttpClientFactory(options, useDaemonThreads);
        final Map<Scheme, ClientFactory> map = new HashMap<>();
        for (ClientFactory f : Collections.singletonList(httpClientFactory)) {
            f.supportedSchemes().forEach(s -> map.put(s, f));
        }
        clientFactories = new ImmutableMap<>(map);
    }

    @Override
    public Set<Scheme> supportedSchemes() {
        return clientFactories.keySet();
    }

    @Override
    public <T> T newClient(URI uri, Class<T> clientType, ClientOptions options) {
        final Scheme scheme = validate(uri, clientType, options);
        return clientFactories.get(scheme).newClient(uri, clientType, options);
    }

    @Override
    public void close() {
        // The global default should never be closed.
        if (this == ClientFactory.DEFAULT) {
            logger.fine(MessageFormat.format("Refusing to close the default {0}; must be closed via closeDefault()",
                    ClientFactory.class.getSimpleName()));
            return;
        }
        doClose();
    }

    void doClose() {
        clientFactories.values().forEach(ClientFactory::close);
    }
}
