package org.xbib.helianthus.client;

import static java.util.Objects.requireNonNull;

import io.netty.bootstrap.Bootstrap;
import io.netty.channel.ChannelFactory;
import io.netty.channel.ChannelOption;
import io.netty.channel.EventLoop;
import io.netty.channel.EventLoopGroup;
import io.netty.channel.epoll.EpollDatagramChannel;
import io.netty.channel.epoll.EpollEventLoopGroup;
import io.netty.channel.epoll.EpollSocketChannel;
import io.netty.channel.nio.NioEventLoopGroup;
import io.netty.channel.socket.DatagramChannel;
import io.netty.channel.socket.SocketChannel;
import io.netty.channel.socket.nio.NioDatagramChannel;
import io.netty.channel.socket.nio.NioSocketChannel;
import io.netty.resolver.AddressResolverGroup;
import io.netty.util.concurrent.DefaultThreadFactory;
import org.xbib.helianthus.common.RequestContext;
import org.xbib.helianthus.common.util.NativeLibraries;

import java.util.Optional;
import java.util.concurrent.ThreadFactory;
import java.util.function.Function;
import java.util.function.Supplier;

/**
 *
 */
public abstract class NonDecoratingClientFactory extends AbstractClientFactory {

    private final EventLoopGroup eventLoopGroup;

    private final boolean closeEventLoopGroup;

    private final SessionOptions options;

    private final Bootstrap baseBootstrap;

    private final Supplier<EventLoop> eventLoopSupplier =
            () -> RequestContext.mapCurrent(RequestContext::eventLoop, () -> eventLoopGroup().next());

    protected NonDecoratingClientFactory(SessionOptions options, boolean useDaemonThreads) {
        this(options, type -> {
            switch (type) {
                case NIO:
                    return new DefaultThreadFactory("org.xbib.helianthus.client.nio", useDaemonThreads);
                case EPOLL:
                    return new DefaultThreadFactory("org.xbib.helianthus.client.epoll", useDaemonThreads);
                default:
                    throw new Error();
            }
        });
    }

    private NonDecoratingClientFactory(SessionOptions options,
                                       Function<TransportType, ThreadFactory> threadFactoryFactory) {
        requireNonNull(options, "options");
        requireNonNull(threadFactoryFactory, "threadFactoryFactory");
        final Bootstrap baseBootstrap = new Bootstrap();
        baseBootstrap.channel(channelType());
        baseBootstrap.option(ChannelOption.CONNECT_TIMEOUT_MILLIS,
                safeLongToInt(options.connectTimeoutMillis()));
        baseBootstrap.option(ChannelOption.SO_KEEPALIVE, true);
        final Optional<EventLoopGroup> eventLoopOption = options.eventLoopGroup();
        if (eventLoopOption.isPresent()) {
            eventLoopGroup = eventLoopOption.get();
            closeEventLoopGroup = false;
        } else {
            eventLoopGroup = createGroup(threadFactoryFactory);
            closeEventLoopGroup = true;
        }
        this.baseBootstrap = baseBootstrap;
        this.options = options;
    }

    private static Class<? extends SocketChannel> channelType() {
        return NativeLibraries.isEpollAvailable() ? EpollSocketChannel.class : NioSocketChannel.class;
    }

    private static Class<? extends DatagramChannel> datagramChannelType() {
        return NativeLibraries.isEpollAvailable() ? EpollDatagramChannel.class : NioDatagramChannel.class;
    }

    private static EventLoopGroup createGroup(Function<TransportType, ThreadFactory> threadFactoryFactory) {
        return NativeLibraries.isEpollAvailable() ?
                new EpollEventLoopGroup(0, threadFactoryFactory.apply(TransportType.EPOLL)) :
                new NioEventLoopGroup(0, threadFactoryFactory.apply(TransportType.NIO));
    }

    @Override
    public final EventLoopGroup eventLoopGroup() {
        return eventLoopGroup;
    }

    @Override
    public final SessionOptions options() {
        return options;
    }

    /**
     * Returns a new {@link Bootstrap} whose {@link ChannelFactory}, {@link AddressResolverGroup} and
     * socket options are pre-configured.
     */
    public Bootstrap newBootstrap() {
        return baseBootstrap.clone();
    }

    @Override
    public final Supplier<EventLoop> eventLoopSupplier() {
        return eventLoopSupplier;
    }

    @Override
    public void close() {
        if (closeEventLoopGroup) {
            eventLoopGroup.shutdownGracefully().syncUninterruptibly();
        }
    }

    private static int safeLongToInt(long longValue) {
        return (int) Math.min(Math.max(longValue, Integer.MIN_VALUE), Integer.MAX_VALUE);
    }

    private enum TransportType {
        NIO, EPOLL
    }
}
