package org.xbib.helianthus.client.http;

import static java.util.Objects.requireNonNull;

import io.netty.bootstrap.Bootstrap;
import io.netty.channel.Channel;
import io.netty.channel.EventLoop;
import io.netty.channel.pool.ChannelHealthChecker;
import io.netty.util.AsciiString;
import io.netty.util.concurrent.Future;
import io.netty.util.concurrent.FutureListener;
import org.xbib.helianthus.client.Client;
import org.xbib.helianthus.client.ClientRequestContext;
import org.xbib.helianthus.client.Endpoint;
import org.xbib.helianthus.client.SessionOptions;
import org.xbib.helianthus.client.pool.DefaultKeyedChannelPool;
import org.xbib.helianthus.client.pool.KeyedChannelPool;
import org.xbib.helianthus.client.pool.KeyedChannelPoolHandler;
import org.xbib.helianthus.client.pool.KeyedChannelPoolHandlerAdapter;
import org.xbib.helianthus.client.pool.PoolKey;
import org.xbib.helianthus.common.ClosedSessionException;
import org.xbib.helianthus.common.SessionProtocol;
import org.xbib.helianthus.common.http.HttpHeaderNames;
import org.xbib.helianthus.common.http.HttpHeaders;
import org.xbib.helianthus.common.http.HttpRequest;
import org.xbib.helianthus.common.http.HttpResponse;
import org.xbib.helianthus.common.util.CompletionActions;

import java.net.InetSocketAddress;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.function.Function;

/**
 *
 */
final class HttpClientDelegate implements Client<HttpRequest, HttpResponse> {

    private static final KeyedChannelPoolHandlerAdapter<PoolKey> NOOP_POOL_HANDLER =
            new KeyedChannelPoolHandlerAdapter<>();

    private static final ChannelHealthChecker POOL_HEALTH_CHECKER =
            ch -> ch.eventLoop().newSucceededFuture(HttpSession.get(ch).isActive());

    private final ConcurrentMap<EventLoop, KeyedChannelPool<PoolKey>> map = new ConcurrentHashMap<>();

    private final HttpClientFactory factory;

    HttpClientDelegate(HttpClientFactory factory) {
        this.factory = requireNonNull(factory, "factory");
    }

    @Override
    public HttpResponse execute(ClientRequestContext ctx, HttpRequest req) throws Exception {
        final Endpoint endpoint = ctx.endpoint().resolve().withDefaultPort(ctx.sessionProtocol().defaultPort());
        autoFillHeaders(ctx, endpoint, req);
        final PoolKey poolKey = new PoolKey(InetSocketAddress.createUnresolved(endpoint.host(), endpoint.port()),
                ctx.sessionProtocol());
        final EventLoop eventLoop = ctx.eventLoop();
        final Future<Channel> channelFuture = pool(eventLoop).acquire(poolKey);
        final DecodedHttpResponse res = new DecodedHttpResponse(eventLoop);
        if (channelFuture.isDone()) {
            if (channelFuture.isSuccess()) {
                Channel ch = channelFuture.getNow();
                invoke0(ch, ctx, req, res, poolKey);
            } else {
                res.close(channelFuture.cause());
            }
        } else {
            channelFuture.addListener((Future<Channel> future) -> {
                if (future.isSuccess()) {
                    Channel ch = future.getNow();
                    invoke0(ch, ctx, req, res, poolKey);
                } else {
                    res.close(channelFuture.cause());
                }
            });
        }
        return res;
    }

    private static void autoFillHeaders(ClientRequestContext ctx, Endpoint endpoint, HttpRequest req) {
        requireNonNull(req, "req");
        final HttpHeaders headers = req.headers();
        if (headers.authority() == null) {
            final String hostname = endpoint.host();
            final int port = endpoint.port();
            final String authority;
            if (port == ctx.sessionProtocol().defaultPort()) {
                authority = hostname;
            } else {
                authority = hostname + ':' + port;
            }
            headers.authority(authority);
        }
        if (headers.scheme() == null) {
            headers.scheme(ctx.sessionProtocol().isTls() ? "https" : "http");
        }
        // Add the headers specified in ClientOptions, if not overridden by request.
        if (ctx.hasAttr(ClientRequestContext.HTTP_HEADERS)) {
            HttpHeaders clientOptionHeaders = ctx.attr(ClientRequestContext.HTTP_HEADERS).get();
            clientOptionHeaders.forEach(entry -> {
                AsciiString name = entry.getKey();
                if (!headers.contains(name)) {
                    headers.set(name, entry.getValue());
                }
            });
        }
        if (!headers.contains(HttpHeaderNames.USER_AGENT)) {
            headers.set(HttpHeaderNames.USER_AGENT, HttpHeaderUtil.USER_AGENT.toString());
        }
    }

    private KeyedChannelPool<PoolKey> pool(EventLoop eventLoop) {
        KeyedChannelPool<PoolKey> pool = map.get(eventLoop);
        if (pool != null) {
            return pool;
        }

        return map.computeIfAbsent(eventLoop, e -> {
            final Bootstrap bootstrap = factory.newBootstrap();
            final SessionOptions options = factory.options();
            bootstrap.group(eventLoop);
            Function<PoolKey, Future<Channel>> factory = new HttpSessionChannelFactory(bootstrap, options);
            final KeyedChannelPoolHandler<PoolKey> handler =
                    options.poolHandlerDecorator().apply(NOOP_POOL_HANDLER);
            final KeyedChannelPool<PoolKey> newPool = new DefaultKeyedChannelPool<>(
                    eventLoop, factory, POOL_HEALTH_CHECKER, handler, true);
            eventLoop.terminationFuture().addListener((FutureListener<Object>) f -> {
                map.remove(eventLoop);
                newPool.close();
            });
            return newPool;
        });
    }

    private void invoke0(Channel channel, ClientRequestContext ctx,
                         HttpRequest req, DecodedHttpResponse res, PoolKey poolKey) {
        final HttpSession session = HttpSession.get(channel);
        res.init(session.inboundTrafficController());
        final SessionProtocol sessionProtocol = session.protocol();
        if (sessionProtocol == null) {
            res.close(ClosedSessionException.get());
            return;
        }
        if (session.invoke(ctx, req, res)) {
            // Return the channel to the pool.
            final KeyedChannelPool<PoolKey> pool = KeyedChannelPool.findPool(channel);
            if (sessionProtocol.isMultiplex()) {
                pool.release(poolKey, channel);
            } else {
                req.closeFuture()
                        .handle((ret, cause) -> pool.release(poolKey, channel))
                        .exceptionally(CompletionActions::log);
            }
        }
    }

    void close() {
        map.values().forEach(KeyedChannelPool::close);
    }
}
