package org.xbib.helianthus.client.http;

import io.netty.channel.Channel;
import io.netty.channel.ChannelHandlerContext;
import io.netty.util.collection.IntObjectHashMap;
import io.netty.util.collection.IntObjectMap;
import io.netty.util.concurrent.ScheduledFuture;
import org.xbib.helianthus.client.ResponseTimeoutException;
import org.xbib.helianthus.common.http.HttpData;
import org.xbib.helianthus.common.http.HttpHeaders;
import org.xbib.helianthus.common.http.HttpObject;
import org.xbib.helianthus.common.http.HttpRequest;
import org.xbib.helianthus.common.http.HttpResponseWriter;
import org.xbib.helianthus.common.http.HttpStatus;
import org.xbib.helianthus.common.http.HttpStatusClass;
import org.xbib.helianthus.common.logging.RequestLogBuilder;
import org.xbib.helianthus.common.util.Exceptions;
import org.xbib.helianthus.internal.InboundTrafficController;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import java.util.logging.Level;
import java.util.logging.Logger;

abstract class HttpResponseDecoder {

    private static final Logger logger = Logger.getLogger(HttpResponseDecoder.class.getName());

    private final IntObjectMap<HttpResponseWrapper> responses = new IntObjectHashMap<>();
    private final InboundTrafficController inboundTrafficController;
    private boolean disconnectWhenFinished;

    HttpResponseDecoder(Channel channel) {
        inboundTrafficController = new InboundTrafficController(channel);
    }

    final InboundTrafficController inboundTrafficController() {
        return inboundTrafficController;
    }

    final HttpResponseWrapper addResponse(int id, HttpRequest req, DecodedHttpResponse res, RequestLogBuilder logBuilder,
            long responseTimeoutMillis, long maxContentLength) {
        final HttpResponseWrapper newRes =
                new HttpResponseWrapper(req, res, logBuilder, responseTimeoutMillis, maxContentLength);
        final HttpResponseWriter oldRes = responses.put(id, newRes);
        if (oldRes != null) {
            throw new IllegalStateException();
        }
        return newRes;
    }
    final HttpResponseWrapper getResponse(int id) {
        return responses.get(id);
    }

    final HttpResponseWrapper getResponse(int id, boolean remove) {
        return remove ? removeResponse(id) : getResponse(id);
    }

    final HttpResponseWrapper removeResponse(int id) {
        return responses.remove(id);
    }

    final boolean hasUnfinishedResponses() {
        return !responses.isEmpty();
    }

    final void failUnfinishedResponses(Throwable cause) {
        try {
            for (HttpResponseWrapper res : responses.values()) {
                res.close(cause);
            }
        } finally {
            responses.clear();
        }
    }

    final void disconnectWhenFinished() {
        disconnectWhenFinished = true;
    }

    final boolean needsToDisconnect() {
        return disconnectWhenFinished && !hasUnfinishedResponses();
    }

    static final class HttpResponseWrapper implements HttpResponseWriter, Runnable {
        private final HttpRequest request;
        private final DecodedHttpResponse delegate;
        private final RequestLogBuilder logBuilder;
        private final long responseTimeoutMillis;
        private final long maxContentLength;
        private ScheduledFuture<?> responseTimeoutFuture;

        HttpResponseWrapper(HttpRequest request, DecodedHttpResponse delegate, RequestLogBuilder logBuilder,
                            long responseTimeoutMillis, long maxContentLength) {
            this.request = request;
            this.delegate = delegate;
            this.logBuilder = logBuilder;
            this.responseTimeoutMillis = responseTimeoutMillis;
            this.maxContentLength = maxContentLength;
        }

        void scheduleTimeout(ChannelHandlerContext ctx) {
            if (responseTimeoutFuture != null || responseTimeoutMillis <= 0 || !isOpen()) {
                return;
            }
            responseTimeoutFuture = ctx.channel().eventLoop().schedule(
                    this, responseTimeoutMillis, TimeUnit.MILLISECONDS);
        }

        long maxContentLength() {
            return maxContentLength;
        }

        long writtenBytes() {
            return delegate.writtenBytes();
        }

        @Override
        public void run() {
            final ResponseTimeoutException cause = ResponseTimeoutException.get();
            delegate.close(cause);
            logBuilder.endResponse(cause);
        }

        @Override
        public boolean isOpen() {
            return delegate.isOpen();
        }

        @Override
        public boolean write(HttpObject o) {
            if (o instanceof HttpHeaders) {
                // NB: It's safe to call logBuilder.start() multiple times.
                //     See AbstractMessageLog.start() for more information.
                logBuilder.startResponse();
                final HttpHeaders headers = (HttpHeaders) o;
                final HttpStatus status = headers.status();
                if (status != null && status.codeClass() != HttpStatusClass.INFORMATIONAL) {
                    logBuilder.statusCode(status.code());
                    logBuilder.requestEnvelope(headers);
                }
            } else if (o instanceof HttpData) {
                logBuilder.increaseResponseLength(((HttpData) o).length());
            }
            return delegate.write(o);
        }

        @Override
        public boolean write(Supplier<? extends HttpObject> o) {
            return delegate.write(o);
        }

        @Override
        public CompletableFuture<Void> onDemand(Runnable task) {
            return delegate.onDemand(task);
        }

        @Override
        public void close() {
            if (request != null) {
                request.abort();
            }
            if (cancelTimeout()) {
                delegate.close();
                logBuilder.endResponse();
            }
        }

        @Override
        public void close(Throwable cause) {
            if (request != null) {
                request.abort();
            }
            if (cancelTimeout()) {
                delegate.close(cause);
                logBuilder.endResponse(cause);
            } else {
                if (!Exceptions.isExpected(cause)) {
                    logger.log(Level.WARNING, "Unexpected exception:", cause);
                }
            }
        }

        private boolean cancelTimeout() {
            final ScheduledFuture<?> responseTimeoutFuture = this.responseTimeoutFuture;
            if (responseTimeoutFuture == null) {
                return true;
            }
            this.responseTimeoutFuture = null;
            return responseTimeoutFuture.cancel(false);
        }

        @Override
        public void respond(HttpStatus status) {
            delegate.respond(status);
        }

        @Override
        public void respond(HttpStatus status,
                            String mediaType, String content) {
            delegate.respond(status, mediaType, content);
        }

        @Override
        public void respond(HttpStatus status,
                            String mediaType, String format, Object... args) {
            delegate.respond(status, mediaType, format, args);
        }

        @Override
        public void respond(HttpStatus status,
                            String mediaType, byte[] content) {
            delegate.respond(status, mediaType, content);
        }

        @Override
        public void respond(HttpStatus status,
                            String mediaType, byte[] content, int offset, int length) {
            delegate.respond(status, mediaType, content, offset, length);
        }

        @Override
        public String toString() {
            return delegate.toString();
        }
    }
}
