package org.xbib.helianthus.client;

import static java.util.Objects.requireNonNull;

import org.xbib.helianthus.common.Scheme;

import java.net.URI;
import java.util.Optional;
import java.util.Set;

/**
 *
 */
public abstract class AbstractClientFactory implements ClientFactory {

    protected AbstractClientFactory() {
    }

    protected static Endpoint newEndpoint(URI uri) {
        return Endpoint.parse(uri.getAuthority());
    }

    @Override
    public final <T> T newClient(String uri, Class<T> clientType, ClientOptionValue<?>... options) {
        requireNonNull(uri, "uri");
        requireNonNull(options, "options");
        return newClient(URI.create(uri), clientType, ClientOptions.of(options));
    }

    @Override
    public final <T> T newClient(String uri, Class<T> clientType, ClientOptions options) {
        requireNonNull(uri, "uri");
        return newClient(URI.create(uri), clientType, options);
    }

    @Override
    public final <T> T newClient(URI uri, Class<T> clientType, ClientOptionValue<?>... options) {
        requireNonNull(options, "options");
        return newClient(uri, clientType, ClientOptions.of(options));
    }

    protected final Scheme validate(URI uri, Class<?> clientType, ClientOptions options) {
        requireNonNull(uri, "uri");
        requireNonNull(clientType, "clientType");
        requireNonNull(options, "options");

        final String scheme = uri.getScheme();
        if (scheme == null) {
            throw new IllegalArgumentException("URI with missing scheme: " + uri);
        }

        if (uri.getAuthority() == null) {
            throw new IllegalArgumentException("URI with missing authority: " + uri);
        }

        final Optional<Scheme> parsedSchemeOpt = Scheme.tryParse(scheme);
        if (!parsedSchemeOpt.isPresent()) {
            throw new IllegalArgumentException("URI with unknown scheme: " + uri);
        }

        final Scheme parsedScheme = parsedSchemeOpt.get();
        final Set<Scheme> supportedSchemes = supportedSchemes();
        if (!supportedSchemes.contains(parsedScheme)) {
            throw new IllegalArgumentException(
                    "URI with unsupported scheme: " + uri + " (expected: " + supportedSchemes + ')');
        }

        return parsedScheme;
    }
}
