package org.xbib.helianthus.client;

import java.net.URI;

/**
 * Creates a new client that connects to a specified {@link URI}.
 */
public final class Clients {

    private Clients() {
    }

    /**
     * Creates a new client that connects to the specified {@code uri} using the default
     * {@link ClientFactory}.
     *
     * @param uri        the URI of the server endpoint
     * @param clientType the type of the new client
     * @param options    the {@link ClientOptionValue}s
     */
    public static <T> T newClient(String uri, Class<T> clientType, ClientOptionValue<?>... options) {
        return newClient(ClientFactory.DEFAULT, uri, clientType, options);
    }

    /**
     * Creates a new client that connects to the specified {@code uri} using the default
     * {@link ClientFactory}.
     *
     * @param uri        the URI of the server endpoint
     * @param clientType the type of the new client
     * @param options    the {@link ClientOptions}
     */
    public static <T> T newClient(String uri, Class<T> clientType, ClientOptions options) {
        return newClient(ClientFactory.DEFAULT, uri, clientType, options);
    }

    /**
     * Creates a new client that connects to the specified {@code uri} using an alternative
     * {@link ClientFactory}.
     *
     * @param factory    an alternative {@link ClientFactory}
     * @param uri        the URI of the server endpoint
     * @param clientType the type of the new client
     * @param options    the {@link ClientOptionValue}s
     */
    public static <T> T newClient(ClientFactory factory, String uri,
                                  Class<T> clientType, ClientOptionValue<?>... options) {

        return new ClientBuilder(uri).factory(factory).options(options)
                .build(clientType);
    }

    /**
     * Creates a new client that connects to the specified {@code uri} using an alternative
     * {@link ClientFactory}.
     *
     * @param factory    an alternative {@link ClientFactory}
     * @param uri        the URI of the server endpoint
     * @param clientType the type of the new client
     * @param options    the {@link ClientOptions}
     */
    public static <T> T newClient(ClientFactory factory, String uri,
                                  Class<T> clientType, ClientOptions options) {
        return new ClientBuilder(uri).factory(factory).options(options)
                .build(clientType);
    }

    /**
     * Creates a new client that connects to the specified {@link URI} using the default
     * {@link ClientFactory}.
     *
     * @param uri        the URI of the server endpoint
     * @param clientType the type of the new client
     * @param options    the {@link ClientOptionValue}s
     */
    public static <T> T newClient(URI uri, Class<T> clientType, ClientOptionValue<?>... options) {
        return newClient(ClientFactory.DEFAULT, uri, clientType, options);
    }

    /**
     * Creates a new client that connects to the specified {@link URI} using the default
     * {@link ClientFactory}.
     *
     * @param uri        the URI of the server endpoint
     * @param clientType the type of the new client
     * @param options    the {@link ClientOptions}
     */
    public static <T> T newClient(URI uri, Class<T> clientType, ClientOptions options) {
        return newClient(ClientFactory.DEFAULT, uri, clientType, options);
    }

    /**
     * Creates a new client that connects to the specified {@link URI} using an alternative
     * {@link ClientFactory}.
     *
     * @param factory    an alternative {@link ClientFactory}
     * @param uri        the URI of the server endpoint
     * @param clientType the type of the new client
     * @param options    the {@link ClientOptionValue}s
     */
    public static <T> T newClient(ClientFactory factory, URI uri, Class<T> clientType,
                                  ClientOptionValue<?>... options) {
        return new ClientBuilder(uri).factory(factory).options(options)
                .build(clientType);
    }

    /**
     * Creates a new client that connects to the specified {@link URI} using an alternative
     * {@link ClientFactory}.
     *
     * @param factory    an alternative {@link ClientFactory}
     * @param uri        the URI of the server endpoint
     * @param clientType the type of the new client
     * @param options    the {@link ClientOptions}
     */
    public static <T> T newClient(ClientFactory factory, URI uri, Class<T> clientType,
                                  ClientOptions options) {
        return new ClientBuilder(uri).factory(factory).options(options)
                .build(clientType);
    }

    /**
     * Creates a new derived client that connects to the same {@link URI} with the specified {@code client}
     * with the specified {@code additionalOptions}. Note that the derived client will use the options of
     * the specified {@code client} unless specified in {@code additionalOptions}.
     */
    public static <T> T newDerivedClient(T client, ClientOptionValue<?>... additionalOptions) {
        return asDerivable(client).withOptions(additionalOptions);
    }

    /**
     * Creates a new derived client that connects to the same {@link URI} with the specified {@code client}
     * with the specified {@code additionalOptions}. Note that the derived client will use the options of
     * the specified {@code client} unless specified in {@code additionalOptions}.
     */
    public static <T> T newDerivedClient(T client, Iterable<ClientOptionValue<?>> additionalOptions) {
        return asDerivable(client).withOptions(additionalOptions);
    }

    private static <T> ClientOptionDerivable<T> asDerivable(T client) {
        if (!(client instanceof ClientOptionDerivable)) {
            throw new IllegalArgumentException("client does not support derivation: " + client);
        }

        @SuppressWarnings("unchecked")
        final ClientOptionDerivable<T> derivable = (ClientOptionDerivable<T>) client;
        return derivable;
    }
}
