package org.xbib.helianthus.client;

import static java.util.Objects.requireNonNull;

import io.netty.channel.EventLoop;
import io.netty.channel.EventLoopGroup;
import io.netty.resolver.AddressResolverGroup;
import io.netty.util.ConstantPool;
import org.xbib.helianthus.client.pool.KeyedChannelPoolHandler;
import org.xbib.helianthus.client.pool.PoolKey;
import org.xbib.helianthus.common.util.AbstractOption;

import java.net.InetSocketAddress;
import java.time.Duration;
import java.util.function.Function;

import javax.net.ssl.TrustManagerFactory;

/**
 * An option that affects the session management of a {@link ClientFactory}.
 */
public class SessionOption<T> extends AbstractOption<T> {

    @SuppressWarnings("rawtypes")
    private static final ConstantPool pool = new ConstantPool() {
        @Override
        protected SessionOption<Object> newConstant(int id, String name) {
            return new SessionOption<>(id, name);
        }
    };

    /**
     * The timeout of a socket connection attempt.
     */
    public static final SessionOption<Duration> CONNECT_TIMEOUT = valueOf("CONNECT_TIMEOUT");

    /**
     * The idle timeout of a socket connection. The connection is closed if there is no invocation in progress
     * for this amount of time.
     */
    public static final SessionOption<Duration> IDLE_TIMEOUT = valueOf("IDLE_TIMEOUT");

    /**
     * The maximum number of concurrent in-progress invocations.
     */
    public static final SessionOption<Integer> MAX_CONCURRENCY = valueOf("MAX_CONCURRENCY");

    /**
     * The {@link TrustManagerFactory} of a TLS connection.
     */
    public static final SessionOption<TrustManagerFactory> TRUST_MANAGER_FACTORY =
            valueOf("TRUST_MANAGER_FACTORY");

    /**
     * The {@link AddressResolverGroup} to use to resolve remote addresses into {@link InetSocketAddress}es.
     */
    public static final SessionOption<AddressResolverGroup<? extends InetSocketAddress>>
            ADDRESS_RESOLVER_GROUP = valueOf("ADDRESS_RESOLVER_GROUP");

    /**
     * The {@link EventLoopGroup} that will provide the {@link EventLoop} for I/O and asynchronous invocations.
     * If unspecified, a new one is created automatically.
     */
    public static final SessionOption<EventLoopGroup> EVENT_LOOP_GROUP = valueOf("EVENT_LOOP_GROUP");

    /**
     * The {@link Function} that decorates the {@link KeyedChannelPoolHandler}.
     */
    public static final SessionOption<Function<KeyedChannelPoolHandler<PoolKey>,
            KeyedChannelPoolHandler<PoolKey>>> POOL_HANDLER_DECORATOR = valueOf("POOL_HANDLER_DECORATOR");

    /**
     * Whether to send an HTTP/2 preface string instead of an HTTP/1 upgrade request to negotiate the protocol
     * version of a cleartext HTTP connection.
     */
    public static final SessionOption<Boolean> USE_HTTP2_PREFACE = valueOf("USE_HTTP2_PREFACE");

    private SessionOption(int id, String name) {
        super(id, name);
    }

    /**
     * Returns the {@link SessionOption} of the specified name.
     */
    @SuppressWarnings("unchecked")
    public static <T> SessionOption<T> valueOf(String name) {
        return (SessionOption<T>) pool.valueOf(name);
    }

    /**
     * Creates a new value of this option.
     */
    public SessionOptionValue<T> newValue(T value) {
        requireNonNull(value, "value");
        return new SessionOptionValue<>(this, value);
    }
}
