package org.xbib.helianthus.client;

import static java.util.Objects.requireNonNull;
import static org.xbib.helianthus.client.SessionOption.ADDRESS_RESOLVER_GROUP;
import static org.xbib.helianthus.client.SessionOption.CONNECT_TIMEOUT;
import static org.xbib.helianthus.client.SessionOption.EVENT_LOOP_GROUP;
import static org.xbib.helianthus.client.SessionOption.IDLE_TIMEOUT;
import static org.xbib.helianthus.client.SessionOption.MAX_CONCURRENCY;
import static org.xbib.helianthus.client.SessionOption.POOL_HANDLER_DECORATOR;
import static org.xbib.helianthus.client.SessionOption.TRUST_MANAGER_FACTORY;
import static org.xbib.helianthus.client.SessionOption.USE_HTTP2_PREFACE;

import io.netty.channel.EventLoopGroup;
import io.netty.channel.epoll.Epoll;
import io.netty.channel.epoll.EpollDatagramChannel;
import io.netty.channel.socket.DatagramChannel;
import io.netty.channel.socket.nio.NioDatagramChannel;
import io.netty.resolver.AddressResolverGroup;
import io.netty.resolver.dns.DnsServerAddresses;
import org.xbib.helianthus.client.dns.CustomDnsAddressResolverGroup;
import org.xbib.helianthus.client.pool.KeyedChannelPoolHandler;
import org.xbib.helianthus.client.pool.PoolKey;
import org.xbib.helianthus.common.util.AbstractOptions;

import java.net.InetSocketAddress;
import java.text.MessageFormat;
import java.time.Duration;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.logging.Logger;

import javax.net.ssl.TrustManagerFactory;

/**
 * A set of {@link SessionOption}s and their respective values.
 */
public class SessionOptions extends AbstractOptions {

    private static final Logger logger = Logger.getLogger(SessionOptions.class.getName());

    private static final Duration DEFAULT_CONNECTION_TIMEOUT = Duration.ofMillis(3200);
    private static final Duration DEFAULT_IDLE_TIMEOUT = Duration.ofSeconds(10);
    private static final Integer DEFAULT_MAX_CONCURRENCY = Integer.MAX_VALUE;
    private static final Boolean DEFAULT_USE_HTTP2_PREFACE =
            "true".equals(System.getProperty("org.xbib.helianthus.defaultUseHttp2Preface", "false"));
    private static final SessionOptionValue<?>[] DEFAULT_OPTION_VALUES = {
            CONNECT_TIMEOUT.newValue(DEFAULT_CONNECTION_TIMEOUT),
            IDLE_TIMEOUT.newValue(DEFAULT_IDLE_TIMEOUT),
            MAX_CONCURRENCY.newValue(DEFAULT_MAX_CONCURRENCY),
            USE_HTTP2_PREFACE.newValue(DEFAULT_USE_HTTP2_PREFACE),
            ADDRESS_RESOLVER_GROUP.newValue(new CustomDnsAddressResolverGroup(datagramChannelType(),
                    DnsServerAddresses.defaultAddresses()))
    };
    /**
     * The default {@link SessionOptions}.
     */
    public static final SessionOptions DEFAULT = new SessionOptions(DEFAULT_OPTION_VALUES);

    static {
        logger.info(MessageFormat.format("defaultUseHttp2Preface: {0}", DEFAULT_USE_HTTP2_PREFACE));
    }

    private SessionOptions(SessionOptionValue<?>... options) {
        super(SessionOptions::validateValue, options);
    }

    private SessionOptions(SessionOptions baseOptions, SessionOptionValue<?>... options) {
        super(SessionOptions::validateValue, baseOptions, options);
    }

    private SessionOptions(
            SessionOptions baseOptions, Iterable<SessionOptionValue<?>> options) {
        super(SessionOptions::validateValue, baseOptions, options);
    }

    /**
     * Creates a new {@link SessionOptions} with the specified {@link SessionOptionValue}s.
     */
    public static SessionOptions of(SessionOptionValue<?>... options) {
        return new SessionOptions(DEFAULT, options);
    }

    /**
     * Returns the {@link SessionOptions} with the specified {@link SessionOptionValue}s.
     */
    public static SessionOptions of(Iterable<SessionOptionValue<?>> options) {
        return new SessionOptions(DEFAULT, options);
    }

    private static <T> SessionOptionValue<T> validateValue(SessionOptionValue<T> optionValue) {
        requireNonNull(optionValue, "value");

        SessionOption<?> option = optionValue.option();
        T value = optionValue.value();

        if (option == CONNECT_TIMEOUT) {
            validateConnectionTimeout((Duration) value);
        } else if (option == IDLE_TIMEOUT) {
            validateIdleTimeout((Duration) value);
        } else if (option == MAX_CONCURRENCY) {
            validateMaxConcurrency((Integer) value);
        }

        return optionValue;
    }

    private static Duration validateConnectionTimeout(Duration connectionTimeout) {
        requireNonNull(connectionTimeout, "connectionTimeout");
        if (connectionTimeout.isNegative() || connectionTimeout.isZero()) {
            throw new IllegalArgumentException(
                    "connectTimeout: " + connectionTimeout + " (expected: > 0)");
        }
        return connectionTimeout;
    }

    private static Duration validateIdleTimeout(Duration idleTimeout) {
        requireNonNull(idleTimeout, "idleTimeout");
        if (idleTimeout.isNegative()) {
            throw new IllegalArgumentException(
                    "idleTimeout: " + idleTimeout + " (expected: >= 0)");
        }
        return idleTimeout;
    }

    private static int validateMaxConcurrency(int maxConcurrency) {
        if (maxConcurrency <= 0) {
            throw new IllegalArgumentException("maxConcurrency: " + maxConcurrency + " (expected: > 0)");
        }
        return maxConcurrency;
    }

    /**
     * Returns the value of the specified {@link SessionOption}.
     *
     * @return the value of the {@link SessionOption}, or
     * {@link Optional#empty()} if the default value of the specified {@link SessionOption} is
     * not available
     */
    public <T> Optional<T> get(SessionOption<T> option) {
        return get0(option);
    }

    /**
     * Returns the value of the specified {@link SessionOption}.
     *
     * @return the value of the {@link SessionOption}, or
     * {@code defaultValue} if the specified {@link SessionOption} is not set.
     */
    public <T> T getOrElse(SessionOption<T> option, T defaultValue) {
        return getOrElse0(option, defaultValue);
    }

    /**
     * Converts this {@link SessionOptions} to a {@link Map}.
     */
    public Map<SessionOption<Object>, SessionOptionValue<Object>> asMap() {
        return asMap0();
    }

    public Duration connectTimeout() {
        return getOrElse(CONNECT_TIMEOUT, DEFAULT_CONNECTION_TIMEOUT);
    }

    public long connectTimeoutMillis() {
        return connectTimeout().toMillis();
    }

    public Optional<EventLoopGroup> eventLoopGroup() {
        return get(EVENT_LOOP_GROUP);
    }

    public Optional<TrustManagerFactory> trustManagerFactory() {
        return get(TRUST_MANAGER_FACTORY);
    }

    public Optional<AddressResolverGroup<InetSocketAddress>> addressResolverGroup() {
        final Optional<AddressResolverGroup<? extends InetSocketAddress>> value = get(ADDRESS_RESOLVER_GROUP);

        @SuppressWarnings("unchecked")
        final Optional<AddressResolverGroup<InetSocketAddress>> castValue =
                (Optional<AddressResolverGroup<InetSocketAddress>>) (Optional<?>) value;

        return castValue;
    }

    public Duration idleTimeout() {
        return getOrElse(IDLE_TIMEOUT, DEFAULT_IDLE_TIMEOUT);
    }

    public long idleTimeoutMillis() {
        return idleTimeout().toMillis();
    }

    public int maxConcurrency() {
        return getOrElse(MAX_CONCURRENCY, DEFAULT_MAX_CONCURRENCY);
    }

    public Function<KeyedChannelPoolHandler<PoolKey>, KeyedChannelPoolHandler<PoolKey>> poolHandlerDecorator() {
        return getOrElse(POOL_HANDLER_DECORATOR, Function.identity());
    }

    public boolean useHttp2Preface() {
        return getOrElse(USE_HTTP2_PREFACE, DEFAULT_USE_HTTP2_PREFACE);
    }

    private static Class<? extends DatagramChannel> datagramChannelType() {
        return Epoll.isAvailable() ? EpollDatagramChannel.class : NioDatagramChannel.class;
    }

}
